package jp.sourceforge.acerola3d.a3;

import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.*;
import javax.vecmath.*;

import java.io.*;
import java.awt.*;

/**
 * A3Objectを表示するためのWindowです。このウィンドウには
 * デフォルトでJA3Canvasが内包されており、
 * JA3Canvasと同名のメソッドは単に内包されているA3Canvas
 * の同じ名前のメソッドを呼び出しているだけです。
 * このクラスのaddメソッドを用いてA3Objectを登録すると、
 * A3ObjectがこのWindowの中に表示されるように
 * なっています。また、3D仮想空間におけるカメラの操作
 * も、このクラスのメソッドを通じて行うことができます。
 * それと、A3Listenerの登録先にもなります。
 */
public class JA3Window extends JFrame implements A3CanvasInterface {
    private static final long serialVersionUID = 1L;
    JA3Canvas canvas;

    /**
     * (w,h)の大きさのA3Windowを生成します。
     */
    public JA3Window(int w,int h) {
        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent we) {
                dispose();
                System.exit(0);
            }
        });
        canvas = JA3Canvas.createJA3Canvas(w,h);
        getContentPane().add(canvas);
        pack();
        setVisible(true);
        EventQueue.invokeLater(new Runnable() {
            public void run() {
                System.out.println("invokeLater");
                canvas.requestFocusInWindow();
            }
        });
    }

    /**
     * このA3Windowに内蔵されるJA3Canvasを返します。
     */
    public JA3Canvas getJA3Canvas() {
        return canvas;
    }

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a) {
        canvas.add(a);
    }

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a) {
        canvas.del(a);
    }

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll() {
        canvas.delAll();
    }

    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a) {
        canvas.setBackground(a);
    }

    /**
     * 背景を削除します。
     */
    public void delBackground() {
        canvas.delBackground();
    }

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a) {
        canvas.setAvatar(a);
    }

    /**
     * セットされたアバタを取得します。
     */
    public A3Object getAvatar() {
        return canvas.getAvatar();
    }

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l) {
        canvas.addA3Listener(l);
    }

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l) {
        canvas.removeA3Listener(l);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z) {
        canvas.setDefaultCameraLoc(x,y,z);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc) {
        canvas.setDefaultCameraLoc(loc);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w) {
        canvas.setDefaultCameraQuat(x,y,z,w);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat) {
        canvas.setDefaultCameraQuat(quat);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z) {
        canvas.setDefaultCameraRot(x,y,z);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot) {
        canvas.setDefaultCameraRot(rot);
    }

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s) {
        canvas.setDefaultCameraScale(s);
    }

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera() {
        canvas.resetCamera();
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z) {
        canvas.setCameraLoc(x,y,z);
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc) {
        canvas.setCameraLoc(loc);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z) {
        canvas.setCameraLocImmediately(x,y,z);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc) {
        canvas.setCameraLocImmediately(loc);
    }

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc() {
        return canvas.getCameraLoc();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w) {
        canvas.setCameraQuat(x,y,z,w);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat) {
        canvas.setCameraQuat(quat);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w) {
        canvas.setCameraQuatImmediately(x,y,z,w);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat) {
        canvas.setCameraQuatImmediately(quat);
    }

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat() {
        return canvas.getCameraQuat();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x, double y, double z) {
        canvas.setCameraRot(x,y,z);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot) {
        canvas.setCameraRot(rot);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x, double y, double z) {
        canvas.setCameraRotImmediately(x,y,z);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot) {
        canvas.setCameraRotImmediately(rot);
    }

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s) {
        canvas.setCameraScale(s);
    }

    /**
     * カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s) {
        canvas.setCameraScaleImmediately(s);
    }

    /**
     * カメラの拡大率を返します。
     */
    public double getCameraScale() {
        return canvas.getCameraScale();
    }

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b) {
        canvas.setHeadLightEnable(b);
    }

    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(A3Canvas.NaviMode m) {
        canvas.setNavigationMode(m);
    }

    /**
     * ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s) {
        canvas.setNavigationSpeed(s);
    }

    /**
     * ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed() {
        return canvas.getNavigationSpeed();
    }

    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c) {
        canvas.setA3Controller(c);
    }
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * アバタの座標を利用して、スクリーン上の点(x,y)を仮想空間上の
     * 点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直な仮想の面を想定して
     * その面上にあると仮定して計算します。その仮想の面は
     * A3CanvasInterfaceに設定されたアバタの座標を含むように
     * カメラからの距離が設定されます。
     */
    public Point3d canvasToVirtualCS(int x,int y) {
        return canvas.canvasToVirtualCS(x,y);
    }

    /**
     * スクリーン上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直で距離がdisとなる仮想の
     * 面上にあるとして計算するようになっています。この距離disは
     * <em>物理座標系</em>における長さで指定して下さい。
     * 物理座標系における長さとなるので、カメラの拡大縮小率を
     * 1.0以外に設定している場合には注意して下さい。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis) {
        return canvas.canvasToVirtualCS(x,y,dis);
    }

    /**
     * アバタの座標を利用して、スクリーン上の点(x,y)を物理空間上の
     * 点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直な仮想の面を想定して
     * その面上にあると仮定して計算します。その仮想の面は
     * A3CanvasInterfaceに設定されたアバタの座標を含むように
     * カメラからの距離が設定されます。つまり変換後のZ座標は
     * アバタの物理空間におけるZ座標と等しくなります。
     */
    public Point3d canvasToPhysicalCS(int x,int y) {
        return canvas.canvasToPhysicalCS(x,y);
    }

    /**
     * スクリーン上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直で距離がdisとなる仮想の
     * 面の上にあるとして計算するようになっています。この距離disは
     * 物理座標系における長さで指定して下さい。つまり変換後の
     * z座標は必ず-disになります。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis) {
        return canvas.canvasToPhysicalCS(x,y,dis);
    }

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v) {
        return canvas.physicalCSToVirtualCS(v);
    }

    /**
     * 物理空間上の点(x,y,z)をスクリーン上の点(x,y)に変換します。
     */
    public Point physicalCSToCanvas(Point3d p) {
        return canvas.physicalCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)をスクリーン上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p) {
        return canvas.virtualCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v) {
        return canvas.virtualCSToPhysicalCS(v);
    }

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX() {
        return canvas.getCameraUnitVecX();
    }

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY() {
        return canvas.getCameraUnitVecY();
    }

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ() {
        return canvas.getCameraUnitVecZ();
    }

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y) {
        return canvas.pickA3(x,y);
    }

    /**
     * A3Canvasに表示されている内容をPNG画像としてファイルに保存します。
     */
    public void saveImage(File file) throws IOException {
        canvas.saveImage(file);
    }
//  ----------シーン関係のメソッド---------
    /**
     * シーンを準備します。すでにその番号のシーンが
     * 存在していれば何もしません(既存の内容はそのまま)。
     */
    public void prepareScene(int s) {
        canvas.prepareScene(s);
    }
    /**
     * シーンを切り替えます。
     */
    public void changeActiveScene(int s) {
        canvas.changeActiveScene(s);
    }
    /**
     * シーンを指定してA3Objectを追加します。
     */
    public void add(A3Object a,int s) {
        canvas.add(a,s);
    }
    /**
     * シーンを指定してA3Objectを削除します。
     */
    public void del(A3Object a,int s) {
        canvas.add(a,s);
    }
    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z,int scene) {
        canvas.setDefaultCameraLoc(x,y,z);
    }

    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc,int scene) {
        canvas.setDefaultCameraLoc(loc,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w,int scene) {
        canvas.setDefaultCameraQuat(x,y,z,w,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat,int scene) {
        canvas.setDefaultCameraQuat(quat,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z,int scene) {
        canvas.setDefaultCameraRot(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot,int scene) {
        canvas.setDefaultCameraRot(rot,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s,int scene) {
        canvas.setDefaultCameraScale(s,scene);
    }

    /**
     * シーンを指定して、カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera(int scene) {
        canvas.resetCamera(scene);
    }

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z,int scene) {
        canvas.setCameraLoc(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc,int scene) {
        canvas.setCameraLoc(loc,scene);
    }

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z,int scene) {
        canvas.setCameraLocImmediately(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc,int scene) {
        canvas.setCameraLocImmediately(loc,scene);
    }

    /**
     * シーンを指定して、カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc(int scene) {
        return canvas.getCameraLoc(scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w,int scene) {
        canvas.setCameraQuat(x,y,z,w,scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat,int scene) {
        canvas.setCameraQuat(quat,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w,int scene) {
        canvas.setCameraQuatImmediately(x,y,z,w,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat,int scene) {
        canvas.setCameraQuatImmediately(quat,scene);
    }

    /**
     * シーンを指定して、カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat(int scene) {
        return canvas.getCameraQuat(scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z,int scene) {
        canvas.setCameraRot(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot,int scene) {
        canvas.setCameraRot(rot,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z,int scene) {
        canvas.setCameraRotImmediately(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot,int scene) {
        canvas.setCameraRotImmediately(rot,scene);
    }

    /**
     * シーンを指定して、カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s,int scene) {
        canvas.setCameraScale(s,scene);
    }

    /**
     * シーンを指定して、カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s,int scene) {
        canvas.setCameraScaleImmediately(s,scene);
    }

    /**
     * シーンを指定して、カメラの拡大率を返します。
     */
    public double getCameraScale(int scene) {
        return canvas.getCameraScale(scene);
    }
    /**
     * シーンを指定して、ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m,int scene) {
        canvas.setNavigationMode(m,scene);
    }

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s,int scene) {
        canvas.setNavigationSpeed(s,scene);
    }

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed(int scene) {
        return canvas.getNavigationSpeed(scene);
    }

    /**
     * シーンを指定して、A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c,int scene) {
        canvas.setA3Controller(c,scene);
    }

    /**
     * シーンを指定して、背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a,int scene) {
        canvas.setBackground(a,scene);
    }

    /**
     * シーンを指定して、背景を削除します。
     */
    public void delBackground(int scene) {
        canvas.delBackground(scene);
    }

    /**
     * シーンを指定して、アバタをセットします。
     */
    public void setAvatar(A3Object a,int scene) {
        canvas.setAvatar(a,scene);
    }

    /**
     * シーンを指定して、セットされたアバタを取得します。
     */
    public A3Object getAvatar(int scene) {
        return canvas.getAvatar(scene);
    }
//  ---------- Component2D関係 ----------
    /**
     * A3Canvasに2Dの描画を行うためのComponent2Dを登録します。
     */
    public void add(Component2D c) {
        canvas.add(c);;
    }
    /**
     * A3Canvasから2Dの描画を行うためのComponent2Dを削除します。
     */
    public void del(Component2D c) {
        canvas.del(c);
    }
    /**
     * シーンを指定して、A3Canvasに2Dの描画を行うためのComponent2Dを登録します。
     */
    public void add(Component2D c,int scene) {
        canvas.add(c,scene);
    }
    /**
     * シーンを指定して、A3Canvasから2Dの描画を行うためのComponent2Dを削除します。
     */
    public void del(Component2D c,int scene) {
        canvas.del(c,scene);
    }
//  ---------- KeyListener関係 ----------
    /**
     * KeyListenerを登録します。実際にはこのA3WindowのKeyListenerを
     * 登録するのではなく、このA3Windowが表示しているA3Canvasに登録
     * するようにオーバーライドしています。
     */
    public void addKeyListener(KeyListener l) {
        canvas.addKeyListener(l);
    }
    /**
     * 指定されたKeyListenerの登録を抹消します。実際にはこのA3Windowの
     * KeyListenerの登録を抹消するのではなく、このA3Windowが表示している
     * A3Canvasのに登録されているKeyListenerを抹消するようにオーバーライド
     * しています。
     */
    public void removeKeyListener(KeyListener l) {
        canvas.removeKeyListener(l);
    }
//  ---------- LockedA3の処理 ----------
    /**
     * A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    public void addLockedA3(A3Object a) {
        canvas.addLockedA3(a);
    }
    /**
     * 指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    public void delLockedA3(A3Object a) {
        canvas.delLockedA3(a);
    }
    /**
     * カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    public void delAllLockedA3() {
        canvas.delAllLockedA3();
    }
    /**
     * シーンを指定して、A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    public void addLockedA3(A3Object a,int scene) {
        canvas.addLockedA3(a,scene);
    }
    /**
     * シーンを指定して、指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    public void delLockedA3(A3Object a,int scene) {
        canvas.delLockedA3(a,scene);
    }
    /**
     * シーンを指定して、カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    public void delAllLockedA3(int scene) {
        canvas.delAllLockedA3(scene);
    }
    /**
     * 上方向をY軸とするのかZ軸とするのかの変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d) {
        canvas.setUpperDirection(d);
    }
    /**
     * シーンを指定して、上方向をY軸とするのかZ軸とするのかの
     * 変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d,int scene) {
        canvas.setUpperDirection(d,scene);
    }
    /**
     * 上方向をY軸としているのかZ軸としているのかを得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection() {
        return canvas.getUpperDirection();
    }
    /**
     * シーンを指定して、上方向をY軸としているのかZ軸としているのか
     * を得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection(int scene) {
        return canvas.getUpperDirection(scene);
    }
}
