package jp.sourceforge.acerola3d.a3;

import javax.vecmath.*;
import javax.media.j3d.*;

/**
 * A3パッケージにおける全ての3Dオブジェクトの
 * スーパークラス。A3CanvasやA3Windowにaddすることで、
 * 3Dオブジェクトが表示されるようになります。
 * 全ての3Dオブジェクトで必要となる、移動、回転、拡大縮小
 * などの機能を提供するメソッドが定義されています。
 */
public abstract class A3Object {
    static BoundingSphere bs = new BoundingSphere(new Point3d(0.0,0.0,0.0),Double.MAX_VALUE);
//    protected static final Serializable NULL = A23.NULL;


    A3Behavior behavior = null;
    Scene scene = null;
    A3Label label = null;
    A3Balloon balloon = null;
    A3Selected selected = null;
    boolean isSelected = false;
    Object userData = null;
    boolean lockedA3 = false;
    boolean pickable = true;
    /**
     * 座標系の上を指定するための列挙型。
     */
    public enum UpperDirection {Y,Z}
    Vector3d upperVector = new Vector3d(0.0,1.0,0.0);
    /**
     * 吹き出しの位置を指定するための列挙型。
     */
    public enum BalloonDir {RIGHT,LEFT,TOP,BOTTOM}

    /**
     * A3InitDataのデータをもとにしてA3Objectのインスタンスを
     * 生成するコンストラクタ。このコンストラクタは、このクラスを
     * 継承する全てのクラスで用意されることが望まれます。この
     * A3InitDataの中に含めるべきデータについては、それぞれの
     * 継承したクラスのAPIで説明されています。
     */
    public A3Object(A3InitData d) {
        behavior = new A3Behavior(this);
        behavior.setSchedulingBounds(bs);
        setEnableBehavior(d.getEnableBehavior());
        setAutoDirectionControl(d.getAutoDirectionControl());
        if (d.loc!=null)
            setLocImmediately(d.loc);
        if (d.quat!=null)
            setQuatImmediately(d.quat);
        setScaleImmediately(d.scale);
        if (d.label!=null)
            setLabel(d.label);
        if (d.balloon!=null)
            setBalloon(d.balloon);
        upperVector = d.upperVector;
        pickable = d.pickable;
    }

    /**
     * A3Canvasなどにaddされる直前に呼び出されるメソッドで、
     * 各種初期設定を行う。
     * このメソッドを上書きする場合はsuper.init();を
     * 記述してスーパークラスでの処理をする必要がある。
     */
    protected void init() {
        behavior.init();
    }

    /**
     * behaviorの働きのON、OFFを設定します。onにすると
     * 自動補完機能が有効になり、座標の変更などの表示が
     * なめらかに行われるようになります。
     * (さらにAction3DではAutoActionControlにも関係。)
     */
    public final void setEnableBehavior(boolean b) {
        behavior.setEnableBehavior(b);
    }

    /**
     * 進行方向を自動検出して、常に正面を向くように回転を補正する
     * モードのON、OFFを設定する。
     */
    public final void setAutoDirectionControl(boolean b) {
        behavior.setAutoDirectionControl(b);
    }

    /**
     * このオブジェクトの上方向を設定します。この上方向とは、
     * A3Objectのオートコントロール機能における正面の計算で、
     * 必要になる上方向ベクトルとして使用されます。
     * デフォルトの上方向はY軸の正の方向です。
     */
    public void setUpperVector(Vector3d v) {
        upperVector.set(v);
    }

    /**
     * このオブジェクトの上方向を取得します。この上方向とは、
     * A3Objectのオートコントロール機能における正面の計算で、
     * 必要になる上方向ベクトルとして使用されます。
     * デフォルトの上方向はY軸の正の方向です。
     */
    public Vector3d getUpperVector() {
        return new Vector3d(upperVector);
    }

    /**
     * この3Dオブジェクトが表すJava3DのNodeをセットします。
     * このクラスを拡張して使用する場合に、自分自身の
     * コンストラクタの中から呼び出してJava3DのNodeを
     * 初期化するというのが、このメソッドの主な用途になります。
     */
    protected void setNode(Node n) {
//        if (this.getUpperDirection()==UpperDirection.Z) {
//            Transform3D tt = new Transform3D();
////          tt.set(new Quat4d(1.0,0.0,0.0,0.0));
//            tt.set(new AxisAngle4d(1.0,0.0,0.0,Math.PI/2.0));
//            TransformGroup tg = new TransformGroup(tt);
//            tg.addChild(n);
//            behavior.setNode(tg);
//        } else {
//            behavior.setNode(n);
//        }
        behavior.setNode(n);
    }

    /**
     * このAction3Dオブジェクトの表すJava3DのNode
     * を返します。自前のJava3Dプログラムの中で、アクション
     * データを持つ3Dオブジェクトを使いたい場合には、
     * このメソッドで得られるNodeをJava3Dのシーングラフに
     * 追加して下さい。注意点としてprotected void setNode(Node n)
     * でセットしたNodeではなく、それをラップしたA3BranchGroup
     * というクラスのNodeが帰ります。
     */
    public Node getNode() {
        return behavior.topGroup;
    }

    /**
     * A3UpdateDataの情報をもとに現在の3Dオブジェクトの
     * 状態を更新します。A3UpdateDataの中に
     * 含まれるべきデータについては、A3Objectを継承したクラスの
     * APIで説明されます。A3Objectの実装では、
     * A3UpdateDataの中から位置、回転、拡大率の3つの情報を
     * 取り出し更新する処理がなされます。A3Objectを拡張する場合は
     * このメソッドをオーバーライドして他の必要な処理をする
     * ように拡張すると同時にsuper.update(d)というようにして、
     * A3Objectのupdateメソッドを実行しなければ位置、回転、拡大率の
     * 情報が更新されなくなるので注意。
     */
    public void update(A3UpdateData d) {
        if (d.loc!=null)
            setLoc(d.loc);
        if (d.quat!=null)
            setQuat(d.quat);
        //if (d.scale!=t.getScale())
        setScale(d.scale);
        //if (d.label!=null)
        setLabel(d.label);
        //if (d.balloon!=null)
        setBalloon(d.balloon);
    }

    void setScene(Scene s) {
        scene = s;
        behavior.setA3VirtualUniverse(s.universe);
        if (scene==null)
            return;
        if (label!=null)
            scene.add(label);
        if (balloon!=null)
            scene.add(balloon);
        if (selected!=null)
            scene.add(selected);
    }

    final A3BranchGroup getA3BranchGroup() {
        return behavior.topGroup;
    }

    /**
     * 位置、回転、拡大率を同時にセットするためのメソッド。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void move(Vector3d v, Quat4d q, double s) {
        behavior.move(v,q,s);
    }

    /**
     * 位置、回転、拡大率を同時にセットするためのメソッド。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     * 引数rotはx軸、y軸、z軸の回転からなるベクトルです。
     */
    public final void move(Vector3d loc, Vector3d rot, double scale) {
        move(loc,rot2quat(rot),scale);
    }

    /**
     * 位置、回転、拡大率を同時に、即時にセットするためのメソッド。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void moveImmediately(Vector3d v, Quat4d q, double s) {
        behavior.moveImmediately(v,q,s);
    }

    /**
     * 位置、回転、拡大率を同時に、即時にセットするためのメソッド。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     * 引数rotはx軸、y軸、z軸の回転からなるベクトルです。
     */
    public final void moveImmediately(Vector3d loc, Vector3d rot, double scale) {
        moveImmediately(loc,rot2quat(rot),scale);
    }

    /**
     * 位置をセットするためのメソッド。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void setLoc(Vector3d v) {
        behavior.setLoc(v);
    }

    /**
     * 位置をセットするためのメソッド。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void setLoc(double x, double y, double z) {
        setLoc(new Vector3d(x,y,z));
    }

    /**
     * 位置を即座にセットするためのメソッド。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void setLocImmediately(Vector3d v) {
        behavior.setLocImmediately(v);
    }

    /**
     * 位置を即座にセットするためのメソッド。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void setLocImmediately(double x,double y,double z) {
        setLocImmediately(new Vector3d(x,y,z));
    }

    /**
     * 現在位置を取得するためのメソッド。
     */
    public final Vector3d getLoc() {
        return behavior.getLoc();
    }
    
    /**
     * 回転をセットするためのメソッド(四元数:Quat4d版)。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void setQuat(Quat4d q) {
        behavior.setQuat(q);
    }

    /**
     * 回転をセットするためのメソッド(四元数:x,y,z,w版)。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void setQuat(double x, double y, double z, double w) {
        setQuat(new Quat4d(x,y,z,w));
    }

    /**
     * 回転を即座にセットするためのメソッド(四元数:Quad4d版)。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void setQuatImmediately(Quat4d q) {
        behavior.setQuatImmediately(q);
    }

    /**
     * 回転を即座にセットするためのメソッド(四元数:x,y,z,w版)。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void setQuatImmediately(double x,double y,double z,double w) {
        setQuatImmediately(new Quat4d(x,y,z,w));
    }

    /**
     * 現在の回転を取得するためのメソッド(四元数:Quad4d版)。
     */
    public final Quat4d getQuat() {
        return behavior.getQuat();
    }

    /**
     * 回転をセットするためのメソッド(オイラー角:Vector3d版)。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void setRot(Vector3d rot) {
        setQuat(rot2quat(rot));
    }

    /**
     * 回転をセットするためのメソッド(オイラー角:x,y,z版)。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void setRot(double x, double y,double z) {
        setQuat(rot2quat(x,y,z));
    }

    /**
     * 回転を即座にセットするためのメソッド(オイラー角:Vector3d版)。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void setRotImmediately(Vector3d rot) {
        setQuatImmediately(rot2quat(rot));
    }

    /**
     * 回転を即座にセットするためのメソッド(オイラー角:x,y,z版)。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void setRotImmediately(double x,double y,double z) {
        setQuatImmediately(rot2quat(x,y,z));
    }

    /**
     * 現在の回転を取得するためのメソッド(オイラー角:Vector3d版)。
     * 未実装！
     */
    public final Vector3d getRot() {
        return null;
    }

    /**
     * 拡大率をセットするためのメソッド。
     * behaviorが有効であれば、補完機能が働き変更はなめらかに
     * 反映されます。behaviorが無効であれば変更は即座に反映されます。
     */
    public final void setScale(double s) {
        behavior.setScale(s);
    }

    /**
     * 拡大率を即時にセットするためのメソッド。
     * behaviorが有効であってもなくても、変更が即座に反映されます。
     */
    public final void setScaleImmediately(double s) {
        behavior.setScaleImmediately(s);
    }

    /**
     * 現在の拡大率を取得するためのメソッド。
     */
    public final double getScale() {
        return behavior.getScale();
    }

    Quat4d rot2quat(double x,double y,double z) {
        Transform3D t0 = new Transform3D();
        Transform3D t1 = new Transform3D();
        t1.rotX(x);
        t0.mul(t1);
        t1.rotY(y);
        t0.mul(t1);
        t1.rotZ(z);
        t0.mul(t1);
        Quat4d q = new Quat4d();
        t0.get(q);
        return q;
    }
    Quat4d rot2quat(Vector3d rot) {
        return rot2quat(rot.x,rot.y,rot.z);
    }

    /**
     * このオブジェクトのリソースを開放します。
     */
    public void dispose() {
        if (scene!=null) {
            if (label!=null)
                scene.del(label);
            if (balloon!=null)
                scene.del(balloon);
            if (selected!=null)
                scene.del(selected);
        }
    }

    /**
     * ラベルをセットします。(デスクトップのアイコンの
     * ファイル名のような感じで)表示されているラベルを
     * 消去したい場合には，引数にnullをわたして下さい．
     */
    public void setLabel(String l) {
        if (label==null)
            label = new A3Label(l,this);
        if (l!=null) {
            label.setString(l);
            if (scene!=null)
                scene.add(label);
        } else {
            if (scene!=null)
                scene.del(label);
        }
    }

    /**
     * ラベルを表示する場合のオフセットを設定します。
     */
    public void setLabelLoc(double x,double y) {
        if (label==null)
            label = new A3Label("",this);
        label.setOffset(x,y);
    }

    /**
     * 吹き出しをセットします。表示されている吹き出しを
     * 消去したい場合には，引数にnullをわたして下さい．
     */
    public void setBalloon(String s) {
        if (balloon==null)
            balloon = new A3Balloon(s,this);
        if (s!=null) {
            balloon.setString(s);
            if (scene!=null)
                scene.add(balloon);
        } else {
            if (scene!=null)
                scene.del(balloon);
        }
    }

    /**
     * 吹き出しを表示する場合の向きとオフセットを設定します。
     */
    public void setBalloonLoc(BalloonDir d,double x,double y) {
        if (balloon==null)
            balloon = new A3Balloon("",this);
        balloon.setOffset(d,x,y);
    }

    /**
     * 吹き出しを表示する場合の向きを設定します。
     */
    public void setBalloonDir(BalloonDir d) {
        if (balloon==null)
            balloon = new A3Balloon("",this);
        balloon.setDir(d);
    }

    /**
     * この3Dオブジェクトが選択されているという目印
     * のON、OFFを設定します。
     */
    public final void setSelected(boolean b) {
        isSelected = b;
        if (selected==null)
            selected =new A3Selected(this);
        if (b) {
            if (scene!=null)
                scene.add(selected);
        } else {
            if (scene!=null)
                scene.del(selected);
        }
    }

    /**
     * この3Dオブジェクトが選択されているかどうかを
     * 返します。
     */
    public final boolean isSelected() {
        return isSelected;
    }

    /**
     * ユーザデータをセットします。A3Objectは
     * 3Dユーザインタフェースでしかないので、
     * この3Dインタフェースが表す、プログラムやエージェントや
     * データの実態を、このメソッドを使ってセットしておけば、
     * これらの関連性が簡単に示せます。
     */
    public final void setUserData(Object o) {
        userData = o;
    }

    /**
     * ユーザデータを返します。A3Objectは
     * 3Dユーザインタフェースでしかないので、
     * この3Dインタフェースが表す、プログラムやエージェントや
     * データの実態を、このメソッドを使ってセットしておけば、
     * これらの関連性が簡単に示せます。
     */
    public final Object getUserData() {
        return userData;
    }
    /**
     * Behaviorが有効である時A3Objectの現在のスピードを返します。
     * Behaviorが無効である場合は常に0を返します。
     */
    public final double getSpeed() {
        return behavior.getSpeed();
    }
    /**
     * ピッキングできるかどうかを設定するメソッドです。
     * デフォルトはtrueです。
     */
    public void setPickable(boolean b) {
        pickable = b;
    }
    /**
     * ピッキングできるかどうかを調べるメソッドです。
     */
    public boolean getPickable() {
        return pickable;
    }
}
