package jp.sourceforge.acerola3d.a3;

import java.awt.*;
import java.awt.image.BufferedImage;
import javax.media.j3d.*;
import javax.vecmath.*;

import java.io.*;
import java.util.*;

import javax.imageio.ImageIO;

/**
 * A3Objectを表示するためのGUIコンポーネントです。
 * このクラスのaddメソッドを用いてA3Objectを登録すると、
 * A3ObjectがこのGUIコンポーネントの中に表示されるように
 * なっています。また、3D仮想空間におけるカメラの操作
 * も、このクラスのメソッドを通じて行います。
 * それと、A3Listenerの登録先でもあります。
 * コンストラクタではなく、createA3Canvasメソッドを使って
 * 生成します。
 */
public class A3Canvas extends Canvas3D implements A3CanvasInterface {
    private static final long serialVersionUID = 1L;

    A3VirtualUniverse universe;

    /**
     * (500,500)の大きさのA3Canvasを作成します。
     */
    public static A3Canvas createA3Canvas() {
        return createA3Canvas(500,500);
    }

    /**
     * (w,h)の大きさのA3Canvasを作ります。
     */
    public static A3Canvas createA3Canvas(int w,int h) {
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        GraphicsDevice gd = ge.getDefaultScreenDevice();
        GraphicsConfiguration gcs[] = gd.getConfigurations();
        GraphicsConfigTemplate3D gct3d = new GraphicsConfigTemplate3D();
        GraphicsConfiguration gc = gct3d.getBestConfiguration(gcs);
        return createA3Canvas(gc,w,h);
    }

    /**
     * GraphicsConfigurationオブジェクトを指定して
     * (500,500)の大きさのA3Canvasを作ります。
     */
    public static A3Canvas createA3Canvas(GraphicsConfiguration graCon) {
        return createA3Canvas(graCon,500,500);
    }

    /**
     * GraphicsConfigurationオブジェクトを指定して
     * (w,h)の大きさのA3Canvasを作ります。
     */
    public static A3Canvas createA3Canvas(GraphicsConfiguration graCon,int w,int h) {
        return new A3Canvas(graCon,w,h);
    }

    A3Canvas(GraphicsConfiguration graCon,int w,int h) {
        //super(graCon.getDevice());
        super(graCon);
        gc = getGraphicsContext3D();
        universe = new A3VirtualUniverse(this);
        setSize(w,h);
    }

    public TimerBehavior getTimerBehavior() {
        return universe.getTimerBehavior();
    }

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a) {
        universe.add(a);
    }

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a) {
        universe.del(a);
    }

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll() {
        universe.delAll();
    }

    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a) {
        universe.setBackground(a);
    }

    /**
     * 背景を削除します。
     */
    public void delBackground() {
        universe.delBackground();
    }

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a) {
        universe.setAvatar(a);
    }

    /**
     * セットされたアバタを取得します。
     */
    public A3Object getAvatar() {
        return universe.getAvatar();
    }

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l) {
        universe.addA3Listener(l);
    }

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l) {
        universe.removeA3Listener(l);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z) {
        universe.setDefaultCameraLoc(x,y,z);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc) {
        universe.setDefaultCameraLoc(loc);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w) {
        universe.setDefaultCameraQuat(x,y,z,w);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat) {
        universe.setDefaultCameraQuat(quat);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z) {
        universe.setDefaultCameraRot(x,y,z);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot) {
        universe.setDefaultCameraRot(rot);
    }

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s) {
        universe.setDefaultCameraScale(s);
    }

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera() {
        universe.resetCamera();
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z) {
        universe.setCameraLoc(x,y,z);
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc) {
        universe.setCameraLoc(loc);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z) {
        universe.setCameraLocImmediately(x,y,z);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc) {
        universe.setCameraLocImmediately(loc);
    }

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc() {
        return universe.getCameraLoc();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w) {
        universe.setCameraQuat(x,y,z,w);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat) {
        universe.setCameraQuat(quat);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w) {
        universe.setCameraQuatImmediately(x,y,z,w);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat) {
        universe.setCameraQuatImmediately(quat);
    }

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat() {
        return universe.getCameraQuat();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z) {
        universe.setCameraRot(x,y,z);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot) {
        universe.setCameraRot(rot);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z) {
        universe.setCameraRotImmediately(x,y,z);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot) {
        universe.setCameraRotImmediately(rot);
    }

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s) {
        universe.setCameraScale(s);
    }

    /**
     * カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s) {
        universe.setCameraScaleImmediately(s);
    }

    /**
     * カメラの拡大率を返します。
     */
    public double getCameraScale() {
        return universe.getCameraScale();
    }

    /**
     * ルックアットポイントを使って、カメラの回転を指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPoint(Vector3d lookAt) {
        universe.setCameraLookAtPoint(lookAt);
    }

    /**
     * ルックアットポイントを使って、カメラの回転を即時に指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPointImmediately(Vector3d lookAt) {
        universe.setCameraLookAtPointImmediately(lookAt);
    }

    /**
     * ルックアットポイントを使って、カメラの回転を指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPoint(double x,double y,double z) {
        universe.setCameraLookAtPoint(x,y,z);
    }

    /**
     * ルックアットポイントを使って、カメラの回転を即時に指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPointImmediately(double x,double y,double z) {
        universe.setCameraLookAtPointImmediately(x,y,z);
    }

    /**
     * ルックアットポイントと上方向ベクトルを使って、カメラの回転を
     * 指定します。
     */
    public void setCameraLookAtPoint(Vector3d lookAt,Vector3d up) {
        universe.setCameraLookAtPoint(lookAt,up);
    }

    /**
     * ルックアットポイントと上方向ベクトルを使って、カメラの回転を
     * 即時に指定します。
     */
    public void setCameraLookAtPointImmediately(Vector3d lookAt,Vector3d up) {
        universe.setCameraLookAtPointImmediately(lookAt,up);
    }

    /**
     * ルックアットポイントと上方向ベクトルを使って、カメラの回転を
     * 指定します。
     */
    public void setCameraLookAtPoint(double x,double y,double z,Vector3d up) {
        universe.setCameraLookAtPoint(x,y,z,up);
    }

    /**
     * ルックアットポイントと上方向ベクトルを使って、カメラの回転を
     * 即時に指定します。
     */
    public void setCameraLookAtPointImmediately(double x,double y,double z,Vector3d up) {
        universe.setCameraLookAtPointImmediately(x,y,z,up);
    }

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b) {
        universe.setHeadLightEnable(b);
    }

    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m) {
        universe.setNavigationMode(m);
    }

    /**
     * ナビゲーションの大まかなスピードを設定します。
     * 単位はm/s。デフォルト値は1.0。
     */
    public void setNavigationSpeed(double s) {
        universe.setNavigationSpeed(s);
    }

    /**
     * ナビゲーションの大まかなスピードを取得します。
     * 単位はm/s。A3Controllerの作成者はこの値を参照して
     * ナビゲーションのスピードを
     * 計算することが望まれます。
     */
    public double getNavigationSpeed() {
        return universe.getNavigationSpeed();
    }

    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     * また，設定されているA3Controllerを解除して無効にするためには引数に
     * nullを指定して呼び出して下さい．
     */
    public void setA3Controller(A3Controller c) {
        universe.setA3Controller(c);
    }
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * アバタの座標を利用して、スクリーン上の点(x,y)を仮想空間上の
     * 点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直な仮想の面を想定して
     * その面上にあると仮定して計算します。その仮想の面は
     * A3CanvasInterfaceに設定されたアバタの座標を含むように
     * カメラからの距離が設定されます。
     */
    public Point3d canvasToVirtualCS(int x,int y) {
        return universe.canvasToVirtualCS(x,y);
    }

    /**
     * スクリーン上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直で距離がdisとなる仮想の
     * 面上にあるとして計算するようになっています。この距離disは
     * <em>物理座標系</em>における長さで指定して下さい。
     * 物理座標系における長さとなるので、カメラの拡大縮小率を
     * 1.0以外に設定している場合には注意して下さい。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis) {
        return universe.canvasToVirtualCS(x,y,dis);
    }

    /**
     * アバタの座標を利用して、スクリーン上の点(x,y)を物理空間上の
     * 点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直な仮想の面を想定して
     * その面上にあると仮定して計算します。その仮想の面は
     * A3CanvasInterfaceに設定されたアバタの座標を含むように
     * カメラからの距離が設定されます。つまり変換後のZ座標は
     * アバタの物理空間におけるZ座標と等しくなります。
     */
    public Point3d canvasToPhysicalCS(int x,int y) {
        return universe.canvasToPhysicalCS(x,y);
    }

    /**
     * スクリーン上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直で距離がdisとなる仮想の
     * 面の上にあるとして計算するようになっています。この距離disは
     * 物理座標系における長さで指定して下さい。つまり変換後の
     * z座標は必ず-disになります。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis) {
        return universe.canvasToPhysicalCS(x,y,dis);
    }

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v) {
        return universe.physicalCSToVirtualCS(v);
    }

    /**
     * 物理空間上の点(x,y,z)をスクリーン上の点(x,y)に変換します。
     */
    public Point physicalCSToCanvas(Point3d p) {
        return universe.physicalCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)をスクリーン上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p) {
        return universe.virtualCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v) {
        return universe.virtualCSToPhysicalCS(v);
    }

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX() {
        return universe.getCameraUnitVecX();
    }

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY() {
        return universe.getCameraUnitVecY();
    }

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ() {
        return universe.getCameraUnitVecZ();
    }

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y) {
        return universe.pickA3(x,y);
    }
//  ----------J3DGraphics2D(文字描画など)---------
    /**
     * A3Canvasに2Dの描画を行うためのComponent2Dを登録します。
     */
    public void add(Component2D c) {
        universe.add(c);
    }
    /**
     * A3Canvasから2Dの描画を行うためのComponent2Dを削除します。
     */
    public void del(Component2D c) {
        universe.del(c);
    }
    /**
     * シーンを指定して、A3Canvasに2Dの描画を行うためのComponent2Dを登録します。
     */
    public void add(Component2D c,int scene) {
        universe.add(c,scene);
    }
    /**
     * シーンを指定して、A3Canvasから2Dの描画を行うためのComponent2Dを削除します。
     */
    public void del(Component2D c,int scene) {
        universe.del(c,scene);
    }
    public synchronized void postRender() {
        //J3DGraphics2D gg = (J3DGraphics2D)getGraphics();
        Component2D cs[] = universe.getComponents2D().toArray(new Component2D[0]);
        for (Component2D c: cs) {
            c.calPhysicalZ(this);
        }
        Arrays.sort(cs,Component2DComparator.comparator);
        J3DGraphics2D gg = getGraphics2D();
        for (Component2D c : cs) {
            if (c.z>0.0)
                continue;
            c.paint(gg,this);
        }
        if (Component2D.advertisement!=null) {
            Component2D.advertisement.paint(gg,this);
        }
        gg.flush(true);
    }
//  ----------おまけ機能---------
    volatile boolean check = false;
    GraphicsContext3D gc;
    Raster readRaster;

    public void postSwap() {
        super.postSwap();
        if (check) {
            gc.readRaster(readRaster);
            check = false;
        }
    }

    /**
     * A3Canvasに表示されている内容をPNG画像としてファイルに保存します。
     */
    public void saveImage(File file) throws IOException {
        int width = getWidth();
        int height = getHeight();
        BufferedImage bImage = new BufferedImage(
                               width,height,BufferedImage.TYPE_INT_RGB);
        ImageComponent2D ic2d = new ImageComponent2D(
                                ImageComponent.FORMAT_RGB,bImage);
        //DepthComponentFloat dcf = new DepthComponentFloat(width,height);
        readRaster = new Raster(new Point3f(0.0f,0.0f,0.0f),
                            Raster.RASTER_COLOR,0,0,width,height,
                            ic2d,null);
        check = true;
        while(check) {
            try{Thread.sleep(300);}catch(Exception e){;}
        }

        ImageComponent2D ic = readRaster.getImage();
        BufferedImage image = ic.getImage();

        ImageIO.write(image,"png",file);
//        FileOutputStream out = new FileOutputStream(file);
//        JPEGImageEncoder e = JPEGCodec.createJPEGEncoder(out);
//        e.encode(image);
//        out.close();
    }
//  ----------シーン関係のメソッド---------
    /**
     * シーンを準備します。すでにその番号のシーンが
     * 存在していれば何もしません(既存の内容はそのまま)。
     */
    public void prepareScene(int scene) {
        universe.prepareScene(scene);
    }
    /**
     * シーンを切り替えます。
     */
    public void changeActiveScene(int s) {
        universe.changeActiveScene(s);
    }
    /**
     * シーンを指定してA3Objectを追加します。
     */
    public void add(A3Object a,int s) {
        universe.add(a,s);
    }
    /**
     * シーンを指定して、A3Objectを削除します。
     */
    public void del(A3Object a,int s) {
        universe.add(a,s);
    }
    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z,int scene) {
        universe.setDefaultCameraLoc(x,y,z);
    }

    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc,int scene) {
        universe.setDefaultCameraLoc(loc,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w,int scene) {
        universe.setDefaultCameraQuat(x,y,z,w,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat,int scene) {
        universe.setDefaultCameraQuat(quat,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z,int scene) {
        universe.setDefaultCameraRot(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot,int scene) {
        universe.setDefaultCameraRot(rot,scene);
    }

    /**
     * シーンを指定して、カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s,int scene) {
        universe.setDefaultCameraScale(s,scene);
    }

    /**
     * シーンを指定して、カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera(int scene) {
        universe.resetCamera(scene);
    }

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z,int scene) {
        universe.setCameraLoc(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc,int scene) {
        universe.setCameraLoc(loc,scene);
    }

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z,int scene) {
        universe.setCameraLocImmediately(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc,int scene) {
        universe.setCameraLocImmediately(loc,scene);
    }

    /**
     * シーンを指定して、カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc(int scene) {
        return universe.getCameraLoc(scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w,int scene) {
        universe.setCameraQuat(x,y,z,w,scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat,int scene) {
        universe.setCameraQuat(quat,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w,int scene) {
        universe.setCameraQuatImmediately(x,y,z,w,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat,int scene) {
        universe.setCameraQuatImmediately(quat,scene);
    }

    /**
     * シーンを指定して、カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat(int scene) {
        return universe.getCameraQuat(scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z,int scene) {
        universe.setCameraRot(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot,int scene) {
        universe.setCameraRot(rot,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z,int scene) {
        universe.setCameraRotImmediately(x,y,z,scene);
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot,int scene) {
        universe.setCameraRotImmediately(rot,scene);
    }

    /**
     * シーンを指定して、カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s,int scene) {
        universe.setCameraScale(s,scene);
    }

    /**
     * シーンを指定して、カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s,int scene) {
        universe.setCameraScaleImmediately(s,scene);
    }

    /**
     * シーンを指定して、カメラの拡大率を返します。
     */
    public double getCameraScale(int scene) {
        return universe.getCameraScale(scene);
    }

    /**
     * シーンを指定し、ルックアットポイントを使ってカメラの回転を指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPoint(Vector3d lookAt,int scene) {
        universe.setCameraLookAtPoint(lookAt,scene);
    }

    /**
     * シーンを指定し、ルックアットポイントを使ってカメラの回転を即時に指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPointImmediately(Vector3d lookAt,int scene) {
        universe.setCameraLookAtPointImmediately(lookAt,scene);
    }

    /**
     * シーンを指定し、ルックアットポイントを使ってカメラの回転を指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPoint(double x,double y,double z,int scene) {
        universe.setCameraLookAtPoint(x,y,z,scene);
    }

    /**
     * シーンを指定し、ルックアットポイントを使ってカメラの回転を即時に指定します。
     * 上方向ベクトルはそのカメラが存在するシーンのupperDirectionの
     * 値を用いて計算します。
     */
    public void setCameraLookAtPointImmediately(double x,double y,double z,int scene) {
        universe.setCameraLookAtPointImmediately(x,y,z,scene);
    }

    /**
     * シーンを指定し、ルックアットポイントと上方向ベクトルを使ってカメラの回転を
     * 指定します。
     */
    public void setCameraLookAtPoint(Vector3d lookAt,Vector3d up,int scene) {
        universe.setCameraLookAtPoint(lookAt,up,scene);
    }

    /**
     * シーンを指定し、ルックアットポイントと上方向ベクトルを使ってカメラの回転を
     * 即時に指定します。
     */
    public void setCameraLookAtPointImmediately(Vector3d lookAt,Vector3d up,int scene) {
        universe.setCameraLookAtPointImmediately(lookAt,up,scene);
    }

    /**
     * シーンを指定し、ルックアットポイントと上方向ベクトルを使ってカメラの回転を
     * 指定します。
     */
    public void setCameraLookAtPoint(double x,double y,double z,Vector3d up,int scene) {
        universe.setCameraLookAtPoint(x,y,z,up,scene);
    }

    /**
     * シーンを指定し、ルックアットポイントと上方向ベクトルを使ってカメラの回転を
     * 即時に指定します。
     */
    public void setCameraLookAtPointImmediately(double x,double y,double z,Vector3d up,int scene) {
        universe.setCameraLookAtPointImmediately(x,y,z,up,scene);
    }

    /**
     * シーンを指定して、ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m,int scene) {
        universe.setNavigationMode(m,scene);
    }

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s,int scene) {
        universe.setNavigationSpeed(s,scene);
    }

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed(int scene) {
        return universe.getNavigationSpeed(scene);
    }

    /**
     * シーンを指定して、A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     * また，設定されているA3Controllerを解除して無効にするためには引数に
     * nullを指定して呼び出して下さい．
     */
    public void setA3Controller(A3Controller c,int scene) {
        universe.setA3Controller(c,scene);
    }

    /**
     * シーンを指定して、背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a,int scene) {
        universe.setBackground(a,scene);
    }

    /**
     * シーンを指定して、背景を削除します。
     */
    public void delBackground(int scene) {
        universe.delBackground(scene);
    }

    /**
     * シーンを指定して、アバタをセットします。
     */
    public void setAvatar(A3Object a,int scene) {
        universe.setAvatar(a,scene);
    }

    /**
     * シーンを指定して、セットされたアバタを取得します。
     */
    public A3Object getAvatar(int scene) {
        return universe.getAvatar(scene);
    }
//  ---------- LockedA3の処理 ----------
    /**
     * A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    public void addLockedA3(A3Object a) {
        universe.addLockedA3(a);
    }
    /**
     * 指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    public void delLockedA3(A3Object a) {
        universe.delLockedA3(a);
    }
    /**
     * カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    public void delAllLockedA3() {
        universe.delAllLockedA3();
    }
    /**
     * シーンを指定して、A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    public void addLockedA3(A3Object a,int scene) {
        universe.addLockedA3(a,scene);
    }
    /**
     * シーンを指定して、指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    public void delLockedA3(A3Object a,int scene) {
        universe.delLockedA3(a,scene);
    }
    /**
     * シーンを指定して、カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    public void delAllLockedA3(int scene) {
        universe.delAllLockedA3(scene);
    }
    /**
     * 上方向をY軸とするのかZ軸とするのかの変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d) {
        universe.setUpperDirection(d);
    }
    /**
     * シーンを指定して、上方向をY軸とするのかZ軸とするのかの
     * 変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d,int scene) {
        universe.setUpperDirection(d,scene);
    }
    /**
     * 上方向をY軸としているのかZ軸としているのかを得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection() {
        return universe.getUpperDirection();
    }
    /**
     * シーンを指定して、上方向をY軸としているのかZ軸としているのか
     * を得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection(int scene) {
        return universe.getUpperDirection(scene);
    }
}
