/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2008 Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.workflow;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.SelectQuery;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowCategory;
import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowFile;
import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowRequest;
import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowRequestMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.fileupload.beans.FileuploadLiteBean;
import com.aimluck.eip.fileupload.util.FileuploadUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;
import com.aimluck.eip.whatsnew.util.WhatsNewUtils;
import com.aimluck.eip.workflow.util.WorkflowUtils;

/**
 * ワークフローのフォームデータを管理するクラスです。 <BR>
 * 
 */
public class WorkflowFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(WorkflowFormData.class.getName());

  /** Request名 */
  private ALStringField request_name;

  /** カテゴリID */
  private ALNumberField category_id;

  /** 重要度 */
  private ALNumberField priority;

  /** メモ */
  private ALStringField note;

  /** 金額 */
  private ALNumberField price;

  /** カテゴリ一覧 */
  private List categoryList;

  /** 申請先ユーザIDリスト */
  private ALStringField positions;

  /** 申請先一覧 */
  private ArrayList memberList;

  /** ファイルアップロードリスト */
  private List fileuploadList;

  /** 添付ファイル */
  private FileuploadLiteBean filebean = null;

  /** 添付フォルダ名 */
  private String folderName = null;

  private String org_id;

  private int uid;

  /** <code>login_user</code> ログインユーザー */
  private ALEipUser login_user;

  private DataContext dataContext;

  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    dataContext = DatabaseOrmService.getInstance().getDataContext();

    login_user = ALEipUtils.getALEipUser(rundata);

    uid = ALEipUtils.getUserId(rundata);
    org_id = DatabaseOrmService.getInstance().getOrgId(rundata);
    folderName = rundata.getParameters().getString("folderName");
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    // リクエスト名
    request_name = new ALStringField();
    request_name.setFieldName("表題");
    request_name.setTrim(true);
    // カテゴリID
    category_id = new ALNumberField();
    category_id.setFieldName("カテゴリ");
    // 重要度
    priority = new ALNumberField(3);
    priority.setFieldName("重要度");
    // メモ
    note = new ALStringField();
    note.setFieldName("申請内容");
    note.setTrim(false);
    // 金額
    price = new ALNumberField();
    price.setFieldName("金額");
    // 申請先のリスト
    positions = new ALStringField();
    positions.setFieldName("ユーザ名リスト");
    positions.setTrim(true);
    // 申請先一覧
    memberList = new ArrayList();
    // ファイルリスト
    fileuploadList = new ArrayList();
  }

  /**
   * 
   * @param rundata
   * @param context
   */
  public void loadCategoryList(RunData rundata, Context context) {
    categoryList = WorkflowUtils.loadCategoryList(rundata, context);
  }

  /**
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean setFormData(RunData rundata, Context context,
      ArrayList msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);
    if (res) {
      try {
        String userNames[] = rundata.getParameters().getStrings("positions");
        if (userNames != null && userNames.length > 0) {

          DataContext dataContext = DatabaseOrmService.getInstance()
              .getDataContext();
          SelectQuery query = new SelectQuery(TurbineUser.class);
          Expression exp1 = ExpressionFactory.inExp(
              TurbineUser.LOGIN_NAME_PROPERTY, userNames);
          Expression exp2 = ExpressionFactory.matchExp(
              TurbineUser.DISABLED_PROPERTY, "F");
          query.setQualifier(exp1);
          query.andQualifier(exp2);

          List list = dataContext.performQuery(query);

          TurbineUser record = null;
          int length = userNames.length;
          for (int i = 0; i < length; i++) {
            record = getEipUserRecord(list, userNames[i]);
            ALEipUser user = new ALEipUser();
            user.initField();
            user.setUserId(record.getUserId().intValue());
            user.setName(record.getLoginName());
            user.setAliasName(record.getFirstName(), record.getLastName());
            memberList.add(user);
          }
          fileuploadList = FileuploadUtils.getFileuploadList(rundata);
        }
      } catch (Exception ex) {
        logger.error("Exception", ex);
      }
    }
    return res;
  }

  /**
   * 指定したユーザ名のオブジェクトを取得する．
   * 
   * @param userList
   * @param userName
   * @return
   */
  private TurbineUser getEipUserRecord(List userList, String userName) {
    int size = userList.size();
    for (int i = 0; i < size; i++) {
      TurbineUser record = (TurbineUser) userList.get(i);
      if (record.getLoginName().equals(userName)) {
        return record;
      }
    }
    return null;
  }

  /**
   * リクエストの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  protected void setValidator() {
    // リクエスト名の文字数制限
    request_name.limitMaxLength(50);
    // 金額
    price.limitMaxValue(1000000000);
    // メモの文字数制限
    note.setNotNull(true);
    note.limitMaxLength(1000);
  }

  /**
   * リクエストのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  protected boolean validate(ArrayList msgList) {
    // リクエスト名
    request_name.validate(msgList);
    // メモ
    note.validate(msgList);
    // 金額
    price.validate(msgList);

    // 申請先
    if (memberList == null || memberList.size() <= 0) {
      msgList.add("『 <span class='em'>申請先</span> 』で指定してください。");
    }
    return (msgList.size() == 0);

  }

  /**
   * リクエストをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  protected boolean loadFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      EipTWorkflowRequest request = WorkflowUtils
          .getEipTWorkflowRequestForOwner(rundata, context);
      if (request == null)
        return false;

      // リクエスト名
      request_name.setValue(request.getRequestName());
      // カテゴリID
      category_id.setValue(request.getEipTWorkflowCategory().getCategoryId()
          .longValue());
      // 優先度
      priority.setValue(request.getPriority().longValue());
      // メモ
      note.setValue(request.getNote());
      // 金額
      price.setValue(request.getPrice().longValue());
      
      List maps = WorkflowUtils.getEipTWorkflowRequestMap(request);
      EipTWorkflowRequestMap map = null;
      int size = maps.size();
      for (int i = 0; i < size; i++) {
        map = (EipTWorkflowRequestMap) maps.get(i);
        int user_id = map.getUserId().intValue();
        if (!WorkflowUtils.DB_STATUS_REQUEST.equals(map.getStatus())) {
          memberList.add(ALEipUtils.getALEipUser(user_id));
        }
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * リクエストをデータベースから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  protected boolean deleteFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      EipTWorkflowRequest request = WorkflowUtils
          .getEipTWorkflowRequestForOwner(rundata, context);
      if (request == null)
        return false;

      // イベントログ用
      String catname = request.getEipTWorkflowCategory().getCategoryName();
      String reqname = request.getRequestName();

      // ファイル削除処理
      ArrayList fpaths = new ArrayList();
      SelectQuery query = new SelectQuery(EipTWorkflowFile.class);
      query.andQualifier(ExpressionFactory.matchDbExp(
          EipTWorkflowFile.EIP_TWORKFLOW_REQUEST_PROPERTY, request
              .getRequestId()));
      List files = dataContext.performQuery(query);
      if (files != null && files.size() > 0) {
        int fsize = files.size();
        for (int j = 0; j < fsize; j++) {
          fpaths.add(((EipTWorkflowFile) files.get(j)).getFilePath());
        }
      }

      if (fpaths.size() > 0) {
        // ローカルファイルに保存されているファイルを削除する．
        File file = null;
        int fsize = fpaths.size();
        for (int i = 0; i < fsize; i++) {
          file = new File(WorkflowUtils.getSaveDirPath(org_id, uid)
              + (String) fpaths.get(i));
          if (file.exists()) {
            file.delete();
          }
        }
      }

      // リクエストを削除
      dataContext.deleteObject(request);
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          request.getRequestId(), ALEventlogConstants.PORTLET_TYPE_WORKFLOW,
          catname + " " + reqname);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * リクエストをデータベースに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean insertFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      Date nowDate = Calendar.getInstance().getTime();

      EipTWorkflowCategory category = WorkflowUtils.getEipTWorkflowCategory(
          dataContext, Long.valueOf(category_id.getValue()));

      // 新規オブジェクトモデル
      EipTWorkflowRequest request = (EipTWorkflowRequest) dataContext
          .createAndRegisterNewObject(EipTWorkflowRequest.class);

      // リクエスト名
      request.setRequestName(request_name.getValue());
      // 親ID
      request.setParentId(Integer.valueOf(0));
      // カテゴリID
      request.setEipTWorkflowCategory(category);
      // ユーザーID
      request.setUserId(Integer
          .valueOf((int) login_user.getUserId().getValue()));
      // 優先度
      request.setPriority(Short.valueOf((short) priority.getValue()));
      // メモ
      request.setNote(note.getValue());
      // 金額
      request.setPrice(Long.valueOf(price.getValue()));
      // 承認フラグ
      request.setProgress(WorkflowUtils.DB_PROGRESS_WAIT);
      // 作成日
      request.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      request.setUpdateDate(Calendar.getInstance().getTime());

      // 申請先の設定
      insertEipTWorkflowRequestMap(request, login_user,
          WorkflowUtils.DB_STATUS_REQUEST, 0, nowDate);
      insertEipTWorkflowRequestMap(request, (ALEipUser) memberList.get(0),
          WorkflowUtils.DB_STATUS_CONFIRM, 1, nowDate);

      int size = memberList.size();
      for (int i = 1; i < size; i++) {
        insertEipTWorkflowRequestMap(request, (ALEipUser) memberList.get(i),
            WorkflowUtils.DB_STATUS_WAIT, (i + 1), nowDate);
      }

      // 添付ファイル処理
      if (!WorkflowUtils.insertFileDataDelegate(rundata, context, request,
          null, fileuploadList, folderName, msgList)) {
        return false;
      }

      // リクエストを登録
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          request.getRequestId(),
          ALEventlogConstants.PORTLET_TYPE_WORKFLOW,
          request.getEipTWorkflowCategory().getCategoryName() + " "
              + request_name.getValue());

      /* 次の申請先に新着ポートレット登録 */
      ALEipUser nextUser = (ALEipUser) memberList.get(0);
      ALAccessControlFactoryService aclservice = (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
          .getInstance())
          .getService(ALAccessControlFactoryService.SERVICE_NAME);
      ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

      if (aclhandler.hasAuthority((int) nextUser.getUserId().getValue(),
          ALAccessControlConstants.POERTLET_FEATURE_WORKFLOW_REQUEST_SELF,
          ALAccessControlConstants.VALUE_ACL_DETAIL)) {
        WhatsNewUtils.insertWhatsNew(dataContext,
            WhatsNewUtils.WHATS_NEW_TYPE_WORKFLOW_REQUEST, request
                .getRequestId().intValue(), (int) nextUser.getUserId()
                .getValue());
      }
      // 次の申請先にメール送信
      WorkflowUtils.sendMail(rundata, request, nextUser, new ArrayList());

      File folder = FileuploadUtils.getFolder(org_id, uid, folderName);
      // 添付ファイル保存先のフォルダを削除
      FileuploadUtils.deleteFolder(folder);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * データベースに格納されているリクエストを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean updateFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      EipTWorkflowRequest oldrequest = WorkflowUtils
          .getEipTWorkflowRequestForOwner(rundata, context);
      if (oldrequest == null)
        return false;

      Date nowDate = Calendar.getInstance().getTime();

      EipTWorkflowCategory category = WorkflowUtils.getEipTWorkflowCategory(
          dataContext, Long.valueOf(category_id.getValue()));

      // 新規オブジェクトモデル
      EipTWorkflowRequest request = (EipTWorkflowRequest) dataContext
          .createAndRegisterNewObject(EipTWorkflowRequest.class);

      // リクエスト名
      request.setRequestName(request_name.getValue());
      // 親ID
      if (oldrequest.getParentId().intValue() == 0) {
        request.setParentId(oldrequest.getRequestId());
      } else {
        request.setParentId(oldrequest.getParentId());
      }
      // カテゴリID
      request.setEipTWorkflowCategory(category);
      // ユーザーID
      request.setUserId(Integer
          .valueOf((int) login_user.getUserId().getValue()));
      // 優先度
      request.setPriority(Short.valueOf((short) priority.getValue()));
      // メモ
      request.setNote(note.getValue());
      // 金額
      request.setPrice(Long.valueOf(price.getValue()));
      // 承認フラグ
      request.setProgress(WorkflowUtils.DB_PROGRESS_WAIT);
      // 作成日
      request.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      request.setUpdateDate(Calendar.getInstance().getTime());

      // 申請先の設定
      insertEipTWorkflowRequestMap(request, login_user,
          WorkflowUtils.DB_STATUS_REQUEST, 0, nowDate);
      insertEipTWorkflowRequestMap(request, (ALEipUser) memberList.get(0),
          WorkflowUtils.DB_STATUS_CONFIRM, 1, nowDate);

      int size = memberList.size();
      for (int i = 1; i < size; i++) {
        insertEipTWorkflowRequestMap(request, (ALEipUser) memberList.get(i),
            WorkflowUtils.DB_STATUS_WAIT, (i + 1), nowDate);
      }

      // 添付ファイル処理
      if (!WorkflowUtils.insertFileDataDelegate(rundata, context, request,
          oldrequest, fileuploadList, folderName, msgList)) {
        return false;
      }

      // リクエストを登録
      dataContext.commitChanges();

      // 再申請済みを設定する
      oldrequest.setProgress(WorkflowUtils.DB_PROGRESS_REAPPLY);
      // リクエストを登録
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          request.getRequestId(),
          ALEventlogConstants.PORTLET_TYPE_WORKFLOW,
          request.getEipTWorkflowCategory().getCategoryName() + " "
              + request_name.getValue());

      /* 次の申請先に新着ポートレット登録 */
      ALEipUser nextUser = (ALEipUser) memberList.get(0);

      ALAccessControlFactoryService aclservice = (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
          .getInstance())
          .getService(ALAccessControlFactoryService.SERVICE_NAME);
      ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

      if (aclhandler.hasAuthority((int) nextUser.getUserId().getValue(),
          ALAccessControlConstants.POERTLET_FEATURE_WORKFLOW_REQUEST_SELF,
          ALAccessControlConstants.VALUE_ACL_DETAIL)) {
        WhatsNewUtils.insertWhatsNew(dataContext,
            WhatsNewUtils.WHATS_NEW_TYPE_WORKFLOW_REQUEST, request
                .getRequestId().intValue(), (int) nextUser.getUserId()
                .getValue());
      }
      // 次の申請先にメール送信
      WorkflowUtils.sendMail(rundata, request, nextUser, new ArrayList());

      File folder = FileuploadUtils.getFolder(org_id, uid, folderName);
      // 添付ファイル保存先のフォルダを削除
      FileuploadUtils.deleteFolder(folder);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 
   * @param request
   * @param user
   * @param status
   *            R: 申請 C : 確認 A: 承認 D: 否認
   */
  private void insertEipTWorkflowRequestMap(EipTWorkflowRequest request,
      ALEipUser user, String status, int order, Date now) {
    EipTWorkflowRequestMap map = (EipTWorkflowRequestMap) dataContext
        .createAndRegisterNewObject(EipTWorkflowRequestMap.class);
    int userid = (int) user.getUserId().getValue();
    map.setEipTWorkflowRequest(request);
    map.setUserId(Integer.valueOf(userid));
    // R: 申請 C : 確認 A: 承認 D: 否認
    map.setStatus(status);
    map.setOrderIndex(Integer.valueOf(order));
    map.setCreateDate(now);
    map.setUpdateDate(now);
  }

  /**
   * カテゴリIDを取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getCategoryId() {
    return category_id;
  }

  /**
   * メモを取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  /**
   * 金額を取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getPrice() {
    return price;
  }

  /**
   * 優先度を取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getPriority() {
    return priority;
  }

  /**
   * リクエスト名を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getRequestName() {
    return request_name;
  }

  /**
   * カテゴリ一覧を取得します。 <BR>
   * 
   * @return
   */
  public List getCategoryList() {
    return categoryList;
  }

  /**
   * グループメンバーを取得します。 <BR>
   * 
   * @return
   */
  public List getMemberList() {
    return memberList;
  }

  /**
   * 
   * @param groupname
   * @return
   */
  public List getUsers(String groupname) {
    return ALEipUtils.getUsers(groupname);
  }

  /**
   * 
   * @return
   */
  public Map getPostMap() {
    return ALEipManager.getInstance().getPostMap();
  }

  public List getAttachmentFileNameList() {
    return fileuploadList;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   * 
   * @return
   */
  public String getAclPortletFeature() {
    return ALAccessControlConstants.POERTLET_FEATURE_WORKFLOW_REQUEST_SELF;
  }
}
