#include "expat_parser.h"

#ifdef USE_EXPAT

#include "../util/platform.h"
#include "../util/sstream.h"
#include "../XML/encoding_name.h"

#include <fstream>

using namespace aka2;

expat_parser::expat_parser() : dochandler_(0), dtdhandler_(0), buffer_size_(1024) {
#ifdef USE_BABEL
  token_translator = babel::manual_translate_engine<std::string, std::string>::
    create(babel::base_encoding::utf8, babel_util::default_encoding_);
  chars_translator = babel::manual_translate_engine<std::string, std::string>::
    create(babel::base_encoding::utf8, babel_util::default_encoding_);
#endif
}

#ifdef USE_BABEL

namespace {

  void create_translator(const std::string &encoding, 
			 babel::bbl_translater<std::string, std::string> &tr) {
    std::string canonicalized = canonicalize_encoding_name(encoding);
    int enc = babel_util::get_encoding_from_label(canonicalized.c_str());
    if (enc == babel::base_encoding::unknown) {
      std::string message("Encoding \"");
      message += encoding + "\" not found.";
      throw tagged_error("encoding", encoding, "not found.", __FILE__, __LINE__);
    }
    
    tr = babel::manual_translate_engine<std::string, std::string>::
      create(enc, babel::base_encoding::utf8);
  }

}

const std::string &expat_parser::translate_token(const std::string &token) {
  token_translator.clear();
  token_translator.translate(token);
  token_translator.flush();
  return token_translator.get_string();
}

#endif


void expat_parser::process_attributes(const XML_Char **atts, sax_attributes &attrs) {
  for (const XML_Char **ptr = atts; *ptr != 0; ptr += 2) {
#if defined(USE_BABEL)
    std::string tagname = translate_token(ptr[0]);
    std::string value = translate_token(ptr[1]);
    attrs.push_back(sax_attribute(tagname, value));
#else
    attrs.push_back(sax_attribute(ptr[0], ptr[1]));
#endif
  }
}


void expat_parser::startElement(void *userData,
				const XML_Char *name,
				const XML_Char **atts) {
  sax_attributes saxattrs;
  expat_parser *handler = static_cast<expat_parser*>(userData);
  handler->process_attributes(atts, saxattrs);

#if !defined(USE_BABEL)
  handler->dochandler_->startElement(name, saxattrs);
#else
  const std::string &tagname = handler->translate_token(name);
  handler->dochandler_->startElement(tagname, saxattrs);
  handler->chars_translator.clear();
#endif

}


void expat_parser::endElement(void *userData,
			      const XML_Char *name) {

  expat_parser *handler = static_cast<expat_parser*>(userData);
#if defined(USE_BABEL)
  const std::string &tagname = handler->translate_token(name);
  handler->dochandler_->endElement(tagname);
#else
  handler->dochandler_->endElement(name);
#endif
}

void expat_parser::characters(void *userData,
			       const XML_Char *data,
			       int len) {
  expat_parser *handler = static_cast<expat_parser*>(userData);
  
#if defined(USE_BABEL)
  handler->chars_translator.translate(std::string(data, len));
  std::string chars = handler->chars_translator.pook_buffer();
  handler->dochandler_->characters(chars.c_str(), chars.size());
#else
  handler->dochandler_->characters(data, len);
#endif
}

int expat_parser::unknownEncodingHandler(void *encodingHandlerData,
					 const XML_Char *name,
					 XML_Encoding *info) {
  static_cast<expat_parser*>(encodingHandlerData)->encoding_ = name;
  return XML_STATUS_OK;
}


void expat_parser::startDoctypeDeclHandler(void *userData,
					   const XML_Char *doctypeName,
					   const XML_Char *sysid,
					   const XML_Char *pubid,
					   int has_internal_subset) {
  expat_parser *parser = static_cast<expat_parser*>(userData);
  parser->dtdhandler_->startDoctypeDeclHandler(doctypeName,
					       sysid, pubid,
					       has_internal_subset);
}

void expat_parser::endDoctypeDeclHandler(void *userData) {
  expat_parser *parser = static_cast<expat_parser*>(userData);
  parser->dtdhandler_->endDoctypeDeclHandler();
}

int expat_parser::externalEntityRefHandler(XML_Parser p,
					   const XML_Char *context,
					   const XML_Char *base,
					   const XML_Char *systemId,
					   const XML_Char *publicId) {

  expat_parser *ep = static_cast<expat_parser*>(XML_GetUserData(p));

  XML_Parser parser = 
    XML_ExternalEntityParserCreate(p, context, 0); // No-encoding support.!!!!!!
  ep->parser_stack_.push(parser);
  if (base != 0)
    XML_SetBase(parser, base);


  unsigned int read_size;
  bool is_final = false;
  std::string read;

  std::string fullpath;
  if (base != 0)
    fullpath = base;
  fullpath += systemId;

  std::ifstream istm(fullpath.c_str());
  
  read_size = readsome(istm, read, ep->buffer_size_);
  XML_Status res;
  
  do {
    is_final = read.size() == 0;
    res = XML_Parse(parser, read.c_str(), read.size(), is_final);
    if (res != XML_STATUS_OK)
      break;
    if (!is_final) {
      read = "";
      readsome(istm, read, ep->buffer_size_);
    }
  } while (!is_final);
  
  ep->free_parser(parser);
  return res;
}




void expat_parser::elementDeclHandler(void *userData, const XML_Char *name, XML_Content *model) {
  expat_parser *ep = static_cast<expat_parser*>(userData);
  ep->dtdhandler_->elementDeclHandler(name, model);
  XML_FreeContentModel(ep->parser_stack_.top(), model);
}

void expat_parser::attlistDeclHandler(void *userData,
				      const XML_Char *elname,
				      const XML_Char *attname,
				      const XML_Char *att_type,
				      const XML_Char *dflt,
				      int isrequired) {
  expat_parser *ep = static_cast<expat_parser*>(userData);
  ep->dtdhandler_->attlistDeclHandler(elname, attname, 
				      att_type, dflt, isrequired);
}


void expat_parser::set_base(const std::string &base) {
  base_ = base;
}




void expat_parser::parse_buffer(const char *buffer, int len,
				const std::string &source_name) {
  
  XML_Parser parser = create_parser(0, dochandler_, dtdhandler_);

  XML_Status res = XML_Parse(parser, buffer, len, 1);
  
  if (res == XML_STATUS_OK) {
    free_parser(parser);
    return;
  }
  
#if !defined(USE_BABEL) // No encoding support.
  std::string message = create_message(parser, source_name);
  free_parser(parser);
  throw error(message, __FILE__, __LINE__);
#endif
  
#if defined(USE_BABEL)
  XML_Error errcode = XML_GetErrorCode(parser);
  if (errcode != XML_ERROR_UNKNOWN_ENCODING) {
    std::string message = create_message(parser, source_name);
    free_parser(parser);
    throw error(message, __FILE__, __LINE__);
  }
  
  free_parser(parser);
  
  babel::bbl_translater<std::string, std::string> translator;
  create_translator(encoding_, translator);
  
  parser = create_parser("UTF-8", dochandler_, dtdhandler_);

  translator.translate(buffer);
  const std::string &translated = translator.get_string();
  res = XML_Parse(parser, translated.c_str(), translated.size(), 1);
  
  if (res == XML_STATUS_OK) {
    free_parser(parser);
    return;
  }
  
  std::string message = create_message(parser, source_name);
  free_parser(parser);
  throw error(message, __FILE__, __LINE__);
#endif
  assert(!"Must not reach here.");
}

void expat_parser::parse_istream(std::istream &istm, 
				 const std::string &source_name) {
  
  unsigned int read_size;
  bool has_xmldecl = false;
  bool is_final = false;
  std::string read;
  
  read_size = readsome(istm, read, buffer_size_);
  
  // Check document has xml declaration.
  if (read.substr(0, 2) == "<?") {
    has_xmldecl = true;
    while (true) {
      // Check the end of XML declaration.
      bool decl_end = read.find("?>") != std::string::npos;
      if (decl_end)
	break;
      read_size = readsome(istm, read, buffer_size_);
      if (read_size == 0) {
	is_final = true;
	break;
      }
    }
  }
  
  // The first trial.  Don't considering encodings.
  XML_Parser parser = create_parser(0, dochandler_, dtdhandler_);

  XML_Status res;
  
  do {
    is_final = read.size() == 0;
    res = XML_Parse(parser, read.c_str(), read.size(), is_final);
    if (res != XML_STATUS_OK)
      break;
    if (!is_final) {
      read = "";
      readsome(istm, read, buffer_size_);
    }
  } while (!is_final);
  
  if (res == XML_STATUS_OK) {
    free_parser(parser);
    return;
  }
  
#if !defined(USE_BABEL) // no encoding support.
  std::string message = create_message(parser, source_name);
  free_parser(parser);
  throw error(message, __FILE__, __LINE__);
#endif
  
#if defined(USE_BABEL)
  
  // If error is not encoding one, or no-xml declaration to spceify encoding,
  // then throw exception.
  XML_Error errcode = XML_GetErrorCode(parser);
  if ((errcode != XML_ERROR_UNKNOWN_ENCODING) || !has_xmldecl) {
    std::string message = create_message(parser, source_name);
    free_parser(parser);
    throw error(message, __FILE__, __LINE__);
  }    
  
  
  // Try translation with given encoding.
  babel::bbl_translater<std::string, std::string> translator;
  create_translator(encoding_, translator);
  
  parser = create_parser("UTF-8", dochandler_, dtdhandler_);
  
  do {
    std::string translated;
    is_final = read.size() == 0;
    if (!is_final) {
      translator.translate(read);
      translated = translator.pook_buffer();
    }
    else {
      translated = "";
    }
    res = XML_Parse(parser, translated.c_str(), translated.size(), is_final);
    if (res != XML_STATUS_OK)
      break;
    if (!is_final) {
      read = "";
      readsome(istm, read, buffer_size_);
    }
  } while (!is_final);

  if (res == XML_STATUS_OK) {
    free_parser(parser);
    return;
  }

  std::string message = create_message(parser, source_name);
  free_parser(parser);
  throw error(message, __FILE__, __LINE__);

#endif // USE_BABEL

  assert(!"Must not reach here.");
}


std::string expat_parser::create_message(XML_Parser parser,
					 const std::string &source_name) {
					       
  std::ostringstream ostm;

  XML_Error errcode = XML_GetErrorCode(parser);
  const XML_LChar *msg = XML_ErrorString(errcode);
  int linenum = XML_GetCurrentLineNumber(parser);
  int columnnum = XML_GetCurrentColumnNumber(parser);
  ostm << source_name << ':' << linenum << ':' << columnnum << ' ' << msg;
  return ostm.rdbuf()->str();
}

XML_Parser expat_parser::create_parser(const char *encoding, 
				       expat_document_handler *dochandler,
				       expat_dtd_handler *dtdhandler) {

  XML_Parser parser = XML_ParserCreate(encoding);
  XML_SetUserData(parser, this);

  if (!base_.empty())
    XML_SetBase(parser, base_.c_str());

  if (dochandler_ != 0) {
    XML_SetStartElementHandler(parser, &expat_parser::startElement);
    XML_SetEndElementHandler(parser, &expat_parser::endElement);
    XML_SetCharacterDataHandler(parser, &expat_parser::characters);
    XML_SetUnknownEncodingHandler(parser, &expat_parser::unknownEncodingHandler, this);
  }
  if (dtdhandler_ != 0) {
    XML_SetParamEntityParsing(parser,
			      XML_PARAM_ENTITY_PARSING_UNLESS_STANDALONE);
    XML_SetDoctypeDeclHandler(parser, 
			      &expat_parser::startDoctypeDeclHandler,
			      &expat_parser::endDoctypeDeclHandler);
    XML_SetExternalEntityRefHandler(parser, 
				    &expat_parser::externalEntityRefHandler);
    XML_SetElementDeclHandler(parser, &expat_parser::elementDeclHandler);
    XML_SetAttlistDeclHandler(parser, &expat_parser::attlistDeclHandler);
  }

  parser_stack_.push(parser);

  return parser;
}

void expat_parser::free_parser(XML_Parser parser) {
  assert(parser_stack_.top() == parser);
  parser_stack_.pop();
  XML_ParserFree(parser);
}


#endif
