#include "osixaka.h"
#include <akaxiso/schema/schema.h>
#include "xschemadeserializer.h"
#include "generator.h"
#include "dtd_processor.h"
#include <iostream>

namespace {
#ifdef _WIN32
  const char to_replace = '/';
  const char replaced = '\\';
#else
  const char to_replace = '\\';
  const char replaced = '/';
#endif

  std::string replace_path_seperator(const std::string &path) {
    std::string ret(path);
    for (std::string::size_type i = 0; i < path.size(); ++i)
      if (ret[i] == to_replace)
        ret[i] = replaced;
    return ret;
  }
  

  std::string get_dirpath(const std::string &path) {
    std::string::size_type pos = path.find_last_of("/\\");
    if (pos == std::string::npos) { // not found!
      return replace_path_seperator("./");
    }      
    else {
      return replace_path_seperator(path.substr(0, pos + 1));
    }
  }

  std::string get_basename(const std::string &path) {
    std::string res;
    std::string::size_type pos = path.find_last_of("/\\");
    if (pos == std::string::npos) { // not found!
      res = path;
    }      
    else {
      res = path.substr(pos + 1);
    }

    pos = res.find_last_of('.');
    if (pos == std::string::npos)
      return replace_path_seperator(res);
    else
      return replace_path_seperator(res.substr(0, pos));
  }
}

void osixaka::initialize() {
  aka::xmlns("aka", "http://akaxiso.sourceforge.jp/akaxiso2");
  aka::xmlns("std", "http://akaxiso.sourceforge.jp/osixaka/c++-std");
  aka::xmlns("osx", "http://akaxiso.sourceforge.jp/osixaka/preference");
  xs::instantiate_xiso();
  registry::initialize();
}

int osixaka::deserialize(const std::string &filepath) {
  xschemadeserializer deserializer(ostm_, verbose_, do_dump_);
  dirpath_ = get_dirpath(filepath);
  basename_ = get_basename(filepath);
  target_schema_ = 
    aka::shared_ptr<xs::schema>(deserializer.deserialize(replace_path_seperator(filepath)));
  return target_schema_.get() == 0 ? 1 : 0;
}

int osixaka::process_dtd(const std::string &filepath) {
  dtd_processor processor(registry_, ostm_, verbose_, do_dump_);
  dirpath_ = get_dirpath(filepath);
  basename_ = get_basename(filepath);
  processor.process(dirpath_, filepath);
  target_schema_ = aka::shared_ptr<xs::schema>(processor.get_schema());
  return target_schema_.get() == 0 ? 1 : 0;
}

int osixaka::process() {
  try {
    processor prc(registry_, dirpath_, ostm_, verbose_);
    prc.preprocess(*target_schema_);
    prc.process(*target_schema_);
    return 0;
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl;
    ostm_ << "Failed to generate files. Aborting..." << std::endl;
    return 1;
  }
  catch ( ... ) {
    ostm_ << "Unknown exception caught. Aborting..." << std::endl;
    return 1;
  }
}  

void osixaka::generate(const std::string &outputdir) {
  generator gen(std::cerr, outputdir);
  gen.generate(registry_, basename_);
}

void osixaka::akaxisonize() {
  akaxisonizer_.replace_any();
  akaxisonizer_.akaxisonize();
  akaxisonizer_.add_missings();
}


void osixaka::load_defaults() {
  registry_.load_defaults();
  registry_.register_ns();
}

void osixaka::load_preference(std::istream &file) {
  registry_.load_preference(file);
  registry_.register_ns();
}

void osixaka::save_preference(std::ostream &file) {
  registry_.save_preference(file);
}
