/* -*- c++ -*- */
#ifndef AKAXISO2_XML_SERIALIZER_FORMATTER_BASE_H__
#define AKAXISO2_XML_SERIALIZER_FORMATTER_BASE_H__

/**
 * @file akaxiso2/XML/serializer/formatter_base.h 
 * @brief Base class of formatter
 */

#include <akaxiso2/util/iosfwd.h>
#include <akaxiso2/util/ustring.h>
#include <akaxiso2/XML/encoding_name.h>

namespace aka2 {

  struct bom {
    char chars_[16];
    unsigned int length_;
  };

  /** 
   * @brief Base class of formatter. 
   *
   * This class implements text-formatter functionality for XML serializer.\n
   * Any derived formatter classes should implement some pure-virtual members 
   * to achive encoding translation. 
   */
  class formatter_base {
  public:
    formatter_base() : bom_(0), ostm_(0), fd_(-1) {}
    virtual ~formatter_base() {  }

    /** 
     * @brief Set encoding name.
     *
     * Overriding method should setup encoding translator.
     * @param encoding encoding name.
     */
    virtual void set_encoding(const std::string &encoding) = 0;

    /**
     * @brief Set std::ostream to output text.
     * @param ostm std::ostream to output text.
     */
    void prepare(std::ostream &ostm);
    /**
     * @brief Finish serialization.
     */ 
    void finish();

    void write(const std::string &value);
    void write_attribute_entity(const std::string &entity);
    void write_text_entity(const std::string &entity);

  private:
    /** 
     * @brief Translating local-code-page to UCS2.
     * @param source source string in LCP to be converted to UCS2.
     * @return converted UCS2 string.
     */
    virtual ustring lcp_to_ucs2(const std::string &source) = 0;

    /**
     * @brief Translate UCS-2 string to output-encoding specified by set_encoding 
     * and output converted string to ostream passed by prepare() method.
     * @param entity UCS2 string to write.
     */
    virtual void write(const ustring &entity) = 0;
    void write_entity(const std::string &entity, const int escape_index);
    void set_binary_mode_for_stdio();
    void revert_stream_mode();

  protected:
    const bom* bom_;
    const bom* check_bom(const std::string &encoding) const;
    std::ostream *ostm_;
  private:
    std::string encoding_;
    int saved_stream_mode_;
    int fd_;
  };

} // namespace aka2

#endif
