/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_NAMESPACES_H__
#define AKAXISO2_FRAMEWORK_NAMESPACES_H__

/**
 * @file akaxiso2/framework/namespaces.h
 * @brief namespace-related classes/functions.
 */

#include <akaxiso2/framework/types.h>
#include <akaxiso2/util/mutex.h>
#include <map>
#include <vector>

namespace aka2 {

  /**
   * @brief Register and associate namespace URI with its prefix.
   *
   * Register and associate namespace URI and its prefix.\n
   * Akaxiso's namespace database is a global-static object.\n
   * Association of URI and prefix is used in the library-scope.\n
   * @see @ref namespace
   *
   * @param prefix prefix of namespace.
   * @param uri of namespace.
   */
  void xmlns(const std::string &prefix, const std::string &uri);

  /**
   * @brief Get namespace URI by its prefix.
   * @param prefix namespace prefix.
   * @return namespace URI.
   */
  const std::string &get_namespace_uri(const std::string &prefix);

  const std::string &get_namespace_uri(const id_type ns_id);

  /**
   * @brief Get namespace prefix by its URI.
   * @param uri Namespace URI
   * @return Namespace URI.
   */
  const std::string &get_prefix(const std::string &uri);

  const std::string &get_prefix(const id_type &ns_id);
  const id_type get_namespace_id(const std::string &uri);
  const id_type get_prefix_id(const std::string &prefix);


  class namespace_map {
    typedef std::map<id_type, std::string> urimap_type;
    typedef std::map<std::string, id_type> idmap_type;

  public:
    namespace_map(id_type initial_id = 1) : current_id_(initial_id) {}
    const id_type register_namespace_uri(const std::string &uri);
    const std::string &get_namespace_uri(const id_type id) const;
    const id_type get_namespace_id(const std::string &uri) const;
    bool is_uri_registered(const std::string &uri) const;

    // Used for system globals.
    void initialize();

    void clear();

  private:
    id_type current_id_;
    urimap_type urimap_;
    idmap_type idmap_;
  };


  class any_namespace_map : public namespace_map {
  public:
    any_namespace_map() : namespace_map(0x8000) {}
    const std::string &get_namespace_uri(const id_type id) const;
    const id_type get_namespace_id(const std::string &uri);
    bool is_uri_registered(const std::string &uri) const;

    void clear();
  private:
    mutable mutex mutex_;
  };


  class any_prefix_cache {
  public:
    any_prefix_cache(const std::string &prefix = "tmpns") : prefix_(prefix) { } //!!!!!!!!!!
    const std::string &get_prefix(id_type nsid) const;
    void clear();
  private:
    std::string prefix_;
    mutable std::vector<std::string> cache_;
  };


  /**
   * @brief associative map for namespace prefix and URI.
   */
  class prefix_map {
    typedef std::vector<std::string> prefixes;
    typedef std::vector<id_type> id_stack;

    // URI -> prefix mapping.
    typedef std::map<id_type, prefixes> prefixmap_type;
    // Preix is unique, but some prefixes may be associated with the same id.
    typedef std::map<std::string, id_stack> idmap_type;
  public:
    typedef prefixmap_type::const_iterator const_iterator;

    prefix_map();
    const_iterator begin() const { return prefixmap_.begin(); }
    const_iterator end() const { return prefixmap_.end(); }

    /**
     * @brief assign prefix to namespace URI.
     *
     * Assign prefix to namespace URI.\n
     * Users can assign prefix for multiple times, and latest prefix assignment is active.\n
     * xml namespace must not be assigned, because it has its own namespace by definition.\n
     * @param prefix prefix
     * @param uri namespace URI
     * @exception aka2::error thrown if xml namespace is re-assigined, 
     * or URI is not unregistered by aka2::xmlns().
     */
    void assign_prefix(const std::string &prefix, const std::string &uri);
    /**
     * @brief clear assigned prefix.
     * @param prefix prefix.
     * @exception aka2::error thrown if prefix not assigned.
     */
    void clear_prefix(const std::string &prefix);

    const id_type get_prefix_id(const std::string &prefix) const;
    const std::string &get_prefix(const id_type id) const;

    static void initialize();
    void clear();

  private:
    static bool is_prefix_xml(const std::string &prefix);
    static std::string xml_prefix_;
    prefixmap_type prefixmap_;
    any_prefix_cache any_cache_;
    idmap_type idmap_;
  };

} // namespace aka2

#endif
