#include "qname.h"
#include <cassert>
#include "namespace_statics.h"
#include "../util/string_funcs.h"

using namespace aka2;

void qname::validate_name() const {
  if (name_.empty())
    throw error("localpart of qualified name is empty.", __FILE__, __LINE__);
  if (name_[0] == ':') {
    std::string message = "Qualified name " + quote(prefix() + ":" + local()) 
      + " has '::' in it's name field.";
    throw error(message, __FILE__, __LINE__);
  }
}

void qname::set(const std::string &uri, const std::string &name) {
  namespace_id_ = g_namespaces_.get_namespace_id(uri);
  if (namespace_id_ == unregistered_token)
    namespace_id_ = g_any_namespaces_.get_namespace_id(uri);
  name_ = name;
  validate_name();
}

void qname::set(const id_type namespace_id, const std::string &name) {
  namespace_id_ = namespace_id;
  name_ = name;
  validate_name();
}

void qname::set(const std::string &rawname, const prefix_map &pfs) {
  std::string::size_type pos = rawname.find_first_of(':');

  if (pos == std::string::npos) { // local, not qualified.
    namespace_id_ = empty_token;
    name_ = rawname;
  }
  else {
    std::string prefix = rawname.substr(0, pos);
    namespace_id_ = pfs.get_prefix_id(prefix);
    if (namespace_id_ == unregistered_token)
      throw error("namespace prefix(\"" + prefix + "\") is not registered.",
		  __FILE__, __LINE__);
    name_ = rawname.substr(pos + 1, rawname.length() - pos);
  }
  validate_name();
}

void qname::set(const std::string &rawname) {
  set(rawname, *g_pfs_);
}


std::string qname::qualified(const prefix_map &pfs) const{
  assert(namespace_id_ != unregistered_token);

  if (is_qualified()) {
    const std::string &prefix = pfs.get_prefix(namespace_id_);
    prefix_cache_ = prefix.c_str();
    if (prefix.empty()) // Default namespace.  Prefix is "".
      return name_;
    std::string qualified = prefix + ':' + name_;
    return qualified;
  }
  else
    return name_;
}

std::string qname::qualified() const{
  return qualified(*g_pfs_);
}

std::string &qname::local() {
  return name_;
}

const std::string &qname::local() const {
  return name_;
}

const std::string &qname::prefix(const prefix_map &pfs) const {
  return pfs.get_prefix(namespace_id_);
}

const std::string &qname::prefix() const {
  return g_pfs_->get_prefix(namespace_id_);
}

bool qname::empty() const {
  return (namespace_id_ == empty_token) && name_.empty();
}
