/* -*- c++ -*- */
#ifndef AKAXISO2_UTIL_DEEP_PTR_H__
#define AKAXISO2_UTIL_DEEP_PTR_H__

/**
 * @file akaxiso2/util/deep_ptr.h
 * @brief deep pointer (smart pointer with deep copy)
 * @author Shin'ya MORINO (smorino@users.sourceforge.jp)
 */

namespace aka2 {

  /**
   * @brief smart pointer with deep copy.
   *
   * Deep ptr is a smart pointer, but it replicate object on copy (deep copy).
   */
  template <class T>
  class deep_ptr {
  public:
    typedef T value_type;

    /**
     * @brief constructor
     * @param ptr pointer
     */
    explicit deep_ptr(T* ptr = 0) : ptr_(ptr) {}
    ~deep_ptr() {
      delete ptr_;
    }

    /**
     * @brief copy constructor
     *
     * Instance(ptr_) in deep_ptr<> is replicated.
     * @param rhs to copy
     */
    deep_ptr(const deep_ptr &rhs) {
      if (rhs.ptr_ == 0)
	ptr_ = 0;
      else
	ptr_ = new T(*rhs.ptr_);
    }

    /**
     * @brief reset with another pointer.
     * @param ptr pointer
     */
    void reset(T* ptr = 0) {
      delete ptr_;
      ptr_ = ptr;
    }

    /**
     * @brief substution operator
     * @param rhs to copy
     * @return *this
     */
    const deep_ptr<T>& operator=(const deep_ptr<T> &rhs) {
      delete ptr_;
      if (rhs.ptr_ == 0)
	ptr_ = 0;
      else
	ptr_ = new T(*rhs.ptr_);
      return *this;
    }

    /**
     * @brief equality operator
     * @param rhs to compare
     * @return true if pointer value is the same, otherwise false.
     */
    bool operator ==(const deep_ptr<T> &rhs) {
      return ptr_ == rhs.ptr_;
    }

    /**
     * @brief getter for pointer
     * @return pointer
     */
    T* get() const { return ptr_; }

    /**
     * @brief dereference operator
     * @return T&
     */
    T& operator*() const { return *ptr_; }
    /**
     * @brief arrow operator
     * @return T*
     */
    T* operator->() const { return ptr_; }

  private:  
    T* ptr_;
  };

} // namespace aka2

#endif
