/* -*- c++ -*- */
#ifndef AKAXISO2_XML_DESERIALIZER_DESERIALIZER_H__
#define AKAXISO2_XML_DESERIALIZER_DESERIALIZER_H__

/**
 * @file akaxiso2/XML/deserializer/deserializer.h 
 * @brief declarations for deserializer interface and deserialize function.
 */

#include <akaxiso2/configuration.h>
#include <akaxiso2/framework/document.h>
#include <akaxiso2/util/iosfwd.h>

namespace aka2 {

  /**
   * @brief XML deserializer interface.
   */
  class deserializer {
  public:
    virtual ~deserializer() {}
    
    /**
     * @brief Deserialize XML document
     * @param istm input source
     * @return parsed document
     * @exception aka2::error any deserialization exception
     */
    virtual document deserialize(std::istream &istm) = 0;  

    /**
     * @brief Deserialize XML document
     * @param doc input source
     * @return parsed document
     */
    virtual document deserialize(const char *doc) = 0;

    /**
     * @brief Deserialize XML document
     * @param doc input source as string
     * @return parsed document
     * @exception aka2::error any deserialization exception
     */
    virtual document deserialize(const std::string &doc) = 0;
    
    /**
     * @brief Deserialize XML document from stdin.
     * @return parsed document. 
     * @exception aka2::error any deserialization exception.
     */
    virtual document deserialize_stdin() = 0;
    
    /** 
     * @brief Deserialize XML document
     * @param filename file path of input XML document.
     * @return parsed document. 
     * @exception aka2::error any deserialization exception.
     */
    virtual document deserialize_file(const std::string &filename) = 0; 

    /**
     * @brief get entity complements.
     * @return entity_complements instance for deserialization (same instance passed to leaf<> classes.
     */
    virtual entity_complements &get_entity_complements() = 0;
  };

  /** 
   * @brief Deserialize XML document by using the default XML deserializer
   * (aka2::xml_deserializer).
   * @param istm input source.
   * @return parsed document. 
   * @exception aka2::error any deserialization exception.
   */
  document deserialize(std::istream &istm);

  /** 
   * @brief Deserialize XML document by using the default XML deserializer
   * (aka2::xml_deserializer).
   * @param doc input source of character buffer.
   * @return parsed document. 
   * @exception aka2::error any deserialization exception.
   */
  document deserialize(const char *doc);

  /** 
   * @brief Deserialize XML document by using the default XML deserializer
   * (aka2::xml_deserializer).
   * @param doc input source as std::string.
   * @return parsed document. 
   * @exception aka2::error any deserialization exception.
   */
  document deserialize(const std::string &doc);

  /** 
   * @brief Deserialize XML document by using the default XML deserializer
   * (aka2::xml_deserializer).
   * @return parsed document. 
   * @exception aka2::error any deserialization exception.
   */
  document deserialize_stdin();

  /**
   * @brief Deserialize XML document by using the default XML deserializer
   * (aka2::xml_deserializer).
   * @param filename file name of input XML document.
   * @return parsed document. 
   * @exception aka2::error any deserialization exception.
   */
  document deserialize_file(const std::string &filename);
} // namespace aka2

#endif
