/* -*- c++ -*- */
#include "akaxiso2.h"
#include "XML/serializer/formatter.h"
#include "framework/namespaces.h"
#include "framework/type_registry.h"
#include "framework/closure.h"
#include "framework/any.h"
#include "framework/namespace_statics.h"
#include "framework/any_op.h"
#include "framework/entity_complements.h"
#include "framework/document_factory.h"
#include "transcoders/encoding_name.h"
#include "builtin/builtin_xiso.h"
#include "builtin/schema_builtin_xiso.h"
#include <stdio.h>

#ifdef AKAXISO2_USE_XERCESC
#include <xercesc/util/PlatformUtils.hpp> /* {Xerces-C++} */
#endif

#ifdef _WIN32
#include "transcoders/win32_transcoder.h"
#endif


namespace aka2 {

  const char akaxiso2_ns_uri[] = { "http://akaxiso.sourceforge.jp/akaxiso2/2.0-beta3" };
  const char xmlschema_ns_uri[] = { "http://www.w3.org/2001/XMLSchema" };

  bool g_initialized_ = false;
  int init_count_ = 0;

  type_registry     g_type_registry_;
  entity_complements *g_ecomp_;
  document_factory  g_doc_factory_;
  std::string g_default_encoding_;
  transcoder_factory g_external_factory_ = 0;
}


using namespace aka2;

entity_complements &aka2::system_entity_complements() {
  return *g_ecomp_;
}

entity_complements *aka2::system_entity_complements_clone(bool copy_prefixes, 
							  transcoder_factory trfact) {
  return entity_complements::clone(*g_ecomp_, copy_prefixes, trfact);
}



document_factory& aka2::system_document_factory() {
  return g_doc_factory_;
} 

type_registry &aka2::system_type_registry() {
  return g_type_registry_;
}

void aka2::set_default_encoding(const std::string &encoding) {
  if (g_initialized_) {
    throw error("aka::set_default_encoding() must be called before aka::initialize().",
		__FILE__, __LINE__);
  }
  g_default_encoding_ = encoding;
}


int aka2::initialize() {
  if (++init_count_ != 1)
    return 0;

#ifdef _WIN32
  win32_encodings::initialize();
#endif

#ifdef AKAXISO2_USE_XERCESC
  xercesc::XMLPlatformUtils::Initialize();
#endif

#ifdef AKAXISO2_DEFAULT_ENCODING
  if (g_default_encoding_.empty())
    g_default_encoding_ = AKAXISO2_DEFAULT_ENCODING;
#endif


  if (g_default_encoding_.empty())
    guess_default_encoding();

  pivot_transcoder::initialize_static_transcoder(g_external_factory_);
  g_ecomp_ = entity_complements::create(g_external_factory_);

  g_namespaces_ .initialize();
  g_doc_factory_  .clear();
  g_type_registry_.initialize();
  prefix_map    ::initialize();
  g_ecomp_->clear_all();
  g_pfs_ = &g_ecomp_->get_prefixes();

  xmlns("aka2", akaxiso2_ns_uri);
  xmlns("xs", xmlschema_ns_uri);

  initialize_builtins();
  xs::initialize_schema_builtins();

  // Depends on ns-prefix "xs:".
  initialize_any_names();

  wildcard_op::initialize();
  any_op::initialize();
  any_array_op::initialize();
  enclose_op::initialize();
  disclose_op::initialize();

  g_initialized_ = true;

  return 1;
}

void aka2::uninitialize() {
  if (init_count_ == 0 || --init_count_ != 0)
    return;

  g_namespaces_ .clear();
  g_ecomp_->clear_all();
  g_pfs_ = 0;
  g_doc_factory_  .clear();
  g_type_registry_.uninitialize();
  delete g_ecomp_;
  pivot_transcoder::uninitialize_static_transcoder();

  g_initialized_ = false;
  
#ifdef AKAXISO2_USE_XERCESC
  xercesc::XMLPlatformUtils::Terminate();
#endif
#ifdef _WIN32
  win32_encodings::uninitialize();
#endif

}


const char *aka2::version() {
  return "2.0-beta3";
}

const char *aka2::compiler() {
  static char compiler_ver[256];

  strcpy(compiler_ver, "unknown");

#ifdef __GNUC__
# ifdef __GNUC_PATCHLEVEL__
  sprintf(compiler_ver, "GCC %d.%d.%d", __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
# else
  sprintf(compiler_ver, "GCC %d.%d", __GNUC__, __GNUC_MINOR__);
# endif
#endif

#ifdef _MSC_VER
  sprintf(compiler_ver, "MSC %04d", _MSC_VER);
#endif

#ifdef __BORLANDC__
  sprintf(compiler_ver, "Borland C %04d", __BORLANDC__);
#endif

  return compiler_ver;
}

#include "XML/deserializer/xml_deserializers.h"

const char *aka2::default_parser_name() {
  return AKAXISO2_DEFAULT_PARSER;
}

#include "transcoders/builtin_transcoders.h"

const char *aka2::default_transcoder_name() {
  return AKAXISO2_DEFAULT_TRANSCODER;
}

