/* -*- c++ -*- */
#include "element_props.h"
#include "operators.h"
#include "fixed.h"
#include "entity_complements.h"

#include <assert.h>
#include <algorithm>

#include "membertype.h"
#include "item.h"

#include "../util/sstream.h"
#include "../util/string_funcs.h"
#include "../util/tokenizer.h"

#include "../builtin/validator.h"



using namespace aka2;



bool element_props::check_emptiable_recursive(const element_props &props, const element_op &op) {
  
  if (props.is_emptiable())
    return true;

  if (props.occ_.in_range(0))
    return true;
  
  switch (op.get_schematype()) {
  case sequence_id: {
    if (props.is_element())
      return false; 
    const sequence_op &sop = static_cast<const sequence_op&>(op);
    const member_types &mtypes = sop.get_member_types();
    for (member_types::const_iterator it = mtypes.begin();
	 it != mtypes.end(); ++it) {
      if (!check_emptiable_recursive(*it, it->op()))
	return false;
    }
    return true;
  }
  case all_id: {
    if (props.is_element())
      return false; 
    const all_op &aop = static_cast<const all_op&>(op);
    const member_types &mtypes = aop.get_member_types();
    for (member_types::const_iterator it = mtypes.begin();
	 it != mtypes.end(); ++it) {
      if (it->get_occurrence().minOccurs_ > 0)
	return false;
    }
    return true;
  }
  case choice_id: {
    if (props.is_element())
      return false;
    const choice_op &cop = static_cast<const choice_op&>(op);
    const item_types &itypes = cop.get_item_types();
    for (item_types::const_iterator it = itypes.begin();
	 it != itypes.end(); ++it) {
      if (check_emptiable_recursive(*it, it->op()))
	return true;
    }
    return false;
  }
  case array_id: {
    if (props.get_occurrence().minOccurs_ == 0)
      return true;
    const array_op &pop = static_cast<const array_op&>(props.op());
    return check_emptiable_recursive(props, pop.get_item_op());
  }
  case ptrmember_id: {
    if (props.get_occurrence().minOccurs_ == 0)
      return true;
    const ptrmember_op &pop = static_cast<const ptrmember_op&>(props.op());
    return check_emptiable_recursive(props, pop.get_value_op());
  }
  case any_array_id:
    return props.get_occurrence().minOccurs_ == 0;
  case any_id:
    return true;
  case simpletype_id:
  case simplecontent_id:
  case enclose_id:
  case disclose_id:
  case wildcard_attribute_id:
  case wildcard_id:
  case fixed_id: // fixed should be always an element.  akaxiso restriction.!!!!!
    assert(props.is_element());
    return false;
  }
  
  assert(!"Must not reach here.");
  return false;
}

void element_props::check_emptiable() {
  emptiable_ = check_emptiable_recursive(*this, *op_);
}


const int aka2::unbounded = -1;

bool occurrence::in_range(int count) const {
  assert(count >=0);
  if (maxOccurs_ == aka2::unbounded)
    return count >= minOccurs_;

  assert(minOccurs_ <= maxOccurs_);
  return (minOccurs_ <= count) && (count <= maxOccurs_);
}

bool occurrence::is_array() const {
  return minOccurs_ != 1 || maxOccurs_ != 1;
}



bool element_props::is_fixed_string(const pstring &val, 
				    entity_complements &ecomp) const {
  const fixed_op *fop;
  if (get_schematype() == array_id) {
    const array_op &aop = static_cast<const array_op&>(*op_);
    assert(aop.get_item_op().get_schematype() == fixed_id);
    fop = &static_cast<const fixed_op&>(aop.get_item_op());
  }
  else if (get_schematype() == fixed_id) {
    fop = static_cast<const fixed_op*>(op_);
  }
  else {
    assert(!"Must not reach here.");
    fop = 0;
  }

  const simpletype_op &sop = fop->get_value_op();
  void *given = sop.create();

  bool res = true;
  try {
    sop.read_unicode_text(given, val, ecomp);
  }
  catch ( ... /*!!!!!*/) {
    res = false;
  }
  if (res)
    res = sop.equals(given, default_->value());
  sop.destroy(given);
  return res;
} 

void element_props::write_fixed_string(std::ostream &ostm, entity_complements &ecomp) const {
  const fixed_op *fop;
  if (get_schematype() == array_id) {
    const array_op &aop = static_cast<const array_op&>(*op_);
    assert(aop.get_item_op().get_schematype() == fixed_id);
    fop = &static_cast<const fixed_op&>(aop.get_item_op());
  }
  else if (get_schematype() == fixed_id) {
    fop = static_cast<const fixed_op*>(op_);
  }
  else {
    assert(!"Must not reach here.");
    fop = 0;
  }
  const simpletype_op &sop = fop->get_value_op();
  sop.write_text(default_->value(), ostm, ecomp);
}

void element_props::simpletype_initialize(void *e) const {
  assert((get_schematype() == simpletype_id) || (get_schematype() == fixed_id));
  const default_op *defop = default_.get();
  if ((defop != 0) && defop->has_default()) {
    op_->copy(e, default_->value());
  }
  else {
    op_->construct(e);
  }
}

bool element_props::is_default(const void *e) const { 
  assert(is_default_specified());
  return op_->equals(e, default_->value());
}


void element_props::set_ns_list(const std::string &ns_list) {
  ustring collapsed = pivot_to_ucs2(to_pivot(ns_list));
  collapsed = xs::whiteSpace_collapse(collapsed);

  if (collapsed.empty())
    return;

  std::vector<ustring> ns_tokens;
  cuppa::tokenizer<uchar_t> tokenizer(collapsed);
  ustring space;
  space.append(1, uchar_t(0x20));

  while (tokenizer.has_more_token(space)) {
    ns_tokens.push_back(tokenizer.next_token(space));
  }

  std::string token = to_lcp(ucs2_to_pivot(ns_tokens.front()));
  if (token == "##any") {
    ns_list_.allow_any();
    return;
  }
  else if (token.find("##other:") == 0) {
    std::string target_ns = token.substr(8);
    ns_list_.allow_other(target_ns);
    return;
  }

  for (size_t index = 0; index < ns_tokens.size(); ++index) {
    token = to_lcp(ucs2_to_pivot(ns_tokens[index]));
    if ((token == "##any") || (token.find("##other:") == 0)) {
      std::string message = "Wrong namespace specification, " + quote(ns_list) + '.';
      throw error(message, __FILE__, __LINE__);
    }
    if (token == "##local") {
      ns_list_.allow_local();
    }
    else {
      ns_list_.allow_uri(token);
    }
  }
}

bool element_props::is_any_name(const qname &name) const {
  return ns_list_.is_name_allowed(name);
}

void element_props::set_default_op(default_op *defop) {
  assert(defop != 0);
  default_ = shared_ptr<default_op>(defop);
}
