#include "generator.h"
#include <akaxiso2/util/string_funcs.h>
#include <iostream>
#include <algorithm>

using namespace osx;


namespace {

  void write_occurrence(const aka::occurrence &occ, std::ostream &ostm) {
    ostm << occ.minOccurs_ << ", ";
    if (occ.maxOccurs_ == aka::unbounded)
      ostm << "aka::unbounded";
    else
      ostm << occ.maxOccurs_;
  }

  std::string tagname(const element_type &et) {
    if (et.is_group_)
      return aka::quote("&" + et.name_.local());
    return "\"" + et.name_.qualified() + "\"";
  }
  std::string tagname(const attribute_type &et) {
    return aka::quote(et.name_.qualified());
  }

  std::string string_literal(const std::string &value) {
    std::string ret;
    for (std::string::size_type index = 0; index < value.size(); ++index) {
      if (value[index] == '\"') //!!!!!!!!!!!!!!!!!!
	ret += "\\\"";
      else if (value[index] == '\\')
	ret += "\\\\";
      else
	ret += value[index];
    }
    return aka::quote(ret);
  }

}

output_file::output_file(const std::string &outputdir, const std::string &basedir)
  : basedir_(basedir) {
  outputdir_ = outputdir;
  if (!outputdir_.empty()) {
    if (outputdir.at(outputdir.size() -1 ) != '/')
      outputdir_ += '/';
  }
}


void output_file::open(const std::string &filename) {
  std::string fullpath = outputdir_ + filename;
  ostm_.open(fullpath.c_str());
  if (!ostm_.is_open())
    throw aka::error("Failed to open file, " + aka::quote(filename) + ".",
		     __FILE__, __LINE__);
  filename_ = filename;
}

void output_file::close() {
  ostm_.close();
}

void output_file::write_include(const std::string &include_name) {
  if (basedir_.empty())
    ostm_ << "#include " << aka::quote(include_name) << std::endl;
  else
    write_system_include(basedir_ + include_name);
}

void output_file::write_system_include(const std::string &include_name) {
  ostm_ << "#include " << aka::tag(include_name) << std::endl;
}

void output_file::write_header() {

  std::string escaped = escape_filename(filename_);

  ostm_ << "/* -*- c++ -*- */" << std::endl
	<< "#ifndef " << escaped << "__" << std::endl
	<< "#define " << escaped << "__" << std::endl
	<< std::endl;
}


void output_file::write_footer() {

  std::string escaped = escape_filename(filename_);
  ostm_ << "#endif // #ifndef " << escaped << "__" << std::endl;
}


std::string output_file::escape_filename(const std::string &fn) {
  std::string escaped;
  for (std::string::size_type pos = 0; pos < fn.size(); ++pos) {
    std::string::value_type ch = fn[pos];
    if (ch == '.')
      ch = '_';
    else if (ch == '-')
      ch = '_';
    escaped += ch;
  }
  return escaped;
}

void output_file::newline() {
  ostm_ << std::endl;
}


void output_file::write_custom_includes(const includes &incs) {
  for (includes::const_iterator it  =incs.begin();
       it != incs.end(); ++it)
    if (it->system_)
      write_system_include(it->value_);
    else
      write_include(it->value_);
}





void generator_base::set_unit(const registry_unit &unit) {
  regunit_ = &registry_[unit.target_ns_id_];
  target_ns_id_ = unit.target_ns_id_;
}

std::string generator_base::get_prefix(aka::id_type nsid) const {
  const std::string &prefix = aka::get_prefix(nsid);
  std::string escaped_prefix = namer_.escape(prefix);
  return escaped_prefix;
}

std::string generator_base::membername(const aka::qname &name) const {
  if ((target_ns_id_ == name.get_namespace_id()) || (name.prefix().empty()))
    return membername(name.local());
  else {
    std::string qualified = name.prefix() + "_" + name.local();
    return membername(qualified);
  }
}

std::string generator_base::membername(const std::string &name) const {
  assert(!name.empty());
  return namer_.get_member_prefix() 
    + namer_.escape(name) 
    + namer_.get_member_suffix();
}


//
// implementation of generator class.
//
void generator::class_resolved(const user_class &us) {
  if (is_class_resolved(type_ref(&us)))
    return;
  resolved_.push_back(&us);
  show_done(us);
}

bool generator::is_class_resolved(const type_ref &type) const {
  if (type.is_imported())
    return true;
  const_user_classes::const_iterator it = 
    std::find(resolved_.begin(), resolved_.end(), type.us_);
  return it != resolved_.end();
}

void generator::class_generated(const user_class &us) {
  assert(!is_class_generated(us));
  generated_.push_back(&us);
}

bool generator::is_class_generated(const user_class &us) const {
  const_user_classes::const_iterator it = 
    std::find(generated_.begin(), generated_.end(), &us);
  return it != generated_.end();
}



void generator::begin_namespace() {
  assert(target_ns_id_ != aka::empty_token);

  elm_.ostm_ << std::endl
	     << "namespace " << get_prefix(target_ns_id_) << " {" << std::endl
	     << std::endl;
  xiso_.ostm_ << std::endl
	      << "namespace " << get_prefix(target_ns_id_) << " {" << std::endl
	      << std::endl;
  ximpl_.ostm_ << std::endl
	       << "namespace " << get_prefix(target_ns_id_) << " {" << std::endl
	       << std::endl;
}    

void generator::end_namespace() {
  assert(target_ns_id_ != aka::empty_token);
  elm_.ostm_ << std::endl
	     << "}" << std::endl
	     << std::endl;
  xiso_.ostm_ << std::endl
	      << "}" << std::endl
	      << std::endl;
  ximpl_.ostm_ << std::endl
	       << "}" << std::endl
	       << std::endl;
}








void generator::begin_model_method(const user_class &us) const {
  ximpl_.ostm_ << "  void " << leaftype(us) << "::model() {" << std::endl;
}

void generator::end_model_method() const {
  ximpl_.ostm_ << "  }" << std::endl
	       << std::endl;
}


std::string generator_base::classname(const type_ref &type) const {
  return namer_.classname(type);
}

std::string generator_base::classname(const user_class &us) const {
  return namer_.classname(type_ref(&us));
}

std::string generator_base::classtype(const user_class &us) const {
  return namer_.classtype(type_ref(&us));
}

std::string generator_base::leafname(const type_ref &type) const {
  return namer_.leafname(type);
}

std::string generator_base::leafname(const user_class &us) const {
  return namer_.leafname(type_ref(&us));
}

std::string generator_base::leaftype(const user_class &us) const {
  return namer_.leaftype(type_ref(&us));
}


std::string generator::get_ns_list(const xs::namespaceList &ns_list) const {
  std::string akx_ns_list;
  for (size_t index = 0; index < ns_list.size(); ++index) {
    const std::string &ns_token = ns_list[index];
    std::string akx_ns;
    if (ns_token == "##any") {
      akx_ns = "##any";
    }
    else if (ns_token == "##other") {
      akx_ns = "##other:" + aka::get_namespace_uri(target_ns_id_);
      assert(!akx_ns.empty()); // !!! OK?
    }
    else if (ns_token == "##local") {
      akx_ns = "##local";
    }
    else if (ns_token == "##targetNamespace") {
      akx_ns = aka::get_namespace_uri(target_ns_id_);
      if (akx_ns.empty())
	akx_ns = "##local"; // !!! OK?
    }
    else {
      akx_ns = ns_token;
    }
    assert(!akx_ns.empty());
    if (akx_ns_list.empty()) {
      akx_ns_list = akx_ns;
    }
    else {
      akx_ns_list += " " + akx_ns;
    }
  }
  return aka::quote(akx_ns_list);
}




bool generator::is_leaf_declared(const type_ref &type) const {
  if (type.is_imported())
    return true;
  const user_class *us = type.us_;
  return qname_set::const_iterator(declared_leafs_.find(us->get_classname()))
    != declared_leafs_.end();
}

void generator::leaf_declared(const user_class &us) {
  if (is_leaf_declared(type_ref(&us)))
    return;
  declared_leafs_.insert(us.get_classname());
}


void generator::prepare() {
  declared_leafs_.clear();
  resolved_.clear();
  generated_.clear();
}


void generator::resolve(const registry_unit &unit, bool generate) {

  set_unit(unit);

  try {

    const_user_classes ns_simpleTypes;
    const_user_classes ns_user_classes;

    user_classes::const_iterator cit;
    for (cit = regunit_->simpleTypes_.begin(); 
	 cit != regunit_->simpleTypes_.end(); ++cit) {
      const user_class *def = cit->second;
      assert(def->get_name().get_namespace_id() == target_ns_id_);
      assert((def->id_ == simpletype_id) || (def->id_ == array_id));
      ns_simpleTypes.push_back(def);
    }

    for (cit = regunit_->complexTypes_.begin(); 
	 cit != regunit_->complexTypes_.end(); ++cit) {
      const user_class *def = cit->second;
      assert ((def->id_ != simpletype_id) || (def->id_ == array_id));
      ns_user_classes.push_back(def);
    }
    // group is included in complexTypes.
    for (cit = regunit_->groups_.begin(); 
	 cit != regunit_->groups_.end(); ++cit) {
      const user_class *def = cit->second;
      assert ((def->id_ != simpletype_id) || (def->id_ == array_id));
      ns_user_classes.push_back(def);
    }

    for (element_defs::const_iterator eit = regunit_->elements_.begin();
	 eit != regunit_->elements_.end(); ++eit) {
      const element_def &edef = eit->second;
      assert(!edef.type_.is_reference());
      if (edef.type_.is_imported())
	continue;
      if (regunit_->local_simpleTypes_.get(edef.type_.get_name()) != 0) {
	ns_simpleTypes.push_back(edef.type_.us_);
      }
      else if (regunit_->local_complexTypes_.get(edef.type_.get_name()) != 0) {
	ns_user_classes.push_back(edef.type_.us_);
      }
    }


    elm_.newline();
    xiso_.newline();
    ximpl_.newline();

    if (generate && (target_ns_id_ != aka::empty_token)) {
      begin_namespace();
    }    

    resolve_one_unit(ns_simpleTypes, ns_user_classes, generate);

    if (generate && (target_ns_id_ != aka::empty_token)) {
      end_namespace();
    }
  }
  catch ( ... ) {
    elm_.ostm_.close();
    xiso_.ostm_.close();
    ximpl_.ostm_.close();
    throw;
  }
}



void generator::write_leaf_decl(const user_class &def, const std::string &leaftmp) const {
  xiso_.ostm_ << "  struct " << leaftype(def) 
	      << " : aka::" << leaftmp << "< " << classname(def)
	      << ", " << leaftype(def) << "> {" << std::endl
	      << "    void model();" << std::endl
	      << "  };" << std::endl
	      << std::endl;
}


void generator::write_member_decls(const element_types &etypes) const {

  if (!etypes.empty())
    elm_.ostm_ << "    /** members */" << std::endl;
  for (element_types::const_iterator eit = etypes.begin();
       eit != etypes.end(); ++eit) {
    if (eit->is_ptrmember_)
      elm_.ostm_ << "    aka::deep_ptr< " << classname(eit->type_) << "> " << membername(eit->member_name_) 
		 << ";" << std::endl;
    else if (!eit->is_fixed_)
      elm_.ostm_ << "    " << classname(eit->type_) << " " << membername(eit->member_name_) << ";" << std::endl;
  }
}

void generator::write_attribute_decls(const attribute_defs &attrs) const {

  if (attrs.has_attributes())
    elm_.ostm_ << "    /** attributes */" << std::endl;
  for (attribute_types::const_iterator ait = attrs.attributes_.begin();
       ait != attrs.attributes_.end(); ++ait) {
    if (ait->is_fixed_) {
      // Nothing to do.
    }
    else if (ait->is_ptrmember_) {
      elm_.ostm_ << "    aka::deep_ptr< " << classname(ait->type_) << "> " << membername(ait->name_) << ";" << std::endl;
    }
    else {
      elm_.ostm_ << "    " << classname(ait->type_) << " " << membername(ait->name_) << ";" << std::endl;
    }
  }
  if (attrs.has_anyAttribute_) {
    elm_.ostm_ << "    aka::wc_attributes " << membername("attributes") << ";" << std::endl;
  }
}

void generator::write_member_leaves(const user_class &us) const {

  const element_types &etypes = us.etypes_;

  if (etypes.empty())
    return;

  ximpl_.ostm_ << "    /** element member definitions */" << std::endl;

  for (element_types::const_iterator eit = etypes.begin();
       eit != etypes.end(); ++eit) {
    if (eit->is_fixed_) {
      ximpl_.ostm_ << "    fixed_member(" << tagname(*eit) << ", " 
		   << string_literal(eit->default_) << ", " << leafname(eit->type_) << ");";
      continue;
    }

    if (eit->type_.get_name() == aka::qname("aka2:any")) {
      if (eit->is_ptrmember_)
	ximpl_.ostm_ << "    any_ptrmember(" << tagname(*eit) << ", &"
		     << classname(us) << "::" << membername(eit->member_name_) << ", "
		     << get_ns_list(eit->namespace_list_) << ");" << std::endl;
      else
	ximpl_.ostm_ << "    any(" << tagname(*eit) << ", &"
		     << classname(us) << "::" << membername(eit->member_name_) << ", "
		     << get_ns_list(eit->namespace_list_) << ");" << std::endl;
      continue;
    }
    if (eit->type_.get_name() == aka::qname("aka2:any_array")) {
      ximpl_.ostm_ << "    any(" << tagname(*eit) << ", &"
		   << classname(us) << "::" << membername(eit->member_name_) << ", ";
      write_occurrence(eit->get_occurrence(), ximpl_.ostm_);
      ximpl_.ostm_ << ", " << get_ns_list(eit->namespace_list_)
		   << ");" << std::endl;
      continue;
    }

    if (eit->is_ptrmember_) {
      ximpl_.ostm_ << "    ptrmember(" << tagname(*eit) << ", &"
		   << classname(us) << "::" << membername(eit->member_name_) << ", "
		   << leafname(eit->type_)
		   << "());" << std::endl;
      continue;
    }

    ximpl_.ostm_ << "    member(" << tagname(*eit) << ", &"
		 << classname(us) << "::" << membername(eit->member_name_) << ", "
		 << leafname(eit->type_) << "()";
    if (eit->occ_required_) {
      ximpl_.ostm_ << ", ";
      write_occurrence(eit->get_occurrence(), ximpl_.ostm_);
    }
    ximpl_.ostm_ << ");" << std::endl;
  }
}


void generator::write_item_leaves(const user_class &us) const {

  const element_types &etypes = us.etypes_;
  if (etypes.empty())
    return;

  ximpl_.ostm_ << "    /** element member definitions */" << std::endl;

  for (element_types::const_iterator eit = etypes.begin();
       eit != etypes.end(); ++eit) {
    if (eit->is_fixed_) {
      ximpl_.ostm_ << "    fixed_item(" << tagname(*eit) << ", " 
		   << string_literal(eit->default_) << ", " << leafname(eit->type_) << ");";
      continue;
    }

    if (registry_.is_aka_any(eit->type_.get_name())) {
      ximpl_.ostm_ << "    any(" << tagname(*eit);
      if (eit->occ_required_) {
	ximpl_.ostm_ << ", ";
	write_occurrence(eit->get_occurrence(), ximpl_.ostm_);
      }
      ximpl_.ostm_ << ", " << get_ns_list(eit->namespace_list_)
		   << ");" << std::endl;
      continue;
    }

    ximpl_.ostm_ << "    item(" << tagname(*eit) << ", "
		 << leafname(eit->type_) << "()";
    if (eit->occ_required_) {
      ximpl_.ostm_ << ", ";
      write_occurrence(eit->get_occurrence(), ximpl_.ostm_);
    }
    ximpl_.ostm_ << ");" << std::endl;
  }
}

void generator::write_attribute_leaves(const user_class &us) const {

  const attribute_defs &attrs = us.adefs_;
  if (!attrs.has_attributes())
    return;
  
  ximpl_.ostm_ << "    /** attribute member definition. */" << std::endl;

  for (attribute_types::const_iterator ait = attrs.attributes_.begin();
       ait != attrs.attributes_.end(); ++ait) {
    if (ait->is_fixed_) {
      ximpl_.ostm_ << "    fixed_attribute< " << leafname(ait->type_) << " >(" 
		   << tagname(*ait) << ", " 
		   << string_literal(ait->default_) << ", "
		   << leafname(ait->type_) << "())";
      if (ait->required_)
	ximpl_.ostm_ << ".required(true)";
      ximpl_.ostm_ << ";";
      continue;
    }

    if (ait->is_ptrmember_) {
      ximpl_.ostm_ << "    ptrattribute(" << tagname(*ait) << ", &" 
		   << classname(us) << "::" << membername(ait->name_) << ", "
		   << leafname(ait->type_) << "())";
    }
    else {
      ximpl_.ostm_ << "    attribute(" << tagname(*ait) << ", &" 
		   << classname(us) << "::" << membername(ait->name_) << ", "
		   << leafname(ait->type_) << "())";
    }
    if (!ait->default_.empty()) {
      ximpl_.ostm_ << ".set_default(" << string_literal(ait->default_) << ")";
      assert(!ait->required_);
    }
    if (ait->required_)
      ximpl_.ostm_ << ".required(true)";
    
    ximpl_.ostm_ << ";" << std::endl;
  }
  if (attrs.has_anyAttribute_)
    ximpl_.ostm_ << "    any_attribute(&" << classname(us) << "::attributes_, "
		 << get_ns_list(attrs.namespace_list_) << ");" << std::endl;
}


void generator::write_xmltype(const user_class &def) const {
  ximpl_.ostm_ << "    /** XML-Schema type name. */" << std::endl
               << "    xmltype(" << aka::quote(def.get_name()) << ");" << std::endl;
}


void generator::write_occurrence_declaration(const user_class &def) const {
  ximpl_.ostm_ << "    /** occurrence used when choice is a root element. */" 
	       << std::endl
	       << "    occurrence(";
  write_occurrence(def.get_occurrence(), ximpl_.ostm_);
  ximpl_.ostm_ << ");" << std::endl;
}


void generator::generate_sequence(const user_class &def) {
  write_ptrmember_forwarded_decl(def.etypes_);
  elm_.ostm_ << "  struct " << classtype(def) << " {" << std::endl;
  write_member_decls(def.etypes_);
  write_attribute_decls(def.adefs_);
  elm_.ostm_ << "  };" << std::endl
	     << std::endl;

  write_leaf_decl(def, "sequence");
  leaf_declared(def);

  begin_model_method(def);
  write_xmltype(def);
  write_member_leaves(def);
  write_attribute_leaves(def);
  end_model_method();
}


void generator::generate_all(const user_class &def) {
  write_ptrmember_forwarded_decl(def.etypes_);
  elm_.ostm_ << "  struct " << classtype(def) << " {" << std::endl;
  write_member_decls(def.etypes_);
  write_attribute_decls(def.adefs_);
  elm_.ostm_ << "  };" << std::endl
	     << std::endl;

  write_leaf_decl(def, "all");
  leaf_declared(def);

  begin_model_method(def);
  write_xmltype(def);
  write_member_leaves(def);
  write_attribute_leaves(def);
  end_model_method();
}


void generator::generate_choice(const user_class &def) {
  // element.h
  std::ostringstream forwarded;

  elm_.ostm_ << "  typedef " << namer_.choice_container_type() << "<aka::item> " 
	     << classtype(def) << ';' << std::endl
	     << std::endl;

  write_leaf_decl(def, "sequential_choice");
  leaf_declared(def);

  begin_model_method(def);
  write_xmltype(def);
  write_occurrence_declaration(def);
  write_item_leaves(def);
  end_model_method();
}


void generator::generate_simplecontent(const user_class &def) {

  elm_.ostm_ << "  struct " << classtype(def) << " {" << std::endl;
  elm_.ostm_ << "    " << classname(def.get_value_type()) << " " 
	     << membername(namer_.simplecontent_value_name()) << ";" << std::endl; 
  write_attribute_decls(def.adefs_);
  elm_.ostm_ << "  };" << std::endl
	     << std::endl;

  write_leaf_decl(def, "simplecontent");
  leaf_declared(def);

  begin_model_method(def);
  write_xmltype(def);
  ximpl_.ostm_ << "    value(&" << classname(def) << "::" 
	       << membername(namer_.simplecontent_value_name()) << ");" << std::endl;
  // write default !!!!!

  write_attribute_leaves(def);
  end_model_method();

}



void generator::generate_array(const user_class &def) {
  assert(def.id_ == array_id);
  if (!is_class_resolved(type_ref(def.get_value_type()))) {
    assert(!def.get_value_type().is_imported());
    const user_class *valus = def.get_value_type().us_;
    elm_.ostm_ << "  /** forwarded declartion */" << std::endl
	       << "  struct " << classtype(*valus) << ";" << std::endl;
  }
  elm_.ostm_ << "  typedef " 
	     << namer_.array_container_type() << "< " << classname(def.get_value_type()) << "> " 
	     << classtype(def) << ";" << std::endl
	     << std::endl;
  
  // Forwareded declaration of leaf type of array value type.
  if (!is_leaf_declared(def.get_value_type())) {
    assert(!def.get_value_type().is_imported());
    const user_class *valus = def.get_value_type().us_;
    xiso_.ostm_ << "  struct " << leaftype(*valus) << ";" << std::endl;
  }

  xiso_.ostm_ << "  typedef aka::sequential_array< " << classname(def)
	      << ", " << leafname(def.get_value_type()) << " > " 
	      << leaftype(def) << ";" << std::endl
	      << std::endl;

  leaf_declared(def);
}

void generator::generate_simpletype(const user_class &def) {
  elm_.ostm_ << "  typedef " << classname(def.get_value_type()) << " " 
	     << classtype(def) << ";" << std::endl
	     << std::endl;

  xiso_.ostm_ << "  typedef " << leafname(def.get_value_type()) << " "
	      << leaftype(def) << ";" << std::endl; 
  leaf_declared(def);
}


void generator::write_ptrmember_forwarded_decl(const element_types &etypes) const {
  for (element_types::const_iterator it = etypes.begin(); it != etypes.end(); ++it) {
    if (!it->is_ptrmember_)
      continue;
    if (!is_class_resolved(it->type_)) {
      assert(!it->type_.is_imported());
      const user_class *valus = it->type_.us_;
      elm_.ostm_ << "  /** forwareded declaration for ptrmember value class. */" << std::endl
		 << "  struct " << classtype(*valus) << ";" << std::endl;
    }
  }
}



void generator::resolve_one_unit(const_user_classes &simpleTypes, 
				 const_user_classes &user_classes, 
				 bool generate) {

  if (!simpleTypes.empty() && generate)
    elm_.ostm_ << "  /** simpleType definitions */" << std::endl
	       << std::endl;
  
  while (!simpleTypes.empty()) {
    bool resolved = false;
    const_user_classes::iterator it;
    for (it = simpleTypes.begin(); it != simpleTypes.end(); ++it) {
      if (resolve_simpleType(**it, simpleTypes, generate)) {
	resolved = true;
	break;
      }
    }

    if (resolved)
      continue;

    std::ostringstream ostm;
    ostm << "Unresolved simpleTypes: ";
    for (it = simpleTypes.begin(); it != simpleTypes.end(); ++it)
      ostm << (*it)->get_name().qualified() << " ";
    throw aka::error(ostm.str(), __FILE__, __LINE__);
  }
  
  
  if (!user_classes.empty() && generate)
    elm_.ostm_ << "  /** complexType definitions */" << std::endl
	       << std::endl;
  
  while (!user_classes.empty()) {
    bool resolved = false;
    const_user_classes::iterator it;
    for (it = user_classes.begin(); it != user_classes.end(); ++it) {
      if (is_class_resolved(type_ref(*it))) {
	user_classes.erase(it); // already resolved.
	resolved = true;
	break;
      }
    }      
      // Force hard resolve only.
    for (it = user_classes.begin(); it != user_classes.end(); ++it) {
      if (resolve_user_class(**it, user_classes, true, generate)) {
// 	user_classes.erase(it);
	resolved = true;
	break;
      }
    }
    
    if (!resolved) {
      for (it = user_classes.begin(); it != user_classes.end(); ++it) {
	if (resolve_user_class(**it, user_classes, false, generate)) {
// 	  user_classes.erase(it);
	  resolved = true;
	  break;
	}
      }
    }

    if (!resolved) {
      std::ostringstream ostm;
      ostm << "Could not resolve dependencies.";
      for (it = user_classes.begin(); it != user_classes.end(); ++it)
	ostm << (*it)->get_classname().qualified() << " ";
      throw aka::error(ostm.str(), __FILE__, __LINE__);
    }
  }
}

bool generator::resolve_simpleType(const user_class &to_resolve, 
				   const_user_classes &simpleTypes,
				   bool generate_source) {

  const aka::qname &name_to_resolve = to_resolve.get_name();
  if ((regunit_->simpleTypes_.get(name_to_resolve) == 0)
      && (regunit_->local_simpleTypes_.get(name_to_resolve) == 0)) {
    raise_name_error("resolver", to_resolve.get_name(), "could not find type.", __FILE__, __LINE__);
  }

  const user_class *resolved = 0;

  if ((regunit_->simpleTypes_.get(name_to_resolve) != 0) ||
      (regunit_->local_simpleTypes_.get(name_to_resolve) != 0)) {
    if (is_class_resolved(to_resolve.get_value_type()))
      resolved = &to_resolve;
    else {
      const user_class *value_class = to_resolve.get_value_type().us_;
      return resolve_simpleType(*value_class, simpleTypes, generate_source);
    }
  }
  else {
    return false;
  }

  // Try to resolve value_type.
  if (resolved == 0)
    return false;
    
  if (generate_source) {
    if (resolved->id_ == simpletype_id)
      generate_simpletype(*resolved);
    else if (resolved->id_ == array_id)
      generate_array(*resolved); 
    else {
      assert(!"Must not reach here.");
    }
  }
  class_resolved(*resolved);

  const_user_classes::iterator it = std::find(simpleTypes.begin(), simpleTypes.end(), resolved);
  if (it != simpleTypes.end())
    simpleTypes.erase(it);
  return true;
}


bool generator::resolve_user_class(const user_class &to_resolve, 
				   const_user_classes &user_classes, 
				   bool force_hard_resolve,
				   bool generate_source) {

  const_user_classes to_generate;
  const_user_classes forwarded;

  assert(regunit_->type_exists(type_ref(&to_resolve)));

  if (force_hard_resolve) {
    hard_resolve(to_resolve, to_generate, forwarded);
  }
  else {
    if (!hard_resolve(to_resolve, to_generate, forwarded))
      soft_resolve(to_resolve, to_generate, forwarded);
  }

  for (const_user_classes::iterator it = to_generate.begin(); it != to_generate.end(); ++it) {
    const user_class *def = *it;
    if (is_class_generated(*def))
      continue;
    class_generated(*def);

    if (generate_source && (def->get_name().get_namespace_id() == target_ns_id_)) {
      switch (def->id_) {
      case sequence_id:
	generate_sequence(*def);
	break;
      case choice_id:
	generate_choice(*def);
	break;
      case all_id:
	generate_all(*def);
	break;
      case array_id:
	generate_array(*def);
	break;
      case simplecontent_id:
	generate_simplecontent(*def);
	break;
      case simpletype_id:
	generate_simpletype(*def);
	break;
      default:
	assert(!"Must not reach here.");
	break;
      }
    }
  }
  
  bool new_class_added = false;

  //generation of forwarded classes. 
  for (const_user_classes::iterator fit = forwarded.begin(); fit != forwarded.end(); ++fit) {
    if (is_class_resolved(type_ref(*fit)))
      continue;
    if (std::find(user_classes.begin(), user_classes.end(), *fit) == user_classes.end()) {
      user_classes.insert(user_classes.begin(), *fit);
      new_class_added = true;
    }
  }

  return new_class_added || !to_generate.empty();
}


bool generator::soft_resolve(const user_class &to_resolve, 
			     const_user_classes &to_generate, 
			     const_user_classes &forwarded) {
  
  if (is_class_resolved(type_ref(&to_resolve)))
    return true;

  assert(regunit_->type_exists(type_ref(&to_resolve))); // must in the current namespace.

  // aka::item holds pointer to the member.
  // imcomplete type allowed.
  if (to_resolve.id_ == choice_id) {
    to_generate.push_back(&to_resolve);
    for (element_types::const_iterator it = to_resolve.etypes_.begin(); 
	 it != to_resolve.etypes_.end(); ++it)
      if (!it->type_.is_imported())
	forwarded.push_back(it->type_.us_);
    class_resolved(to_resolve);
    return true;
  }

  if (to_resolve.id_ == array_id) {
    if (!use_soft_array_) {
      const_user_classes processing;
      if (hard_resolve(to_resolve, to_generate, processing)) {
	return true;
      }
      return false;
    }
    else {
      // Allow imcomplete type of array element.
      to_generate.push_back(&to_resolve);
      if (!to_resolve.get_value_type().is_imported())
	forwarded.push_back(to_resolve.get_value_type().us_); 
      class_resolved(to_resolve);
      return true;
    }
  }

  for (element_types::const_iterator eit = to_resolve.etypes_.begin();
       eit != to_resolve.etypes_.end(); ++eit) {
    const element_type& childtype = *eit;
    if (childtype.type_.is_imported())
      continue;
    else if (&to_resolve == childtype.type_.us_) { // cyclic reference check.
      if (childtype.occ_required_ || childtype.is_ptrmember_)
	continue;
    }
    else if (childtype.is_ptrmember_) {
      if (!is_class_resolved(childtype.type_))
	forwarded.push_back(childtype.type_.us_);
      continue;
    }
    else {
      /**
       * Check cyclic dependency of soft-resolving type. 2005/10/19 smorino
       * xmlschema2002-01-16/msxsdtest/Group/groupB015.xsd.
       */
      if (soft_resolving_.find(to_resolve.get_name()) != soft_resolving_.end()) {
	raise_name_error("type", to_resolve.get_name(), " has cyclic dependency.", 
			 __FILE__, __LINE__);
      }
      soft_resolving_.insert(to_resolve.get_name());
      bool res = soft_resolve(*childtype.type_.us_, to_generate, forwarded);
      soft_resolving_.erase(to_resolve.get_name());
      if (res)
	continue;
    }
    return false;
  }
  
  for (attribute_types::const_iterator ait = to_resolve.adefs_.attributes_.begin();
       ait != to_resolve.adefs_.attributes_.end(); ++ait) {
    if (is_class_resolved(ait->type_))
      continue;
    const_user_classes processing;
    if (hard_resolve(*ait->type_.us_, to_generate, processing)) 
      continue;
    return false;
  }
  
  to_generate.push_back(&to_resolve);
  class_resolved(to_resolve);
  
  return true;
}


bool generator::hard_resolve(const user_class &to_resolve, 
			     const_user_classes &to_generate, 
			     const_user_classes &processing) {
  
  if (is_class_resolved(type_ref(&to_resolve)))
    return true;

  /**
   * check to_resolve is defined. 2005/10/19 smorino
   * xmlschema2002-01-16/msxsdtest/identityConstraint/idG018.xsd
   */
  if (!regunit_->type_exists(type_ref(&to_resolve))) {
    raise_name_error("type", to_resolve.get_name(), " not defined.",
		     __FILE__, __LINE__);
  }

  if (std::find(processing.begin(), processing.end(), &to_resolve) != processing.end())
    return false;
  
  processing.push_back(&to_resolve);
 
  if (to_resolve.id_ == array_id) {
    if (!is_class_resolved(to_resolve.get_value_type()))
      if (!hard_resolve(*to_resolve.get_value_type().us_, to_generate, processing))
	return false;
  }
  else {
    for (element_types::const_iterator eit = to_resolve.etypes_.begin();
	 eit != to_resolve.etypes_.end(); ++eit) {
      if (!is_class_resolved(eit->type_))
	if (!hard_resolve(*eit->type_.us_, to_generate, processing))
	  return false;
    }
  }

  for (attribute_types::const_iterator ait = to_resolve.adefs_.attributes_.begin();
       ait != to_resolve.adefs_.attributes_.end(); ++ait) {
    if (!is_class_resolved(ait->type_))
      if (!hard_resolve(*ait->type_.us_, to_generate, processing))
	return false;
  }

  class_resolved(to_resolve);
  to_generate.push_back(&to_resolve);
  return true;
}




void generator::open(const std::string &basename) {
  elm_.open(basename + ".h");
  xiso_.open(basename + "_xiso.h");
  ximpl_.open(basename + "_xiso.cpp");

  elm_.write_header();
  xiso_.write_header();

//   elm_.write_system_include("akaxiso2/builtin/builtin.h");
//   elm_.write_system_include("akaxiso2/builtin/schema_builtin.h");
  elm_.newline();

  xiso_.write_include(elm_.get_filename());
//   xiso_.write_system_include("akaxiso2/content_model.h");
  xiso_.newline();

  ximpl_.write_include(xiso_.get_filename());
//   ximpl_.write_system_include("akaxiso2/akaxiso2.h");
  ximpl_.newline();

}


void generator::close() {
  elm_.write_footer();
  xiso_.write_footer();

  elm_.close();
  xiso_.close();
  ximpl_.close();
}


void generator::write_include(const std::string &basename) const {
  elm_.write_include(basename + ".h");
  xiso_.write_include(basename + "_xiso.h");
}

void generator::write_custom_includes(const registry_unit &unit) const {
  elm_.write_custom_includes(unit.includes_);
  xiso_.write_custom_includes(unit.xiso_includes_);
  ximpl_.write_custom_includes(unit.ximpl_includes_);
}

void generator::show_done(const user_class &def) {
  if (verbose_)
    ostm_ << def.get_name().qualified() << " done." << std::endl;
}

void generator::show_not_resolved(const user_class &def) {
  if (verbose_)
    ostm_ << def.get_name().qualified() << " not resolved yet." << std::endl;
}


namespace {
  struct doc_tag {
    doc_tag(const aka::qname &name,
	    const type_ref &type)
      : name_(name), 
	type_(type){}
    aka::qname name_;
    type_ref type_;
  };
}

void func_generator::generate_serialize_methods() {

  typedef std::vector<doc_tag> doc_tags;
  typedef std::map<std::string, doc_tags> element_map;
  element_map emap;

  units uts = registry_.get_units();

  for (units::iterator uit = uts.begin(); uit != uts.end(); ++uit) {
    
    set_unit(**uit);

    for (element_defs::iterator eit = regunit_->elements_.begin();
	 eit != regunit_->elements_.end(); ++eit) {
      std::string realname = namer_.resolve_typedef(eit->second.type_);
      assert(!realname.empty());
      doc_tag dt(eit->first, eit->second.type_);
      emap[realname].push_back(dt);
    }
  }    
  
  for (element_map::iterator eit = emap.begin(); eit != emap.end(); ++eit) {
    const doc_tags &tags = eit->second;

    if (tags.size() == 1) {
      const doc_tag &dtag = tags.front();
      header_.ostm_ << "/** serialization of " << aka::quote(dtag.name_) << ". */" << std::endl
		    << "void serialize(const " << eit->first 
		    << " &root, std::ostream &ostm);" << std::endl;
      
      impl_.ostm_ << "void serialize(const " << eit->first 
		  << "& root, std::ostream &ostm) {" << std::endl
		  << "  aka::xml_serializer ser;" << std::endl
      << "  ser.serialize(root, " << aka::quote(dtag.name_) << ", ostm);" << std::endl
		  << "}" << std::endl
		  << std::endl;
    }
    else {

      header_.ostm_ << "/** serialization of ";
      
      doc_tags::const_iterator tit = tags.begin();
      std::string doc_cpptype = classname(tit->type_);
      header_.ostm_ << aka::quote(tit->name_) << "(" << classname(tit->type_) << ")";
      for (++tit; tit != tags.end(); ++tit) {
        header_.ostm_ << ", " << aka::quote(tit->name_) << "(" << classname(tit->type_) << ")";
      }

      header_.ostm_ << ". */" << std::endl
		    << "void serialize(const " << doc_cpptype
		    << "& root, const std::string &tagname, std::ostream &ostm);" << std::endl;
      
      impl_.ostm_ << "void serialize(const " << doc_cpptype
		  << "& root, const std::string &tagname, std::ostream &ostm) {" << std::endl
		  << "  aka::xml_serializer ser;" << std::endl
		  << "  ser.serialize(root, tagname, ostm);" << std::endl
		  << "}" << std::endl
		  << std::endl;
    }
  }
}


void func_generator::begin_instantiate_xiso() const {

  if (instantiate_xiso_name_.empty()) {
    header_.ostm_ << "void instantiate_xiso();" << std::endl
		  << std::endl;
    impl_.ostm_ << "void instantiate_xiso() {" << std::endl;
  }
  else {
    header_.ostm_ << "void " << instantiate_xiso_name_ << "();" << std::endl
		  << std::endl;
    impl_.ostm_ << "void " << instantiate_xiso_name_ << "() {" << std::endl;
  }
}


void func_generator::generate_instantiate_xiso() {

  units uts = registry_.get_units();
  for (units::iterator uit = uts.begin(); uit != uts.end(); ++uit) {

    set_unit(**uit);
    if ((target_ns_id_ != aka::empty_token) && (target_ns_id_ != aka::xml_ns_token))
      impl_.ostm_ << "  aka::xmlns(" << aka::quote(aka::get_prefix(target_ns_id_))
		  << ", " << aka::quote(aka::get_namespace_uri(target_ns_id_)) << ");" << std::endl;
    
    for (element_defs::iterator nit = regunit_->elements_.begin();
	 nit != regunit_->elements_.end(); ++nit) {
      const aka::qname &name = nit->first;
      if (name.get_namespace_id() == target_ns_id_) {
	if (nit->second.type_.get_classname() == aka::qname("aka2:wildcard")) {
    impl_.ostm_ << "  aka::doctype(" << aka::quote(nit->first) << ");" << std::endl;
	}
	else if (nit->second.is_fixed_) {
    impl_.ostm_ << "  aka::doctype(" << aka::quote(nit->first) << ", " 
		      << leafname(nit->second.type_) << "(), "
		      << string_literal(nit->second.default_)
		      << ");" << std::endl;
	}
	else {
    impl_.ostm_ << "  aka::doctype(" << aka::quote(nit->first) << ", "
		      << leafname(nit->second.type_) << "());" << std::endl;
	}
      }
    }
  }
}

void func_generator::write_includes() {
  units uts = registry_.get_units();
  for (units::const_iterator uit = uts.begin(); uit != uts.end(); ++uit) {
    header_.write_include((*uit)->basename_ + ".h");
    impl_.write_include((*uit)->basename_ + "_xiso.h");
  }
}


void func_generator::end_instantiate_xiso() const {
  impl_.ostm_ << "}" << std::endl
	      << std::endl;
}

void func_generator::generate() {
  begin_instantiate_xiso();
  generate_instantiate_xiso();
  end_instantiate_xiso();
  generate_serialize_methods();
}

void func_generator::open(const std::string &basename) {

  header_.open(basename + ".h");
  header_.write_header();
  header_.newline();

  impl_.open(basename + ".cpp");
  impl_.write_include(header_.get_filename());
  impl_.write_system_include("akaxiso2/akaxiso2.h");
  impl_.newline();
}


void func_generator::close() {
  header_.newline();
  header_.write_footer();
  header_.close();

  impl_.newline();
  impl_.close();
}
