#include "rep_validator.h"
#include "exception.h"

using namespace osx;

namespace {
  template<class A>
  void _attribute_valid(const A& a, const aka::qname &name) {
    /* 3.2.3 Constraints on XML Representations of Attribute Declarations. */ 
    
    /* 3.2.3.1 */
    if (!a.default_.empty() && !a.fixed_.empty()) {
      raise_name_error("xs:attribute", name, "both @fixed and @default are specified.",
		       __FILE__, __LINE__);
    }
    if (!a.type_.empty() && a.simpleType_.present()) {
      raise_name_error("xs:attribute", name, 
		       "both @type and local xs:simpleType are specified.", 
		       __FILE__, __LINE__);
    }
    
    /* 3.2.6 */
    if (a.name_ == "xmlns") {
      raise_name_error("xs:attribute@name", name, 
		       "should not be match 'xmlns'.", 
		       __FILE__, __LINE__);
      
    }
    /* 3.2.6-7 */
    //   if (attr.get_namespace_id() == xsi_id) {
    //     /** They are builtin, there for must not be declared. */
    //     const std::string &localname = attr.local();
    //     if ((localname == "type") || 
    // 	(localname == "nil") ||
    // 	(localname == "schemaLocation") ||
    // 	(localname == "noNamespaceSchemaLocation")) {
    //     }
    //   }
  }
}


rep_validator::rep_validator(const xs::topLevelAttribute &tla, aka::id_type ns_id) {
  _attribute_valid(tla, aka::qname(ns_id, tla.name_));
}

rep_validator::rep_validator(const xs::attribute &attr, aka::id_type ns_id, 
			     const aka::qname &hint) {
  
  /* 3.2.3.3.1 */
  /* processing local xs:attribute, therefore its parent is not xs:schema. */
  if (!attr.name_.empty() && !attr.ref_.empty()) {
    raise_name_error("xs:attribute", aka::qname(attr.name_), "both @ref and @name are specified.",
		     __FILE__, __LINE__);
  }


  if (attr.ref_.empty() && attr.name_.empty()) {
    raise_name_error("xs:attribute under " + hint.qualified(), aka::qname ("Unknown"),
		     "@ref or @name must be specified.",
		     __FILE__, __LINE__);
  }


  /* 3.2.3.3.2 */
  if (!attr.ref_.empty()) {
    if (attr.form_.present() || attr.simpleType_.present() || !attr.type_.empty()) {
      raise_name_error("xs:attribute@ref", attr.ref_, 
		       "@ref is specified, but @form, xs:simpleType or @type is speicifed.",
		       __FILE__, __LINE__);
    }
  }
  else {
    aka::qname name(ns_id, attr.name_);
    /* 3.2.3.2 */
    if (!attr.default_.empty() && !xs::is_use_optional(attr.use_)) {
      raise_name_error("xs:attribute", name, "@default specified but @use is not optional.",
		       __FILE__, __LINE__);
    }
    _attribute_valid(attr, name);
  }
}


namespace {

  template <class E>
  void _element(const E &elm, const aka::qname &name) {
    /* 3.3.3.1 */
    if (!elm.default_.empty() && !elm.fixed_.empty()) {
      raise_name_error("xs:element", name, "@default and @fixed are specified.",
		       __FILE__, __LINE__);
    }

    /* 3.3.3.3 */
    if (elm.c0_.size() == 1) {
      if (!elm.type_.empty()) {
	raise_name_error("xs:element", name, 
			 "both @type and <xs:simpleType>/<xs:complexType> are specified.",
			 __FILE__, __LINE__);
      }
    }
    else {
      assert(elm.c0_.empty());
    }
  
    /* 3.3.3.4 */
    // not implemented.
  }
}



rep_validator::rep_validator(const xs::topLevelElement &tel, aka::id_type ns_id) {
  _element(tel, aka::qname(ns_id, tel.name_));
}

rep_validator::rep_validator(const xs::localElement &elm, aka::id_type ns_id, 
			     const aka::qname &hint) {
  /* 3.3.3.2 */
  if (!elm.ref_.empty() && !elm.name_.empty()) { 
    raise_name_error("xs:element", aka::qname(ns_id, elm.name_), "@ref and @name are specified.",
		     __FILE__, __LINE__);
  }
  
  if (elm.ref_.empty() && elm.name_.empty()) { 
    raise_name_error("xs:element under " + hint.qualified(), 
		     aka::qname("unknown"), "@ref or @name must be specified.",
		     __FILE__, __LINE__);
  }

  
  if (!elm.ref_.empty()) {
    if (!elm.c0_.empty() ||
	elm.nillable_ ||
	!elm.default_.empty() ||
	!elm.fixed_.empty() ||
	elm.form_.present() ||
	// 	(elm.block_ != xs::blockSet_none) ||
	!elm.type_.empty()) {
      raise_name_error("xs:element@ref", elm.ref_, "@ref is specified, "
		       "but one (or some) of <simpleType>, <complexType>, "
		       "@nillable, @default, @fixed, @form, @block, @type is (are) specified.",
		       __FILE__, __LINE__);
    }
  }
  else {
    _element(elm, aka::qname(ns_id, elm.name_));
  }
}
