/* -*- c++ -*- */
#ifndef AKAXISO_BUILTIN_IMPL_H__
#define AKAXISO_BUILTIN_IMPL_H__

#include <akaxiso/classes/types.h>
#include <akaxiso/classes/element.h>
#include <akaxiso/classes/membertypes.h>

namespace akaxiso {

  template<class V>
  class builtin_member_type_impl : public simpletype_member_type {
  public:
    builtin_member_type_impl(int offset) : offset_(offset), has_default_(false) {}
    virtual void write_text(const element &element, std::ostream &ostm, const global_attributes &gattrs) const; 
    virtual void read_text(element &element, std::istringstream &istm, const global_attributes &gattrs) const;

    virtual bool is_equal(const element &element1, const element &element2) const;
    virtual void copy(element &dst, const element &src) const;
    virtual void initialize_instance(element &element) const;
    virtual baseclass_id get_baseclass_id() const;

    virtual void set_default_value(const char *defval);
    virtual bool is_default(const element &element) const;

  protected:
    int offset_;
    V default_;
    bool has_default_;
 };

  template<class V>
  void builtin_member_type_impl<V>::set_default_value(const char *defval) {
    std::istringstream istm(defval);
    istm >> default_;
    has_default_ = true;
  }

  template<class V>
  bool builtin_member_type_impl<V>::is_default(const element &element) const{
    if (has_default_) {
      const V& v= *get_member_ptr<const V>(element, offset_);
      return v == default_;
    }
    return false;
  }


  template<class V>
  void builtin_member_type_impl<V>::write_text(const element &element, std::ostream &ostm, const global_attributes &gattrs) const {
    const V& v = * get_member_ptr<const V>(element, offset_);
    ostm << v;
  }

  template<class V>
  void builtin_member_type_impl<V>::read_text(element &element, std::istringstream &istm, const global_attributes &gattrs) const {
    V& v = * get_member_ptr<V>(element, offset_);
    istm >> v;
    if (istm.fail())
      throw parse_exception(__FILE__, __LINE__, "Failed to parse simpletype value.");
  }

  template<class V>
  bool builtin_member_type_impl<V>::is_equal(const element &element1, const element &element2) const {
    const V& v1 = *get_member_ptr<const V>(element1, offset_);
    const V& v2 = *get_member_ptr<const V>(element2, offset_);
    return v1 == v2;
  }

  template<class V>
  void builtin_member_type_impl<V>::initialize_instance(element &element) const{
    if (has_default_) {
      V& v = *get_member_ptr<V>(element, offset_);
      v = default_;
    }
  }

  template<class V>
  baseclass_id builtin_member_type_impl<V>::get_baseclass_id() const {
    return simpletype_id;
  }



  template<class V>
  void builtin_member_type_impl<V>::copy(element &dst, const element &src) const {
    V& vdst = *get_member_ptr<V>(dst, offset_);
    const V& vsrc = *get_member_ptr<const V>(src, offset_);
    vdst = vsrc;
  }


  //
  template<class V> class simpletype_member_type_impl;

  // Partial specializations for builtin types.

  template<> 
  class simpletype_member_type_impl<float> : public builtin_member_type_impl<float> {
  public:
    simpletype_member_type_impl(int offset) : builtin_member_type_impl<float>(offset){}
  };

  template<> 
  class simpletype_member_type_impl<double> : public builtin_member_type_impl<double> {
  public:
    simpletype_member_type_impl(int offset) : builtin_member_type_impl<double>(offset){}
  };

  template<> 
  class simpletype_member_type_impl<long> : public builtin_member_type_impl<long> {
  public:
    simpletype_member_type_impl(int offset) : builtin_member_type_impl<long>(offset){}
  };

  template<> 
  class simpletype_member_type_impl<short> : public builtin_member_type_impl<short> {
  public:
    simpletype_member_type_impl(int offset) : builtin_member_type_impl<short>(offset){}
  };


  template<> 
  class simpletype_member_type_impl<unsigned long> : public builtin_member_type_impl<unsigned long> {
  public:
    simpletype_member_type_impl(int offset) : builtin_member_type_impl<unsigned long>(offset){}
    virtual void read_text(element &element, std::istringstream &istm, const global_attributes &gattrs) const;
    virtual void set_default_value(const char *defval);
  };

  template<> 
  class simpletype_member_type_impl<std::string> : public builtin_member_type_impl<std::string> {
  public:
    simpletype_member_type_impl(int offset) : builtin_member_type_impl<std::string>(offset){}
    virtual void read_text(element &element, std::istringstream &istm, const global_attributes &gattrs) const;
    virtual void set_default_value(const char *defval);
  };


}



#endif
