#include "xmlserializer.h"
#include <iostream>
#include <sstream>

using namespace akaxiso;

void xmlserializer::escape_attribute_value(const std::string &value, std::ostream &ostm){

  for (std::string::const_iterator it = value.begin(); it != value.end(); ++it)
    if (*it == '\"')
      ostm << "&quot;";
    else if (*it == '<')
      ostm << "&lt;";
    else if (*it == '&')
      ostm << "&amp;";
    else 
      ostm << *it;
}

void xmlserializer::escape_text_value(const std::string &value, std::ostream &ostm){

  for (std::string::const_iterator it = value.begin(); it != value.end(); ++it)
    if (*it == '<')
      ostm << "&lt;";
    else if (*it == '&')
      ostm << "&amp;";
    else 
      ostm << *it;
  
}

void xmlserializer::serialize(const document *doc, std::ostream &ostm) {
  indent_ = 0;
  ostm << "<?xml version=\"1.0\" encoding=\"" << encoding_ << "\" ?>" << std::endl
       << std::endl;

  const element *e = doc->get_root_element();
  open_beginning_tag(doc->get_root_tagname().get_name(), ostm);
  write_namespace_attributes(ostm); 
  write_attributes(e, ostm);
  close_beginning_tag(ostm);

  write_entity(e, ostm);
  write_ending_tag(doc->get_root_tagname().get_name(), ostm);
  ostm << std::endl;
}


void xmlserializer::write_namespace_attributes(std::ostream &ostm) {
  const prefix_map &prefixes = static_global_attributes().get_prefix_map();
  for (prefix_map::const_iterator it = prefixes.begin();
    it != prefixes.end(); ++it) {
      const std::string &prefix = it->second;
      id_type namespace_id = it->first;
      new_line(ostm);
      ostm << " xmlns:" << prefix << "=\"" << namespaces().get_namespace_uri(namespace_id) << "\"";
  }
}

void xmlserializer::write_element(const element *element, const std::string &tagname, std::ostream &ostm) {
  new_line(ostm);
  open_beginning_tag(tagname, ostm);
  write_attributes(element, ostm);
  close_beginning_tag(ostm);
  write_entity(element, ostm);
  write_ending_tag(tagname, ostm);
}


void xmlserializer::open_beginning_tag(const std::string &tagname, std::ostream &ostm) {
  ostm << "<" << tagname;
}

void xmlserializer::close_beginning_tag(std::ostream &ostm) {
  ostm << '>';
}

void xmlserializer::write_attributes(const element *elm, std::ostream &ostm){
  const attribute_info &info = elm->get_attribute_info();
  for (attribute_info::const_iterator it = info.begin(); it != info.end(); ++it) {
    const member_operator &op = it->get_member_operator();
    const attribute_member_type &memtype = static_cast<const attribute_member_type&>(op);
    if (memtype.should_write(*elm)) {
      ostm << ' ' << it->get_localname() << "=\"";
      std::ostringstream ostring;
      memtype.write_text(*elm, ostring, static_global_attributes());
      escape_text_value(ostring.rdbuf()->str(), ostm);
      ostm << '\"';
    }
  }
}


void xmlserializer::write_ending_tag(const std::string &tagname, std::ostream &ostm) {
  ostm << "</" << tagname << ">";
}

void xmlserializer::write_entity(const element* e, std::ostream &ostm) {


  if (e->get_baseclass_id() == sequence_id) {

    inc_indent_level();

    const sequence *seq = static_cast<const sequence*>(e);
    const sequence_info &info = seq->get_sequence_info();
    for (sequence_info::const_iterator it = info.begin(); it != info.end(); ++it) {

      const member_operator &op = it->get_member_operator();
      switch (op.get_baseclass_id()) 
	{
	case simpletype_id:        {
	  new_line(ostm);
	  open_beginning_tag(it->get_tagname(), ostm); close_beginning_tag(ostm);
	  
	  std::ostringstream ostring;
	  static_cast<const simpletype_member_type&>(op).write_text(*seq, ostring, static_global_attributes());
          escape_text_value(ostring.rdbuf()->str(), ostm);
	  
          write_ending_tag(it->get_tagname(), ostm);
          break;
        }
	case sequence_id:
        case choice_id:        {
          const element *memelm = 
            static_cast<const element_member_type&>(op).get_member_element(*seq);
          write_element(memelm, it->get_tagname(), ostm);
          break;
        }
	case simplecontent_id:	{
	  const element *memelm = 
	    static_cast<const element_member_type&>(op).get_member_element(*seq);
	  const simplecontent *sim = static_cast<const simplecontent*>(memelm);
	  new_line(ostm);
	  open_beginning_tag(it->get_tagname(), ostm); 
	  write_attributes(sim, ostm); 
	  close_beginning_tag(ostm);
	  
	  std::ostringstream ostring;
	  sim->write_text(ostring, static_global_attributes());
          escape_text_value(ostring.rdbuf()->str(), ostm);
	  
          write_ending_tag(it->get_tagname(), ostm);
          break;
	}
	case memberarray_id: {
	  const memberarray_member_type &mt = static_cast<const memberarray_member_type&>(op);
	  const memberarray *ma = mt.get_memberarray(*seq);

	  for (memberarray::const_iterator mit = ma->begin(); mit != ma->end(); ++mit) {
	    new_line(ostm);
	    open_beginning_tag(it->get_tagname(), ostm);
	    write_attributes(*mit, ostm);
	    close_beginning_tag(ostm);
	    write_entity(*mit, ostm);
	    write_ending_tag(it->get_tagname(), ostm);
	  }
	  break;
	}
        default:
          throw internal_error();
      }
    }

    dec_indent_level();
    new_line(ostm);

  }
  else if (e->get_baseclass_id() == choice_id) {

    inc_indent_level();

    const choice *cho = static_cast<const choice*>(e);
    for (choice::const_iterator it = cho->begin(); it != cho->end(); ++it) {
      const element *e = it->get_element();
      write_element(e, it->get_tagname(), ostm);
    }

    dec_indent_level();
    new_line(ostm);

  }
  else if (e->get_baseclass_id() == simplecontent_id) {
    const simplecontent *sim = static_cast<const simplecontent*>(e);
    sim->write_text(ostm, static_global_attributes());
  }


}

