#include "platform.h"

#include <config.h>

#ifdef USE_XERCESC
#include <xercesc/util/PlatformUtils.hpp>
#endif

#include <akaxiso/akaxiso.h>
#include <akaxiso/xerces/parser.h>
#include <iostream>
#include <iomanip>
#include <fstream>

#include "exception.h"
#include "schemastructure.h"
#include "classgen.h"


void usage() {

  std::cout << "Osixaka (" << PACKAGE << "-" << VERSION << ")" << std::endl
	    << " usage  osixaka [-o output directory] [-h]" << std::endl
	    << " osixaka take XML-schema document as an input from stdin,"
	    << " and generates serializable class declaration/implementation for akaxiso." << std::endl
	    << "  -o: specify output directory to write generated files." << std::endl
	    << "  -h: show this message." << std::endl;

}


int main(int argc, char* argv[]){
  
  // Initialize Xerces.
#ifdef USE_XERCESC
  XMLPlatformUtils::Initialize();
#endif
  akaxiso::initialize();
  // Initialize XML-Schema Document Classes.
  xml_schema::initialize();

  std::ostream & logstm = std::cerr;
  std::string outputdir = ".";

  osixaka::generator gen;
  char c;
  
  while ((c = getopt(argc, argv, "o:h")) != -1) {
    switch (c) {
    case 'o': {
      outputdir = optarg;
      break;
    }
    case 'h' :{
      usage();
      exit(0);
    }
    default: {
      logstm << "Unknown option." << std::endl;
      exit(1);
    }
    }
  }


  // Parse
#ifdef USE_XERCESC
  akaxiso::xerces_parser parser;
#else
  akaxiso::yggxml_parser parser;
#endif


  akaxiso::document *doc = NULL;
  try {
    doc = parser.parse_stdin();
  }
  catch (const akaxiso::parse_exception &e) {
    logstm << e.what() << std::endl;
    logstm << "Parse error detected.  Abort immediately." << std::endl;
    exit(1);
  }
   catch ( ... ) {
     logstm << "Unhandled parse error detected.  Abort immediatedly." << std::endl;
     exit(1);
   }


//   akaxiso::xmlserializer ser;
//   ser.serialize(doc, std::cout);
  
  try {
    gen.generate_classes(static_cast<xml_schema::SchemaDoc*>(doc));

    std::ofstream docfile(std::string(outputdir +"/documents.h").c_str());
    gen.output_document_file(docfile);
    
    std::ofstream setupfile_header(std::string(outputdir + "/setup.h").c_str());
    std::ofstream setupfile_impl(std::string(outputdir + "/setup.cpp").c_str());
    gen.output_setup_file(setupfile_header, setupfile_impl);
    
    std::ofstream elementfile(std::string(outputdir + "/elements.h").c_str());
    gen.output_element_file(elementfile);
    
  }
  catch (const osixaka::fatal_error &e) {
    logstm << e.what() << std::endl;
    logstm << "Failed to generate files. Abort immediately." << std::endl;
    exit(1);
  }
  catch ( ... ) {
    logstm << "Unknown exception caught. Abort immediately." << std::endl;
    exit(1);
  }
        
  delete doc;

  logstm << "Generatiion completed." << std::endl;

  // Uninitializers.
  xml_schema::uninitialize();
  akaxiso::uninitialize();
#ifdef USE_XERCEC
  XMLPlatformUtils::Terminate();
#endif

  return 0;
}
