/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with GNU Emacs; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
*/
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iconv.h>
#include <list>
#include "imconn.h"
#include "packet.h"

#define PORT_NUMBER 9010

int iiim_trace_flag;
int trace_flag;

static int port_number=PORT_NUMBER;
static std::list<IMConn *> ims;

static void parse_args(int argc,char **argv);
static int event_loop();
static int accept_proc(int fd);


#if __GLIBC_MINOR__ == 1 || FreeBSD
#define ICONV_ARG2 (const char **)
#else
// conforming to new standard draft 5/2000
#define ICONV_ARG2
#endif

static char *sa2str (struct sockaddr*, socklen_t); 

static std::list<int> serv_fds;

static int init_serv_fd(char *hostname,char *port);

STRING createSTRING(char *s)
{
  STRING str;
  iconv_t conv;

  conv = iconv_open("UTF-16BE","EUC-JP");
  if ( (int)conv == -1 ){
    printf("Failed to create iconv.\n");
    exit(0);
  }

  size_t ilen,olen,buflen,ret,len;
  int i;
  len = strlen(s);
  ilen = len;
  buflen = ilen*5;
  char *outbuf = (char *)alloca(buflen);
  char *pp = outbuf;
  ret = iconv(conv,ICONV_ARG2 &s,&ilen,&outbuf,&buflen);
  if ( ret == -1 ){
    printf("failed iconv.\n");
    return str;
  }
  unsigned short *p;
  p = (unsigned short*)pp;
  olen = len*5 - buflen;

  for ( i = 0 ; i < olen /2 ; i++){
    // Is UTF BIG ENDIAN?
    str.push_back((p[i]>>8)|((p[i]<<8)&65280));
  }
  iconv_close(conv);
  return str;
}

int init_serv_fd(char *hostname,char *port)
{
  int foo;
  int s;
#ifdef INET6
  struct in_addr ia;
  struct addrinfo hints, *res, *ai; 
 
  memset(&hints, 0, sizeof(hints)); 
  hints.ai_family = AF_UNSPEC; 
  hints.ai_socktype = SOCK_STREAM; 
  hints.ai_protocol = IPPROTO_TCP; 
  hints.ai_flags = AI_PASSIVE; 
 
  if (getaddrinfo(NULL, port, &hints, &res) != 0) {
    return -1; 
  }
  
  for (ai = res; ai != NULL; ai= ai->ai_next) { 
    if ((s = socket(ai->ai_family, ai->ai_socktype, 
		    ai->ai_protocol)) < 0) {
      continue; 
    }
 
    foo = 1;
    foo = setsockopt(s,SOL_SOCKET,SO_REUSEADDR,&foo ,4);
    if ( foo == -1){
      // IGNORE. No problem.
      //perror("failed to set reuse option");
    }

    if ( bind(s, ai->ai_addr, ai->ai_addrlen) < 0) {
      close(s); 
      continue; 
    } 
 
    if (listen(s,1) < 0) { 
      close(s); 
      continue; 
    } 
 
    serv_fds.push_back(s); 
  } 

  if (res != NULL) {
    freeaddrinfo(res);
  }

#else /*!INET6*/

  struct sockaddr_in myhost;

  s = socket(AF_INET,SOCK_STREAM,0);
  if (  s == -1 ){
    perror("failed in socket()");
    return -1;
  }

  foo = 1;
  foo = setsockopt(s,SOL_SOCKET,SO_REUSEADDR,&foo ,4);
  if ( foo == -1){
    // IGNORE. No problem.
    //perror("failed to set reuse option");
    //return -1;
  }

  myhost.sin_family=AF_INET;
  myhost.sin_port=htons(port_number);
  myhost.sin_addr.s_addr=htonl(INADDR_ANY);

  foo = bind(s,(struct sockaddr *)&myhost,16);
  if( foo == -1 ){
    perror("failed in bind()");
    return -1;
  }
  foo = listen(s,5);
  if ( foo == -1){
    perror("failed in listen()");
    return -1;
  } 
  serv_fds.push_back(s);
#endif
  if ( !serv_fds.size() ){
    return -1;
  }
  return 0;
}

int accept_proc(int fd)
{
  socklen_t len;
#ifdef INET6
  struct sockaddr_storage peer;
#else
  struct sockaddr peer;
#endif
  len = sizeof(peer);
  int new_fd = accept(fd,(struct sockaddr *)&peer,&len);
  if ( new_fd == -1 ){
    perror("Failed to accept.\n");
    return -1;
  }else{
    printf("Accepted TCP connection from %s\n",
	   sa2str((struct sockaddr *)&peer,len));
  }
  if ( trace_flag ){
    printf("Accept\n");
  }
  IMConn *conn = new IMConn(new_fd);
  ims.push_back(conn);
  return 0;
}

int event_loop()
{
  /* setup poll table */
  struct pollfd *polltab;
  int nr_fd ,nth,nr_accept_fd;
  nr_fd = serv_fds.size() + ims.size();
  polltab = (struct pollfd *)alloca(sizeof(struct pollfd)*nr_fd);
  nth = 0;

  /* setup polltab for fd to accept */
  {
    std::list<int>::iterator it;
    for ( it = serv_fds.begin() ; it != serv_fds.end() ; it ++ ){
      polltab[nth].fd = *it;
      polltab[nth].events = POLLIN;
      nth++;
    }
  }

  /* setup polltab for each connection */
  {
    std::list<IMConn *>::iterator it;
    for ( it = ims.begin() ; it != ims.end() ; it ++){
      polltab[nth].fd = (*it)->getFd();
      polltab[nth].events = POLLIN;
      if ( (*it)->isWritePending()){
	polltab[nth].events |= POLLOUT;
      }
      nth++;
    }
  }

  /* do poll */
  poll(polltab,nth,1000);

  /* check result for each connection */
  nth = 0;
  /*accept fds*/
  {
    std::list<int>::iterator it;
    for ( it = serv_fds.begin() ; it != serv_fds.end() ; it ++){
      if ( polltab[nth].revents & POLLIN ){
	accept_proc(*it);
      }
      nth++;
    }
  }
  {
    /*established connections*/
    std::list<IMConn *>::iterator it;
    for ( it = ims.begin() ; it != ims.end() && nth< nr_fd ; it ++){
      if ( polltab[nth].revents & POLLIN ){
	(*it)->read_proc();
      }
      if ( polltab[nth].revents & POLLOUT ){
	(*it)->write_proc();
      }
      nth ++;
    }
  }

  /* reap invalid connection */
  {
    std::list<IMConn *>::iterator it;
    for ( it = ims.begin() ; it != ims.end() ; ){
      if ( (*it)->isValid() ){
	it++;
      }else{
	if ( trace_flag ){
	  printf("Disconnect\n");
	}
	IMConn *im = *it;
	it = ims.erase(it);
	delete im;
      }
    }
  }
  return 0;
}

int main(int argc,char **argv)
{
  int newfd,addr_len;
  struct sockaddr_in peer;

  trace_flag = 1;
  iiim_trace_flag = 1;

  char portnum[5];
  char buf[NI_MAXSERV];
  sprintf(portnum,"%d",PORT_NUMBER);

  if (init_serv_fd("localhost",portnum) < 0) {
    printf("Failed to find usable address.\n"
	   " Maybe other instance is running.\n");
    exit(-1);
  }

  printf("Intentionally broken now.\n");
  while(!event_loop());
  return 0;
}

static char *sa2str(struct sockaddr *sa, socklen_t salen) 
{ 
  static char retbuf[NI_MAXHOST + NI_MAXSERV + 3]; 
  char hbuf[NI_MAXHOST], sbuf[NI_MAXSERV]; 
 
#ifdef INET6 
  if(getnameinfo(sa, salen, hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), 
                NI_NUMERICHOST | NI_NUMERICSERV) != 0) 
    return("???"); 
 
  sprintf(retbuf, "[%s]:%s", hbuf, sbuf); 
#else
  // IPv4 Only
  sprintf(retbuf,"[IPv4 Connection]");
#endif
  return retbuf;
}
