/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.tmx.core;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;

import benten.core.BentenConstants;
import benten.core.io.TmxFileFilter;
import benten.core.model.BentenTmx;
import benten.core.model.BentenXliff;
import benten.core.text.Strings;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import benten.twa.tmx.core.valueobject.BentenExtractTmxFragmentProcessInput;
import benten.twa.tmx.messages.BentenExtractTmxFragmentMessages;
import blanco.tmx.valueobject.BlancoTmxTu;
import blanco.tmx.valueobject.BlancoTmxTuv;
import blanco.xliff.BlancoXliffUtil;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * TMX フラグメントの抽出
 *
 * <pre>
 * XLIFF フラグメントから、TMX フラグメントを抽出します。
 *   1.  XLIFF から翻訳単位を抽出して TMX を作成します。
 * </pre>
 * 
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳メモリー断片抽出・マージ機能: TMXフラグメント抽出機能」に対応します。
 * 
 * @author IGA Tosiki
 */
public class BentenExtractTmxFragmentProcessImpl extends AbstractTraverseDir implements BentenExtractTmxFragmentProcess {
	/**
	 * TMXフラグメント抽出機能のためのメッセージ。
	 */
	protected static final BentenExtractTmxFragmentMessages fMsg = new BentenExtractTmxFragmentMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenExtractTmxFragmentProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/** 
	 * ファイル名のサフィックスに利用する今日の日付。
	 * <UL>
	 * <LI>この文字列は、ファイル名のサフィックスの用途のみで利用します。
	 * <LI>この文字列は、国際化(i18n)の必要はありません。
	 * </UL>
	 */
	private final String nowDate = new SimpleDateFormat("yyyyMMdd").format(new Date()); //$NON-NLS-1$

	/** 出力 TMX オブジェクト。 */
	private BentenTmx fOutTmx;

	/**
	 * この処理の実行結果情報を取得します。
	 * 
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenExtractTmxFragmentProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenExtractTmxFragmentProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirXliff = new File(input.getXliffdir());
		final File dirTmx = new File(input.getTmxdir());
		if (dirXliff.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE013(input.getXliffdir()));
		}
		if (dirXliff.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE014(input.getXliffdir()));
		}
		if (dirTmx.exists() == false) {
			if (dirTmx.mkdirs() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE015(input.getTmxdir()));
			}
		} else {
			if (dirTmx.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE016(input.getTmxdir()));
			}
		}

		final File fileTmx = createFile(input.getTranstargetid(), dirTmx, input.getIncludedate());
		fOutTmx = BentenTmx.newInstance(fileTmx, input.getTranssourcelang());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(dirXliff);

		if (progress(fMsg.getCoreP003())) {
			return 6;
		}

		Collections.sort(fOutTmx.getTuList(), new BentenTmxTuComparator());
		BentenTmxTuUtil.removeDuplicatedTuv(fOutTmx.getTuList());

		if (progress(fMsg.getCoreP004(fileTmx.getName()))) {
			return 6;
		}

		try {
			fOutTmx.save();
		} catch (IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE012(fileTmx.getName(), e.toString()));
		}

		if (progress(fMsg.getCoreP005(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	/**
	 * ファイルの作成。
	 * @param transtargetid 翻訳対象 id。
	 * @param toDir あて先ディレクトリー。
	 * @param includeDate 日付をファイル名に含めるかどうか。
	 * @return 作成されたファイル・オブジェクト。
	 */
	protected File createFile(final String transtargetid, final File toDir, final boolean includeDate) {
		final StringBuilder name = new StringBuilder();
		name.append(transtargetid);
		if (includeDate) {
			name.append("-" + nowDate); //$NON-NLS-1$
		}
		return TmxFileFilter.appendExtension(new File(toDir, name.toString()));
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	protected void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenExtractTmxFragmentProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		BentenXliff inXliff = null;
		try {
			inXliff = BentenXliff.loadInstance(file);
		} catch (IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE011(file.getName(), e.toString()));
		}

		int transUnitCount = 0;
		int transUnitTotalCount = 0;
		for (BlancoXliffFile inFile : inXliff.getFileList()) {
			transUnitTotalCount += inFile.getBody().getTransUnitList().size();
			for (BlancoXliffTransUnit inUnit : inFile.getBody().getTransUnitList()) {
				if (progress(fMsg.getCoreP101(file.getName(), BigDecimal.valueOf(++transUnitCount), BigDecimal
						.valueOf(transUnitTotalCount)))) {
					break;
				}

				processTransUnit(inUnit, inFile.getSourceLanguage(), inFile.getTargetLanguage());
			}
		}
	}

	/**
	 * 翻訳単位を処理します。
	 * 
	 * @param inUnit 処理対象となる処理単位。
	 * @param sourceLang 翻訳元言語。
	 * @param targetLang 翻訳先言語。
	 */
	void processTransUnit(final BlancoXliffTransUnit inUnit, final String sourceLang, final String targetLang) {
		if (inUnit.getTranslate() == false) {
			return;
		}
		if (inUnit.getSource() == null || inUnit.getSource().length() == 0) {
			return;
		}
		if (inUnit.getTarget() == null || inUnit.getTarget().getTarget() == null) {
			return;
		}

		// TMX反映除外のチェック。
		if (BlancoXliffUtil.isXTmOmit(inUnit)) {
			return;
		}

		// state のチェック
		if (fInput.getExtractbystate() != null) {
			if (fInput.getExtractbystate().trim().length() == 0) {
				// state 無しを探したい。
				if (inUnit.getTarget().getState() == null || inUnit.getTarget().getState().trim().length() == 0) {
					// これが出力対象。
				} else {
					// 出力対象外。
					return;
				}
			} else {
				// 通常の state 指定。
				if (fInput.getExtractbystate().equals(inUnit.getTarget().getState())) {
					// state が一致します。処理対象です。
				} else {
					// state が異なるので処理対象外とします。
					return;
				}
			}
		}

		fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);

		// 翻訳元、翻訳先にニーモニック・キーが含まれるかどうか。
		boolean isMnemonicKeyExist = false;
		if (fInput.getIgnoremnemonickeytmxextract()) {
			if (findMnemonicKey(inUnit.getSource()) && findMnemonicKey(inUnit.getTarget().getTarget())) {
				isMnemonicKeyExist = true;
			}
		}

		final BlancoTmxTu tu = new BlancoTmxTu();
		fOutTmx.getTuList().add(tu);
		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
		{
			final BlancoTmxTuv tuv = new BlancoTmxTuv();
			tu.getTuvList().add(tuv);
			tuv.setLang(sourceLang);
			tuv.setSeg(inUnit.getSource());

			if (fInput.getIgnorewhitespacetmxextract()) {
				// ホワイトスペースを無視した内容の TMX を抽出する場合。
				tuv.setSeg(Strings.removeRedundantWhitespace(tuv.getSeg()));
			}

			if (isMnemonicKeyExist) {
				tuv.setSeg(Strings.stripMnemonicKey(tuv.getSeg()));
			}
		}
		{
			final BlancoTmxTuv tuv = new BlancoTmxTuv();
			tu.getTuvList().add(tuv);
			tuv.setLang(targetLang);
			tuv.setSeg(inUnit.getTarget().getTarget());

			if (fInput.getIgnorewhitespacetmxextract()) {
				// ホワイトスペースを無視した内容の TMX を抽出する場合。
				tuv.setSeg(Strings.removeRedundantWhitespace(tuv.getSeg()));
			}

			if (isMnemonicKeyExist) {
				tuv.setSeg(Strings.stripMnemonicKey(tuv.getSeg()));
			}
		}
	}

	/**
	 * 与えられた文字列にニーモニック・キーが含まれているかどうかチェックします。
	 * 
	 * <UL>
	 * <LI>日本風にもーニック・キーにのみ対応します。
	 * </UL>
	 * 
	 * @param input ニーモニック付きの文字列。
	 * @return ニーモニック・キーが含まれるかどうか。
	 */
	boolean findMnemonicKey(final String input) {
		final String after = Strings.stripMnemonicKey(input);
		return (input.length() != after.length());
	}
}
