/*******************************************************************************
 * Copyright (c) 2010  NEC Soft, Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.engine.ecma376;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;

import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.BentenTwaFilterEngine;
import benten.twa.filter.model.SentencePartitionList;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffBody;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTransUnit;
import blanco.xml.bind.BlancoXmlMarshaller;
import blanco.xml.bind.BlancoXmlUnmarshaller;
import blanco.xml.bind.valueobject.BlancoXmlCharacters;
import blanco.xml.bind.valueobject.BlancoXmlDocument;
import blanco.xml.bind.valueobject.BlancoXmlElement;

/**
 * ECMA-376 の xlsx ファイルの翻訳処理をおこないます。
 * 
 * <UL>
 * <LI>このクラスは、ECMA-376 -> xlsx ファイルのフィルターです。
 * <LI>Microsoft Office Excel の xlsx 形式ファイルの入出力を担います。これは ECMA-376 の ExcelProcessing に相当します。
 * <LI>対象ファイル: ファイル名が拡張子「.xlsx」で終了するもの。ただしファイル名は「~$」からは始まらないこと。ZIP ファイルとして読み込んだ場合に ZIP エントリが 1 件以上存在するもの。
 * <LI>対象データ: ECMA-376 -> Excel のうち、<t> のテキスト。
 * </UL>
 *
 * @author IGA Tosiki
 */
public class BentenTwaFilterEcma376XlsxEngine implements BentenTwaFilterEngine {
	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		final String fileName = file.getName().toLowerCase();
		if (fileName.endsWith(".xlsx") && false == fileName.startsWith("~$")) { //$NON-NLS-1$ //$NON-NLS-2$
			// ファイル名が .xlsx で終わり、かつ、テンポラリファイルを表す ~$ では始まらないものを処理対象とします。

			// ZIP として適切に読み込みうことができるかどうかチェックします。
			try {
				byte[] bytesFile = BlancoFileUtil.file2Bytes(file);
				new Ecma376XlsxProcessor() {
					@Override
					public void processXml(final InputStream inStream, final OutputStream outStream) {
					}
				}.processZip(file.getName(), bytesFile, new ByteArrayOutputStream());
			} catch (IllegalArgumentException e) {
				// ZIP ファイルとして不正であるか、または読み取りパスワードで保護されています。そのため、このファイルは処理しません。
				return false;
			} catch (IOException e) {
				// それ以外の入出力例外が発生しました。処理を中断させます。
				throw new IllegalArgumentException(e);
			}

			// このファイルは処理可能であると判断しました。
			return true;
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliff(final File sourceFile, final BlancoXliff xliff, final HelpTransUnitId id,
			final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliffFile xliffFile = xliff.getFileList().get(0);

		// ECMA-376 の xlsx ファイルを表現します。
		xliffFile.setDatatype("x-xlsx"); //$NON-NLS-1$

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();

		try {
			new Ecma376XlsxProcessor() {
				@Override
				public void processXml(final InputStream inStream, final OutputStream outStream) {
					final BlancoXmlDocument document = new BlancoXmlUnmarshaller().unmarshal(inStream);

					new Ecma376XlsxXmlParser() {
						@Override
						public void fireT(final BlancoXmlElement eleChild, final String text) {
							splitText(text);
						}

						/**
						 * 与えられたテキストを句読点などで分割してから翻訳単位として登録します。
						 * 
						 * @param text 分割対象となる翻訳前テキスト。
						 */
						private void splitText(final String text) {
							final BlancoXliffBody body = xliffFile.getBody();
							final SentencePartitionList sentenceList = new SentencePartitionList(text);
							if (sentenceList.size() > 1) {
								for (final String sentence : sentenceList) {
									id.incrementSubSeq(sentenceList.size());
									createTransUnit(body, id, sentence, resultInfo);
								}
							} else {
								createTransUnit(body, id, text, resultInfo);
							}
							id.incrementSeq();
						}
					}.parse(document);

					// 書き出し処理。
					new BlancoXmlMarshaller().marshal(document, outStream);
				}
			}.processZip(sourceFile.getName(), bytesFile, outStream);
		} finally {
			outStream.flush();
			outStream.close();
		}
	}

	/**
	 * trans-unit の作成。
	 *
	 * <UL>
	 * <LI>このメソッドは XLSX -> XLIFF で利用されます。
	 * </UL>
	 *
	 * @param body XLIFF の body 要素モデル
	 * @param helpId trans-unit の id 属性モデル
	 * @param key キーの値
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	private void createTransUnit(final BlancoXliffBody body, final HelpTransUnitId helpId, final String source,
			final BentenProcessResultInfo resultInfo) {
		resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

		final BlancoXliffTransUnit unit = new BlancoXliffTransUnit();
		unit.setId(helpId.toString());
		unit.setSource(source);
		body.getTransUnitList().add(unit);
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliff(final File sourceFile, final BlancoXliff xliff, final BentenProcessResultInfo resultInfo)
			throws IOException {

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();

		try {
			// xlsx ファイル単位で処理をすすめるので、Ecma376XlsxProcessor クラスより外側でインスタンスを保持する必要があります。
			final Iterator<BlancoXliffTransUnit> transUnits = xliff.getFileList().get(0).getBody().getTransUnitList()
					.iterator();

			new Ecma376XlsxProcessor() {
				@Override
				public void processXml(final InputStream inStream, final OutputStream outStream) {

					final BlancoXmlDocument document = new BlancoXmlUnmarshaller().unmarshal(inStream);

					new Ecma376XlsxXmlParser() {
						@Override
						public void fireT(final BlancoXmlElement eleChild, final String text) {
							translate(eleChild, text);
						}

						/**
						 * 与えられた XML エレメントのテキストを翻訳後のものに置き換えます。
						 * 
						 * @param eleChild 処理対象となる XML エレメント。
						 * @param originalText 翻訳前のテキスト。
						 */
						void translate(final BlancoXmlElement eleChild, final String originalText) {

							resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);
							final SentencePartitionList targetList = new SentencePartitionList();

							while (transUnits.hasNext()) {
								final BlancoXliffTransUnit transUnit = transUnits.next();
								if (transUnit.getTranslate() == false || transUnit.getTarget() == null
										|| transUnit.getTarget().getTarget() == null) {
									targetList.add(transUnit.getSource());
								} else {
									targetList.add(transUnit.getTarget().getTarget());
								}
								if (!HelpTransUnitId.hasContinue(transUnit.getId())) {
									break;
								}
							}
							final String target = targetList.join();
							if (target.length() > 0) {
								for (int indexRemove = eleChild.getChildNodes().size(); indexRemove > 0; indexRemove--) {
									eleChild.getChildNodes().remove(indexRemove - 1);
								}
								// 翻訳後データで置換します。
								final BlancoXmlCharacters characters = new BlancoXmlCharacters();
								characters.setValue(target);
								eleChild.getChildNodes().add(characters);
							}
						}
					}.parse(document);

					// 書き出し処理。
					new BlancoXmlMarshaller().marshal(document, outStream);
				}
			}.processZip(sourceFile.getName(), bytesFile, outStream);
		} finally {
			outStream.flush();
			outStream.close();
		}

		return outStream.toByteArray();
	}
}
