/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.model;

import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 文分割リストです。
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings("serial")
public class SentencePartitionList extends LinkedList<String> {

	/**
	 * コンストラクター。
	 */
	public SentencePartitionList() {
	}

	/**
	 * コンストラクター。
	 * @param text テキスト
	 */
	public SentencePartitionList(final String text) {

		// 単純に分割する
		final LinkedList<String> tempList = split(text);

		// このインスタンスのリストに格納
		addAll(tempList);
	}

	/**
	 * テキストを以下のルールで分割します。
	 *
	 * <UL>
	 * <LI>英語の句点および続行するホワイトスペース。
	 * <LI>日本語の句点。
	 * </UL>
	 *
	 * <UL>
	 * <LI>区切り文字には「. : ? !」をもちい、区切り文字のあとに 1 つ以上のホワイトスペースが続いているもの。
	 * <LI>区切り文字には「。 ： ？ ！」であるもの。
	 * <LI>注意: 分割セパレータについて、JapaneseTranslationString クラス内の記述と関連があります。
	 * </UL>
	 *
	 * @param text テキスト
	 * @return テキストを分割した文を要素に持つリスト
	 */
	protected LinkedList<String> split(final String text) {
		final Matcher matcher = Pattern.compile(".+?(\\.|:|\\?|!)(\\s|&nbsp;)+|.+?(。|：|？|！)+").matcher(text); //$NON-NLS-1$
		final LinkedList<String> list = new LinkedList<String>();
		while (matcher.find()) {
			final StringBuffer sb = new StringBuffer();
			matcher.appendReplacement(sb, Matcher.quoteReplacement(matcher.group()));
			list.add(sb.toString());
		}
		final StringBuffer sb = new StringBuffer();
		matcher.appendTail(sb);
		if (list.size() == 0 || sb.length() > 0) {
			list.add(sb.toString());
		}
		return list;
	}

	/**
	 * 要素文字列を連結します。
	 *
	 * <UL>
	 * <LI>要素の末尾が句読点の場合、適切な空白が挿入されます。
	 * </UL>
	 *
	 * @return 要素が連結された文字列。null になることはありません。
	 */
	public String join() {

		final StringBuilder sb = new StringBuilder();
		String previous = null;

		for (final String target : this) {

			// 前の要素の末尾が英語句読点「.?!」の場合、空白を付加します。
			if (previous != null && previous.matches(".+[\\.\\?!]")) { //$NON-NLS-1$

				// 現在の要素の先頭が 空白、HTML 開始、閉じかっこ類など「<)]}」でない場合
				if (!target.matches("[\\s<\\)\\]\\}].*")) { //$NON-NLS-1$
					sb.append(" "); //$NON-NLS-1$
				}
			}
			sb.append(target);
			previous = target;
		}
		return sb.toString();
	}
}
