// $Id: BRbDebug.cpp,v 1.12 2002/12/04 08:17:38 yuya Exp $

#include "BRbDebug.h"
#include <stdarg.h>

////////////////////////////////////////////////////////////////////////////////

static const char* NODE_NAMES[] = {
	"NODE_METHOD",
	"NODE_FBODY",
	"NODE_CFUNC",
	"NODE_IFUNC",
	"NODE_SCOPE",
	"NODE_BLOCK",
	"NODE_IF",
	"NODE_CASE",
	"NODE_WHEN",
	"NODE_OPT_N",
	"NODE_WHILE",
	"NODE_UNTIL",
	"NODE_ITER",
	"NODE_FOR",
	"NODE_BREAK",
	"NODE_NEXT",
	"NODE_REDO",
	"NODE_RETRY",
	"NODE_BEGIN",
	"NODE_RESCUE",
	"NODE_RESBODY",
	"NODE_ENSURE",
	"NODE_AND",
	"NODE_OR",
	"NODE_NOT",
	"NODE_MASGN",
	"NODE_LASGN",
	"NODE_DASGN",
	"NODE_DASGN_CURR",
	"NODE_GASGN",
	"NODE_IASGN",
	"NODE_CDECL",
	"NODE_CVASGN",
	"NODE_CVDECL",
	"NODE_OP_ASGN1",
	"NODE_OP_ASGN2",
	"NODE_OP_ASGN_AND",
	"NODE_OP_ASGN_OR",
	"NODE_CALL",
	"NODE_FCALL",
	"NODE_VCALL",
	"NODE_SUPER",
	"NODE_ZSUPER",
	"NODE_ARRAY",
	"NODE_ZARRAY",
	"NODE_HASH",
	"NODE_RETURN",
	"NODE_YIELD",
	"NODE_LVAR",
	"NODE_DVAR",
	"NODE_GVAR",
	"NODE_IVAR",
	"NODE_CONST",
	"NODE_CVAR",
	"NODE_CVAR2",
	"NODE_NTH_REF",
	"NODE_BACK_REF",
	"NODE_MATCH",
	"NODE_MATCH2",
	"NODE_MATCH3",
	"NODE_LIT",
	"NODE_STR",
	"NODE_DSTR",
	"NODE_XSTR",
	"NODE_DXSTR",
	"NODE_EVSTR",
	"NODE_DREGX",
	"NODE_DREGX_ONCE",
	"NODE_ARGS",
	"NODE_ARGSCAT",
	"NODE_ARGSPUSH",
	"NODE_RESTARGS",
	"NODE_BLOCK_ARG",
	"NODE_BLOCK_PASS",
	"NODE_DEFN",
	"NODE_DEFS",
	"NODE_ALIAS",
	"NODE_VALIAS",
	"NODE_UNDEF",
	"NODE_CLASS",
	"NODE_MODULE",
	"NODE_SCLASS",
	"NODE_COLON2",
	"NODE_COLON3",
	"NODE_CREF",
	"NODE_DOT2",
	"NODE_DOT3",
	"NODE_FLIP2",
	"NODE_FLIP3",
	"NODE_ATTRSET",
	"NODE_SELF",
	"NODE_NIL",
	"NODE_TRUE",
	"NODE_FALSE",
	"NODE_DEFINED",
	"NODE_NEWLINE",
	"NODE_POSTEXE",
#ifdef C_ALLOCA
	"NODE_ALLOCA",
#endif
	"NODE_DMETHOD",
	"NODE_BMETHOD",
	"NODE_MEMO",
	"NODE_LAST"
};

////////////////////////////////////////////////////////////////////////////////

BRbDebug::BRbDebug(bool debug, FILE* out, BRB_EXCEPTION excpfunc)
{
	m_debug    = debug;
	m_out      = out;
	m_excpfunc = excpfunc;
}

BRbDebug::~BRbDebug()
{
}

////////////////////////////////////////////////////////////////////////////////

void
BRbDebug::printf(const char* format, ...)
{
	if ( m_debug ) {
		va_list ap;
		va_start(ap, format);
		::vfprintf(m_out, format, ap);
		va_end(ap);
	}
}

void
BRbDebug::dumpnode(NODE* node)
{
	if ( m_debug ) {
		dump_node(node, 0);
	}
}

void
BRbDebug::raise(const char* fmt, ...)
{
	if ( !m_excpfunc ) {
		return;
	}

	va_list ap;
	va_start(ap, fmt);

#ifdef WIN32
	const int size = 512;
#else
	const int size = ::vsprintf(NULL, fmt, ap);
#endif

	char* msg = new char[size + 1];
	msg[size] = '\0';

	::vsprintf(msg, fmt, ap);
	m_excpfunc(msg);

	delete[] msg;

	va_end(ap);
}

////////////////////////////////////////////////////////////////////////////////

void
BRbDebug::p(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	::vfprintf(m_out, fmt, ap);
	va_end(ap);
}

void
BRbDebug::lp(int level, const char *fmt, ...)
{
	for ( int i = 0; i < level; i++ ) {
		::fprintf(m_out, "  ");
	}

	va_list ap;
	va_start(ap, fmt);
	::vfprintf(m_out, fmt, ap);
	va_end(ap);
}

void
BRbDebug::lpdump(int level, char *name, NODE *node)
{
	lp(level, "%s:\n", name);
	dump_node(node, level + 1);
}

void
BRbDebug::nl()
{
	putc('\n', m_out);
}

char*
BRbDebug::tail(char* s, char* e)
{
	char* t = (char*)::memchr(s, '\n', e - s);
	if ( !t ) return e;

	while ( (*++t == '\n' || *t == '\r') && t < e );

	return t;
}

void
BRbDebug::dump_literal(long lit, int level)
{
	nl();

	if (FIXNUM_P(lit)) {
		lp(level, "nd_lit = %d <Fixnum>\n", FIX2LONG(lit));
	} else if (SYMBOL_P(lit)) {
		lp(level, "nd_lit = %s(%d) <Symbol>\n", ::rb_id2name(SYM2ID(lit)), SYM2ID(lit));
	} else if (TYPE(lit) == T_STRING) {
		lp(level, "nd_lit = \"%.*s\" <String>\n", RSTRING(lit)->len, RSTRING(lit)->ptr);
	} else {
		VALUE str = rb_inspect(lit);
		lp(level, "nd_lit = %.*s\n", RSTRING(str)->len, RSTRING(str)->ptr);
	}

/*
	if ( FIXNUM_P(lit) ) {
		p(" Fixnum: %d\n", FIX2LONG(lit));
	} else if ( SYMBOL_P(lit) ) {
		p(" Symbol: %d (%s)\n", SYM2ID(lit), rb_id2name(SYM2ID(lit)));
	} else if ( TYPE(lit) == T_STRING ) {
		char* pbeg = RSTRING(lit)->ptr;
		char* pend = pbeg + RSTRING(lit)->len;
		char* t    = tail(pbeg, pend);

		VALUE str = ::rb_inspect(::rb_str_substr(lit, 0, t - pbeg));
		p("%.*s", RSTRING(str)->len, RSTRING(str)->ptr);
		nl();

		char* s;
		while ( (s = t) < pend ) {
			t = tail(s, pend);
			str = rb_inspect(rb_str_substr(lit, s - pbeg, t - s));
			lp(level, "         %.*s", RSTRING(str)->len, RSTRING(str)->ptr);
			nl();
		}
	} else {
		VALUE str = ::rb_inspect(lit);
		p("%.*s", RSTRING(str)->len, RSTRING(str)->ptr);
		nl();
	}
*/
}

void
BRbDebug::dump_node(NODE* n, int level)
{
	NODE *n1 = NULL;
	int i   = 0;
	int id  = 0;
	int id1 = 0;

	if ( !n ) return;

	int type = nd_type(n);
	lp(level, "%s: ", NODE_NAMES[type]);

again:

	switch (type) {

	case NODE_ALIAS:
	case NODE_VALIAS:
		nl();
		lp(level + 1, "nd_new = %d (%s)\n", n->nd_new, ::rb_id2name(n->nd_new));
		lp(level + 1, "nd_old = %d (%s)\n", n->nd_old, ::rb_id2name(n->nd_old));
/*
		id = n->nd_old;
		id1 = n->nd_new;
		p(" from %d (%s) to %d (%s)\n", id, rb_id2name(id), id1, rb_id2name(id1));
*/
		break;

	case NODE_AND:
	case NODE_OR:
		nl();
		dump_node(n->nd_1st, level + 1);
		dump_node(n->nd_2nd, level + 1);
		break;

	case NODE_ARGS:
		p(" count = %d\n", n->nd_cnt);
		if ( n->nd_rest == -1 ) {
			lp(level+1, "additional default values:\n");
			n1 = n->nd_opt;
			dump_node(n1, level+2);
		}
		break;

	case NODE_ARGSCAT:
	case NODE_ARGSPUSH:
		nl();
		lp(level+1, "Push onto:\n");
		dump_node(n->nd_head, level+2);
		lp(level+1, "Values:\n");
		dump_node(n->nd_body, level+2);
		break;

	case NODE_ARRAY:
		p(" size = %d\n", n->nd_alen);
		n1 = n;
		while ( n1 ) {
			dump_node(n1->nd_head, level+1);
			n1 = n1->nd_next;
		}
		break;

	case NODE_ATTRSET:
		id = n->nd_vid;
		p(" %d (%s)\n", id, rb_id2name(id));
		break;

	case NODE_BACK_REF:
		p(" ($%c)\n", n->nd_nth);
		break;

	case NODE_BEGIN:
		nl();
		dump_node(n->nd_body, level+1);
		break;

	case NODE_BLOCK:
		nl();
		while ( n->nd_next ) {
			dump_node(n->nd_head, level+1);
			n = n->nd_next;
		}
		dump_node(n->nd_head,level+1);
		break;

	case NODE_BLOCK_ARG:
		p(" (to LV %d (%s))\n", n->nd_cnt, rb_id2name(n->nd_vid));
		break;

	case NODE_BLOCK_PASS:
		nl();
		dump_node(n->nd_body, level+1);
		dump_node(n->nd_iter, level+1);
		break;

	case NODE_BMETHOD:
		lp(level+1, "BMETHOD");
		break;

	case NODE_BREAK:
		nl();
		if ( n->nd_stts ) {
			dump_node(n->nd_stts, level+1);
		}
		break;

	case NODE_CALL:
	case NODE_FCALL:
		if ( type == NODE_FCALL ) {
			p(" to function: %d (%s)\n",  n->nd_mid, rb_id2name(n->nd_mid));
		} else {
			p(" to method: %d (%s)\n",  n->nd_mid, rb_id2name(n->nd_mid));
			lp(level, "Receiver:\n");
			dump_node(n->nd_recv, level+1);
		}
		n1 = n->nd_args;
		if ( n1 ) {
			lp(level+1, "Parameters:\n");
			dump_node(n1, level+2);
		}
		break;

	case NODE_CASE:
		nl();
		if ( n->nd_head ) {
			dump_node(n->nd_head, level+1);
		}
		n = n->nd_body;
		while ( n ) {
			if ( nd_type(n) != NODE_WHEN ) {
				lp(level+1, "Else:\n");
				dump_node(n, level+2);
				break;
			}
			dump_node(n->nd_head, level+1);
			dump_node(n->nd_body, level+3);
			n = n->nd_next;
		}
		break;

	case NODE_WHEN:
		nl();
		n1 = n->nd_head;
		while ( n1 ) {
			dump_node(n1->nd_head, level+2);
			n1 = n1->nd_next;
		}
		dump_node(n->nd_body, level+1);
		break;

#ifdef NODE_CASGN
	case NODE_CASGN:  // UNUSED?
		id = n->nd_vid;
		p(" to constant %d (%s)\n", id, rb_id2name(id));
		dump_node(n->nd_value, level+1);
		break;
#endif

	case NODE_CDECL:
		id = n->nd_vid;
		p(" to %d (%s)\n", id, rb_id2name(id));
		dump_node(n->nd_value, level+1);
		break;

	case NODE_CLASS:
		id = n->nd_cname;
		p(" class %d (%s)\n", id, rb_id2name(id));
		if (n->nd_super) {
			lp(level+1, "Superclass:\n");
			dump_node(n->nd_super, level+2);
		}
		dump_node(n->nd_body, level + 1);
		break;

	case NODE_COLON2:
		p(" ::%s in:\n", rb_id2name(n->nd_mid));
		dump_node(n->nd_head, level+1);
		break;

	case NODE_COLON3:
		p(" ::%s\n", rb_id2name(n->nd_mid));
		break;

	case NODE_CONST:
		p(" %d (%s)\n", n->nd_vid, rb_id2name(n->nd_vid));
		break;

	case NODE_CREF:
		nl();
		if (n->u1.node) {
//			dump_node(n->u1.node, level+1);
		}
		if (n->u2.node) {
			dump_node(n->u2.node, level+1);
		}
		if (n->u3.node) {
			dump_node(n->u3.node, level+1);
		}
		break;

	case NODE_CVAR:
#ifdef NODE_CVAR2
	case NODE_CVAR2:
#endif
		id = n->nd_vid;
		p(" class variable %d (%s)\n", id, rb_id2name(id));
		break;

	case NODE_CVDECL:
#ifdef NODE_CVASGN2		/* RUBY_VERSION_CODE < 162 */
	case NODE_CVASGN2:
#endif
#ifdef NODE_CVASGN		/* RUBY_VERSION_CODE < 160 */
	case NODE_CVASGN:
#endif
		id = n->nd_vid;
		p(" to class variable %d (%s)\n", id, rb_id2name(id));
		dump_node(n->nd_value, level+1);
		break;

	case NODE_DASGN:
	case NODE_DASGN_CURR:
		id = n->nd_vid;
		p(" to dynamic variable %d (%s)\n", id, rb_id2name(id));
		dump_node(n->nd_value, level+1);
		break;

	case NODE_DEFINED:
		nl();
		dump_node(n->nd_head, level+1);
		break;

	case NODE_DEFN:
		if ( n->nd_defn ) {
			p(" method %d (%s)\n", n->nd_mid, rb_id2name(n->nd_mid));
			dump_node(n->nd_defn, level+1);
		}
		break;

	case NODE_DEFS:
		if ( n->nd_defn ) {
			id = n->nd_mid;
			p(" method %d (%s)\n", id, rb_id2name(id));
			lp(level+1, "Receiver:\n");
			dump_node(n->nd_recv, level+2);
			dump_node(n->nd_defn, level+1);
		}
		break;

	case NODE_DMETHOD:
		lp(level+1, "BMETHOD");
		break;

	case NODE_DVAR:
		id = n->nd_vid;
		p(" dynamic variable %d (%s)\n", id, rb_id2name(id));
		break;

	case NODE_DSTR:
	case NODE_DXSTR:
	case NODE_DREGX:
	case NODE_DREGX_ONCE:
		dump_literal(n->nd_lit, level+1);
		n1 = n->nd_next;
		while ( n1 ) {
			dump_node(n1->nd_head, level+1);
			n1 = n1->nd_next;
		}
		break;

	case NODE_ENSURE:
		nl();
		dump_node(n->nd_head, level+1);
		if (n->nd_ensr) {
			lp(level+1, "Ensure:\n");
			dump_node(n->nd_ensr, level+2);
		}
		break;

	case NODE_FALSE:
		nl();
		break;

	case NODE_DOT2:
	case NODE_DOT3:
	case NODE_FLIP2:
	case NODE_FLIP3:
		nl();
		lp(level+1, "From:\n");
		dump_node(n->nd_beg, level+2);
		lp(level+1, "To:\n");
		dump_node(n->nd_end, level+2);
		break;

	case NODE_FBODY:
		p(" for: %d (%s)\n",  n->nd_mid, rb_id2name(n->nd_mid));
		dump_node(n->nd_head, level+1);
		break;

	case NODE_FOR:
	case NODE_ITER:
		nl();

		if (nd_type(n) == NODE_ITER) {
			lp(level+1, "Iterator:\n");
			dump_node(n->nd_iter, level+2);
		} else {
			lp(level+1, "Each on:\n");
			dump_node(n->nd_iter, level+2);
		}
		lp(level+1, "Into:\n");
		dump_node(n->nd_var, level+2);
		dump_node(n->nd_body, level+1);
		break;

	case NODE_GASGN:
		nl();
		dump_node(n->nd_value, level+1);
		id = n->nd_entry->id;
		lp(level+1, "Assign to GV %d (%s)\n", id, rb_id2name(id));
		break;

	case NODE_GVAR:
		id = n->nd_entry->id;
		lp(level+1, "GV %d (%s)\n", id, rb_id2name(id));
		break;

	case NODE_HASH:
		nl();
		n1 = n->nd_head;
		while (n1) {
			dump_node(n1->nd_head, level+1);
			n1 = n1->nd_next;
			if ( !n1 ) {
				break;
			}
			lp(level+2, "=>\n");
			dump_node(n1->nd_head, level+3);
			n1 = n1->nd_next;
		}
		break;

	case NODE_IASGN:
		nl();
		dump_node(n->nd_value, level+1);
		lp(level+1, "Assign to IV %d (%s)\n", n->nd_vid, rb_id2name(n->nd_vid));
		break;

	case NODE_IF:
		nl();
		dump_node(n->nd_cond, level+1);
		lp(level, "then:\n");
		dump_node(n->nd_body, level+1);
		lp(level, "else:\n");
		dump_node(n->nd_else, level+1);
		break;

	case NODE_IVAR:
		id = n->nd_vid;
		p(" instance variable %d (%s)\n", id, rb_id2name(id));
		break;

	case NODE_LASGN:
		nl();
		dump_node(n->nd_value, level+1);
		lp(level+1, "Assign to LV %d (%s)\n", n->nd_cnt, rb_id2name(n->nd_vid));
		break;

	case NODE_LIT:
	case NODE_EVSTR:
	case NODE_XSTR:
		dump_literal(n->nd_lit, level+1);
		break;

	case NODE_LVAR:
		p(" LV %d (%s)\n", n->nd_cnt, rb_id2name(n->nd_vid));
		break;

	case NODE_MASGN:
		nl();
		dump_node(n->nd_head, level+1);
		if ( n->nd_args ) {
			if ( n->nd_args == (NODE *)-1 ) {
				lp(level+1, "*");
			} else {
				dump_node(n->nd_args, level+2);
			}
		}
		dump_node(n->nd_value, level+1);
		break;

	case NODE_MATCH:
		dump_literal(n->nd_head->nd_lit, level+1);
		break;

	case NODE_MATCH3:
	case NODE_MATCH2:
		nl();
		dump_node(n->nd_recv, level+1);
		dump_node(n->nd_value, level+1);
		break;

	case NODE_METHOD:
		nl();
		break;

	case NODE_MODULE:
		nl();
		lp(level + 1, "nd_cname = %d (%s)\n", n->nd_cname, rb_id2name(n->nd_cname));
		lpdump(level + 1, "nd_body", n->nd_body);
/*
		id = n->nd_cname;
		p(" module %d (%s)\n", id, rb_id2name(id));
		dump_node(n->nd_body, level+1);
*/
		break;

	case NODE_NEWLINE:
		nl();
		lp(level + 1, "nd_file = \"%s\"\n", n->nd_file);
		lp(level + 1, "nd_nth  = %d\n", n->nd_nth);
		lpdump(level + 1, "nd_next", n->nd_next);
/*
		p(" [%s:%d]\n", n->nd_file, n->nd_nth);
		n1 = n->nd_next;
		dump_node(n1, level + 1);
*/
		break;

	case NODE_NEXT:
		nl();
		if ( n->nd_stts ) {
			dump_node(n->nd_stts, level+1);
		}
		break;

	case NODE_NIL:
		nl();
		break;

	case NODE_NOT:
		nl();
		dump_node(n->nd_body, level+1);
		break;

	case NODE_NTH_REF:
		p(" ($%d)\n", n->nd_nth);
		break;

	case NODE_OP_ASGN1:
		nl();
		dump_node(n->nd_recv, level+1);
		lp(level+1, "Indices:\n");
		dump_node(n->nd_args->nd_next, level+2);
		switch ( n->nd_mid ) {
		case 0:
			lp(level+1, "OR\n");
			break;
		case 1:
			lp(level+1, "AND\n");
			break;
		default:
			id = n->nd_mid;
			lp(level+1, "using: %d (%s)\n", id, rb_id2name(id));
		}
		dump_node(n->nd_args->nd_head, level+1);
		break;

	case NODE_OP_ASGN2:
		id = n->nd_next->nd_vid;
		p(" to method: %d (%s)\n",  id, rb_id2name(id));
		lp(level, "Receiver:\n");
		dump_node(n->nd_recv, level+1);
		switch (n->nd_next->nd_mid) {
		case 0:
			lp(level+1, "OR");
			break;
		case 1:
			lp(level+1, "AND");
			break;
		default:
			id = n->nd_next->nd_mid;
			lp(level+1, "using: %d (%s)\n", id, rb_id2name(id));
		}
		dump_node(n->nd_value, level+1);
		break;

	case NODE_OP_ASGN_AND:
	case NODE_OP_ASGN_OR:
		nl();
		lp(level+1, "First:\n");
		dump_node(n->nd_head, level+2);
		lp(level+1, "Second:\n");
		dump_node(n->nd_value, level+2);
		break;

	case NODE_OPT_N:
		nl();
		dump_node(n->nd_body, level+1);
		break;

	case NODE_POSTEXE:
		nl();
		break;

	case NODE_REDO:
		nl();
		break;

	case NODE_RESCUE:
		nl();
		dump_node(n->nd_head, level+1);
		n1 = n->nd_resq;
		while (n1) {
			lp(level+1, "When:\n");
			if ( !n1->nd_args ) {
				lp(level+2, "StandardError (default)\n");
			} else {
				dump_node(n1->nd_args, level+2);
			}

			dump_node(n1->nd_body, level+3);
			n1 = n1->nd_head;
		}
		if (n->nd_else) {
			lp(level+1, "Else:\n");
			dump_node(n->nd_else, level+2);
		}
		break;

	case NODE_RESTARGS:
#ifdef NODE_RESTARY
	case NODE_RESTARY:
#endif
#ifdef NODE_REXPAND
	case NODE_REXPAND:
#endif
		nl();
		dump_node(n->nd_head, level+1);
		break;

	case NODE_RETRY:
		nl();
		break;

	case NODE_RETURN:
		if (n->nd_stts) {
			nl();
			dump_node(n->nd_stts, level+1);
		}
		else
		p(" (nil)\n");
		break;

	case NODE_SCLASS:
		nl();
		lp(level+1, "Receiver:\n");
		dump_node(n->nd_recv, level+2);
		dump_node(n->nd_body->nd_body, level+1);
		break;

	case NODE_SCOPE:
		nl();
		dump_node(n->nd_next, level+1);
		break;

	case NODE_SELF:
		nl();
		break;

	case NODE_STR:
		putc(' ', m_out);
		dump_literal(n->nd_lit, level + 1);
		break;

	case NODE_SUPER:
		nl();
		dump_node(n->nd_args, level+1);
		break;

	case NODE_TRUE:
		nl();
		break;

	case NODE_UNDEF:
		id = n->nd_mid;
		p(" %d (%s)\n", id, rb_id2name(id));
		break;

	case NODE_UNTIL:
	case NODE_WHILE:
		nl();
		lp(level+1, "Condition:\n");
		dump_node(n->nd_cond, level+2);
		lp(level+1, "Body:\n");
		dump_node(n->nd_body, level+2);
		break;

	case NODE_VCALL:
		p(" self.%s\n", rb_id2name(n->nd_mid));
		break;

	case NODE_YIELD:
		nl();
		if ( n->nd_stts ) {
			dump_node(n->nd_stts, level+1);
		}
		break;

	case NODE_ZARRAY:
		p(" []\n");
		break;

	case NODE_ZSUPER:
		nl();
		break;

	default:
		p("\n\n** unhandled**\n\n");
		break;

	}
}

////////////////////////////////////////////////////////////////////////////////
