/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_execpath.c - execution path display program                           */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "bfd_if.h"
#include "bt.h"
#include <libgen.h>

#define BT_EXECPATH_VER	"0.0.1"
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2006"

#define	chk_next(argc, i)	if ((argc) <= (i)+1) { usage(); err_exit(); }

struct range_to_name {
	unsigned long	begin;
	unsigned long	end;
	unsigned long	offset;
	char		name[MAX_NAME_LEN];
	char		*dirname;
	char		*basename;
	struct bfd_if	bi;
};

/*----------------------------------------------------------------------------*/
/*  build range to name structure                                             */
/*----------------------------------------------------------------------------*/
struct range_to_name **all_r2n;
int r2n_max = 128;

void dump_r2n(void)
{
	int i;
	struct range_to_name *p;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		p = all_r2n[i];
		if (p)
			printf("0x%08lx:0x%08lx\t%s/%s\n",
			       p->begin, p->end, p->dirname, p->basename);
	}
}

void free_r2n(void)
{
	int i;
	struct range_to_name *p;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		p = all_r2n[i];
		if (p) {
			//free(p->func_names);
			free(p);
		}
	}
	free(all_r2n);
}

struct range_to_name* addr_to_r2n(unsigned long addr)
{
	int i;
	struct range_to_name *p;

	for (i = 0; i < r2n_max; i++) {
		p = all_r2n[i];
		if (p && addr >= p->begin && addr <= p->end)
			return p;
	}
	return NULL;
}

int prepare_obj_file(struct range_to_name *r2n, char *kallsyms)
{
	char path[MAX_LINE_LEN];

	snprintf(path, MAX_LINE_LEN, "%s/%s", r2n->dirname, r2n->basename);
	printf("checking %s...\n", path);
	if (init_bfd_if(&r2n->bi, strdup(path), kallsyms) < 0)
		return -1;
	if (!r2n->begin && !r2n->end) {
		struct addr_range *range = get_pid_addr_range(ALL_PID);
		bfd_vma min, max;
		struct bt_record rec;

		get_min_max_addr(&r2n->bi, &min, &max);
		rec.from = min;
		rec.to = max;
		if (is_addr_match(&rec, range)) {
			r2n->begin = min;
			r2n->end = max;
		} else
			return 1;
	} else {
		r2n->offset = get_offset_addr(&r2n->bi, r2n->begin);
		/* for DEBUG*/
		printf("OFFSET: ");
		printf_bfd_vma(r2n->offset);
		printf("\n");
		/**/
	}
	return 0;
}

int add_range_to_name(unsigned long begin, unsigned long end, char *name,
		      char *kallsyms, int is_module)
{
	int rc, i, step = 128;
	struct range_to_name *r2n;

	r2n = malloc(sizeof(struct range_to_name));
	if (!r2n) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	memset(r2n, 0, sizeof(struct range_to_name));
	r2n->begin = begin;
	r2n->end = end;
	snprintf(r2n->name, MAX_NAME_LEN, "%s", name);
	r2n->basename = basename(r2n->name);
	r2n->dirname = dirname(r2n->name);

	for (i = 0; i < r2n_max && all_r2n[i]; i++);
	if (i >= r2n_max) {
		all_r2n = realloc(all_r2n, (r2n_max + step) * sizeof(r2n));
		if (!all_r2n) {
			free(r2n);
			fprintf(stderr, "realloc failed.(%s)\n",
				strerror(errno));
			return -1;
		}
		memset(&all_r2n[i], 0, step * sizeof(r2n));
		i = r2n_max;
		r2n_max += step;
	}
	all_r2n[i] = r2n;
	rc = prepare_obj_file(r2n, kallsyms);
	if (rc < 0)
		return 0;
		//return -1;
	else if (rc > 0) {
		free(r2n);
		all_r2n[i] = NULL;
	}
	return 0;
}

/*----------------------------------------------------------------------------*/
/*  repeat checking                                                           */
/*----------------------------------------------------------------------------*/
#define MAX_REPEAT_LEN	8
static unsigned long addr_buf[MAX_REPEAT_LEN + 1];
static unsigned long *abp_top = addr_buf;
static unsigned long *abp_next = addr_buf;
#ifndef TEST
static unsigned long *abp_match;
static int repeat_cnt;
#endif

unsigned long* find_abuf(unsigned long addr)
{
	unsigned long *p;

	if (abp_top <= abp_next) {
		for (p = abp_top; p < abp_next; p++)
			if (*p == addr)
				return p;
	} else {
		for (p = abp_top;
		     p < addr_buf + MAX_REPEAT_LEN + 1; p++)
			if (*p == addr)
				return p;
		for (p = addr_buf; p < abp_next; p++)
			if (*p == addr)
				return p;
	}
	return NULL;
}

int get_repeat_checked_addrs(unsigned long *p, unsigned long *p_max,
			     unsigned long *p_dest)
{
	unsigned long *p_save = p_dest;

	if (p <= p_max) {
		for (; p < p_max; p++)
			*p_dest++ = *p;
	} else {
		for (; p < addr_buf + MAX_REPEAT_LEN + 1; p++)
			*p_dest++ = *p;
		for (p = addr_buf; p < p_max; p++)
			*p_dest++ = *p;
	}
	return (p_dest - p_save);
}

unsigned long* get_next_abuf_addr(unsigned long *p)
{
	return (p == addr_buf + MAX_REPEAT_LEN ? addr_buf : p + 1);
}

int proc_each_addr(unsigned long, int);

int flush_repeat_chk_buf(void)
{
	int i;

	for (i = 0; i < MAX_REPEAT_LEN; i++)
		if (proc_each_addr(0, 1) < 0)
			return -1;
	return 0;
}

#ifdef TEST
void chk_repeat(unsigned long addr, unsigned long *p, int *len, int *cnt,
		int is_flush)
{
	*p = addr;
	*len = 1;
	*cnt = 0;
}
#else
void chk_repeat(unsigned long addr, unsigned long *p, int *len, int *cnt,
		int is_flush)
{
	unsigned long *p_tmp = NULL;

	*len = 0;
	*cnt = 0;

	if (abp_match) {
		if (!is_flush && addr == *abp_match) {
			abp_match = get_next_abuf_addr(abp_match);
			if (abp_match == abp_next) {
				abp_match = abp_top;
				repeat_cnt++;
			}
		} else {
			if (repeat_cnt)
				*cnt = repeat_cnt + 1;
			repeat_cnt = 0;
			*len = get_repeat_checked_addrs(abp_top, abp_next, p);
			*abp_match = addr;
			abp_next = get_next_abuf_addr(abp_match);
			abp_match = NULL;
		}
	} else {
		if (!is_flush)
			p_tmp = find_abuf(addr);
		//printf("(F) %d\n", (p_tmp != NULL));
		if (p_tmp) {
			*len = get_repeat_checked_addrs(abp_top, p_tmp, p);
			abp_top = p_tmp;
			abp_match = get_next_abuf_addr(abp_top);
		} else {
			*abp_next = addr;
			abp_next = get_next_abuf_addr(abp_next);
			if (abp_top == abp_next) {
				abp_top = get_next_abuf_addr(abp_top);
				*p = *abp_next;
				*len = 1;
			}
		}
	}
	/* for debug */
	/*
	printf("(D)addr:%08lx ->top:%d, next:%d, match:%d, rcnt:%d\n",
	       addr, abp_top - addr_buf, abp_next - addr_buf,
	       abp_match ? abp_match - addr_buf : -1, repeat_cnt);
	       */
	/*
	{
		int i;
		printf("(B)                ");
		p = addr_buf;
		for (i = 0; i < MAX_REPEAT_LEN + 1; i++)
			printf("%08lx,", *p++);
		printf("\n");
	}
	*/
}
#endif

/*----------------------------------------------------------------------------*/
/*  get function name and mnemonic                                            */
/*----------------------------------------------------------------------------*/
int print_exec_line(unsigned long addr, struct range_to_name *r2n)
{
	int rc, lno;
	size_t offset;
	const char *src_name, *func_name;
	struct bfd_if *bi = &r2n->bi;
	bfd_vma baddr;

	addr -= r2n->offset;
	printf("%s\t0x%08lx\t", r2n->basename, addr);
	rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
	if (rc >= 0 && src_name) {
		addr_to_func_name_and_offset(bi, addr, &func_name, &offset);
		printf_func_name(func_name, offset);
		if (src_name)
			printf("\t%s:%d", src_name, lno);
		printf("\n");
	} else {
		if (!addr_to_func_name_and_offset(bi, addr, &func_name,
						  &offset)) {
			printf_func_name(func_name, offset);
			printf("\t");
		}
		rc = printf_mnemonic(&r2n->bi, addr, &baddr);
		//printf("\nBADDR: 0x%08lx\n", baddr);
		if (rc > 0 &&
		    !addr_to_func_name_and_offset(bi, baddr, &func_name,
						  &offset)) {
			printf(" ");
			printf_func_name(func_name, offset);
		}
		printf("\n");
	}
	return 0;
}

/*----------------------------------------------------------------------------*/
/*  program core                                                              */
/*----------------------------------------------------------------------------*/
int proc_each_addr(unsigned long addr, int is_flush)
{
	struct range_to_name *r2n;
	static unsigned long last_addr = 0;
	unsigned long abuf[MAX_REPEAT_LEN], *p;
	int i, len, cnt, indent;

	if (!is_flush && addr == last_addr)
		return 0;
	last_addr = addr;

	chk_repeat(addr, abuf, &len, &cnt, is_flush);
	if (cnt) {
		printf("===> repeat %d times\n", cnt);
		indent = 1;
	} else {
		indent = 0;
	}
	for (p = abuf; p < abuf + len; p++) {
		r2n = addr_to_r2n(*p);
		if (indent) {
			for (i = 0; i < indent; i++)
				printf("  ");
		}
		if (!r2n) {
			printf("--------\t0x%08lx\n", *p);
			continue;
		}
		if (print_exec_line(*p, r2n) < 0)
			return -1;
	}
	return 0;
}

int proc_each_record(struct bt_record *p, struct addr_range *range)
{
	if (is_pid_record(p) || !is_addr_match(p, range))
		return 0;

	if (proc_each_addr(p->from, 0) < 0)
		return -1;
	if (proc_each_addr(p->to, 0) < 0)
		return -1;
	return 0;
}

int proc_logfile(char *logfile)
{
	int fd, rc = -1;
	size_t size;
	struct bt_record *p, *p_max;
	char *dir, *func_name;
	struct addr_range *range;

	if (open_mmapfile(logfile, &fd, (void**)&p, &size) < 0)
		goto EXIT;

	func_name = basename(logfile);
	dir = dirname(logfile);
	if (is_pid_record(p)) {
		if (parse_maps(add_range_to_name, dir, func_name) < 0)
			goto EXIT;
	} else {
		if (parse_modules(add_range_to_name, dir) < 0)
			goto EXIT;
	}
	printf("start\n");
	range = get_pid_addr_range(ALL_PID);
	for (p_max = (struct bt_record*)((char*)p + size); p < p_max; p++) {
		if (proc_each_record(p, range) < 0)
			goto EXIT;
	}
	if (flush_repeat_chk_buf() < 0)
		goto EXIT;
	rc = 0;
EXIT:
	close_mmapfile(fd, p, size);
	return rc;
}

void err_exit(void)
{
	free_ranges();
	free_r2n();
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_execpath %s\n", BT_EXECPATH_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_execpath [-a top:end [...]] [-d top:end [...]] -f logfile\n");
	fprintf(stderr, "  -a: add address range\n");
	fprintf(stderr, "  -d: delete address range\n");
	fprintf(stderr, "  -f: logfile\n");
}

int main(int argc, char *argv[])
{
	int i;
	char *logfile = NULL;
	unsigned long begin, end;

	if (alloc_pid_range(ALL_PID) < 0)
		err_exit();
	for (i = 1; i < argc;) {
		if (strcmp(argv[i], "-a") == 0) {
			chk_next(argc, i);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				add_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-d") == 0) {
			chk_next(argc, i);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				printf("del_range\n");
				del_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-f") == 0) {
			chk_next(argc, i);
			i++;
			logfile = argv[i];
			i++;
		} else {
			usage();
			err_exit();
		}
	}
	//dump_ranges();	/* for debug */
	if (!logfile) {
		usage();
		err_exit();
	}
	all_r2n = calloc(r2n_max, sizeof(struct range_to_name*));
	if (!all_r2n) {
		fprintf(stderr, "calloc failed.(%s)\n", strerror(errno));
		err_exit();
	}
	if (proc_logfile(logfile) < 0)
		err_exit();
	//dump_r2n();	/* for debug */
	free_ranges();
	free_r2n();
	exit(0);
}
