/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_split.c - log split by each pid program                               */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#define CONFIG_SMP	/* for NR_CPUS */
#include <linux/threads.h>
#include "bt.h"

#define BT_SPLIT_VER	"0.0.1"
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2006"

#define	chk_next(argc, i)	if ((argc) <= (i)+1) { usage(); err_exit(); }

struct __pid_info {
	int			cpu;
	unsigned long		tv_sec;
	unsigned long		tv_usec;
	off_t			offset;
	struct __pid_info	*next;
};

struct pid_log_info {
	pid_t			pid;
	struct __pid_info	*info;
};

struct cpu_file_info {
	int			fd;
	void			*p;
	size_t			size;
};

static struct cpu_file_info cpu_file_info[NR_CPUS];
static struct pid_log_info **pid_log_info;
static size_t pid_max = 1024;

void dump_pid_pos(void)
{
	size_t i;
	struct pid_log_info *p;
	struct __pid_info *__p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		printf("pid:%d------>\n", p->pid);
		for (__p = p->info; __p; __p = __p->next)
			printf("   cpu:%d, sec:%ld, usec:%ld\n",
			       __p->cpu, __p->tv_sec, __p->tv_usec);
	}
}

void free_pid_info(void)
{
	size_t i;
	struct pid_log_info *p;
	struct __pid_info *__p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		for (__p = p->info; __p; __p = __p->next)
			free(__p);
		free(p);
		pid_log_info[i] = NULL;
	}
	free(pid_log_info);
	pid_log_info = NULL;
}

void cpu_file_close(void)
{
	int i;
	struct cpu_file_info *cpu_info;

	for (i = 0; i < NR_CPUS; i++) {
		cpu_info = &cpu_file_info[i];
		if (cpu_info->p)
			munmap(cpu_info->p, cpu_info->size);
		if (cpu_info->fd)
			close(cpu_info->fd);
	}
}

struct pid_log_info* find_pid_info_index(pid_t pid)
{
	size_t i, step = 1024;
	struct pid_log_info *p;

	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			goto ALLOC;
		if (p->pid == pid)
			return p;
	}
	/* enlarge pid_log_info area */
	pid_log_info = realloc(pid_log_info,
			       (pid_max + step) * sizeof(struct pid_log_info*));
	if (!pid_log_info) {
		fprintf(stderr, "realloc failed.(%s)\n", strerror(errno));
		return NULL;
	}
	for (; i < pid_max + step; i++)
		pid_log_info[i] = NULL;
	i = pid_max;
	pid_max += step;
ALLOC:
	p = malloc(sizeof(struct pid_log_info));
	if (!p) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return NULL;
	}
	p->pid = pid;
	p->info = NULL;
	pid_log_info[i] = p;
	return p;
}

int compare_timestamp(struct __pid_info *i1, struct __pid_info *i2)
{
	if (i1->tv_sec < i2->tv_sec)
		return 1;
	else if (i1->tv_sec > i2->tv_sec)
		return -1;
	else {
		if (i1->tv_usec < i2->tv_usec)
			return 1;
		else if (i1->tv_usec > i2->tv_usec)
			return -1;
	}
	return 0;
}

int add_cpu_pid_info(int cpu, struct pid_record *rec, off_t offset)
{
	struct pid_log_info *p;
	struct __pid_info *__p, *__p_dest, *tmp;
	int rc;

	/* get pid-info */
	p = find_pid_info_index(rec->pid);
	if (!p)
		return -1;

	/* setup pid-sub-info for add*/
	__p = malloc(sizeof(struct __pid_info));
	if (!__p) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	__p->cpu = cpu;
	__p->tv_sec = rec->tv_sec;
	__p->tv_usec = rec->tv_usec & ~BT_FLAG_PID;
	__p->offset = offset;
	__p->next = NULL;

	/* find add index (sort by timestamp) */
	__p_dest = NULL;
	for (tmp = p->info; tmp; tmp = tmp->next) {
		rc = compare_timestamp(tmp, __p);
		if (rc < 0)
			break;
		__p_dest = tmp;
	}

	/* add pid-sub-info */
	if (__p_dest) {
		__p->next = __p_dest->next;
		__p_dest->next = __p;
	} else {
		__p->next = p->info;
		p->info = __p;
	}
	return 0;
}

int chk_pid_pos_per_cpu(int cpu,
			struct bt_record *p_top, struct bt_record *p_max)
{
	struct bt_record *p;

	for (p = p_top; p < p_max; p++) {
		if (is_pid_record(p)) {
			if (add_cpu_pid_info(cpu, (struct pid_record*)p,
					     p - p_top) < 0)
				return -1;
		}
	}
	return 0;
}

int chk_pid_pos(char *dir_path)
{
	DIR *dir;
	struct dirent* d;
	int max = 128;
	char path[max];
	struct stat st;
	int rc, cpu;
	struct bt_record *p, *p_max;
	struct cpu_file_info *cpu_info;

	if ((dir = opendir(dir_path)) == NULL) {
		fprintf(stderr, "can't open %s\n", dir_path);
		return -1;
	}
	rc = 0;
	while ((d = readdir(dir)) != NULL) {
		if (strcmp(".", d->d_name) == 0 || strcmp("..", d->d_name) == 0)
			continue;
		snprintf(path, max, "%s/%s", dir_path, d->d_name);
		if (lstat(path, &st) < 0) {
			fprintf(stderr, "%s lstat failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		if (S_ISDIR(st.st_mode) ||
		    sscanf(d->d_name, "cpu%d", &cpu) != 1)
			continue;
		cpu_info = &cpu_file_info[cpu];
		if ((cpu_info->fd = open(path, O_RDONLY)) < 0) {
			fprintf(stderr, "%s open failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		p = mmap(NULL, st.st_size, PROT_READ, MAP_PRIVATE,
			 cpu_info->fd, 0);
		if (p == MAP_FAILED) {
			fprintf(stderr, "mmap failed.(%s)\n", strerror(errno));
			rc = -1;
			break;
		}
		cpu_info->p = p;
		cpu_info->size = st.st_size;
		p_max = (struct bt_record*)((char*)p + st.st_size);
		rc = chk_pid_pos_per_cpu(cpu, p, p_max);
		if (rc < 0)
			break;
	}
	closedir(dir);
	if (rc < 0)
		return rc;
	return 0;
}

int create_pid_files(char *dir_path)
{
	size_t i;
	struct pid_log_info *p;
	struct __pid_info *__p;
	struct cpu_file_info *cpu_info;
	struct bt_record *start, *end, *max;
	int fd;
	int path_max = 128;
	char path[path_max];

	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		snprintf(path, path_max, "%s/%d", dir_path, p->pid);
		if ((fd = open(path, O_CREAT|O_EXCL|O_RDWR, S_IRUSR|S_IWUSR))
		    < 0) {
			fprintf(stderr, "can't create %d.(%s)\n", p->pid,
				strerror(errno));
			return -1;
		}
		for (__p = p->info; __p; __p = __p->next) {
			cpu_info = &cpu_file_info[__p->cpu];
			start = (struct bt_record*)cpu_info->p + __p->offset;
			max = (struct bt_record*)(cpu_info->p + cpu_info->size);
			for (end = start + 1; end < max && !is_pid_record(end);
			     end++);
			write(fd, start, (char*)end - (char*)start);
		}
		close(fd);
	}
	return 0;
}

void err_exit(void)
{
	free_pid_info();
	cpu_file_close();
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_split %s\n", BT_SPLIT_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_split -d dir\n");
}

int main(int argc, char *argv[])
{
	int i;
	char *dir;

	for (i = 1; i < argc;) {
		if (strcmp(argv[i], "-d") == 0) {
			chk_next(argc, i);
			i++;
			dir = argv[i];
			i++;
		} else {
			usage();
			err_exit();
		}
	}
	if (!dir) {
		usage();
		err_exit();
	}
	if ((pid_log_info = calloc(pid_max, sizeof(struct pid_log_info*)))
	    == NULL)
		err_exit();

	if (chk_pid_pos(dir) < 0)
		err_exit();
	if (create_pid_files(dir) < 0)
		err_exit();

	//dump_pid_pos();	/* for debug */
	free_pid_info();
	cpu_file_close();

	exit(0);
}
