/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bfd_if.c - BFD library interface                                         */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <elf.h>
#include "bt.h"
#include "bfd_if.h"

static int last_error;

/*-----------------------------------------------------------------------------
 *  intialize
 *-----------------------------------------------------------------------------
 */
#if 0
static void dump_symbols(const char *header, asymbol **symbols, long cnt)
{
	long i;
	asymbol *sym;
	asection *sect;

	printf("%s\n", header);
	for (i = 0; i < cnt; i++) {
		sym = symbols[i];
		sect = sym->section;
		printf("\tSYM ");
		printf_bfd_vma(bfd_asymbol_value(sym));
		printf(" <%s>%08x\t<%s>%08x:%d\n",
		       sym->name, sym->flags, sect->name, sect->flags,
		       sect->index);
	}
}
static void dump_relocs(struct bfd_if *bi)
{
	long i;
	for (i = 0; i < bi->n_relocs; i++) {
		printf("RELOC=> CODE-OFFSET:");
		printf_bfd_vma(bi->p_relocs[i]->address);
		printf(" SYM-ADDR:");
		printf_bfd_vma((*bi->p_relocs[i]->sym_ptr_ptr)->value);
		printf(" <%s>", (*bi->p_relocs[i]->sym_ptr_ptr)->name);
		printf("\n");
	}
}
#endif

static int chk_elf_header(struct bfd_if *bi, const char *obj_name)
{
	int rc;
	unsigned int i;
	FILE *f;
	unsigned char e_ident[EI_NIDENT];
	Elf32_Ehdr ehdr;
	Elf32_Phdr phdr;

	rc = -1;
	f = fopen(obj_name, "rb");
	if (!f) {
		fprintf(stderr, "%s file can't open.(%s)\n",
			obj_name, strerror(errno));
		return rc;
	}
	if (fread(e_ident, EI_NIDENT, 1, f) != 1) {
		fprintf(stderr, "fread failed.(%s)\n", strerror(errno));
		goto EXIT;
	}
	if (e_ident[EI_DATA] != ELFDATA2LSB) {
		fprintf(stderr, "not little endian object.\n");
		goto EXIT;
	}
	if (e_ident[EI_CLASS] != ELFCLASS32) {
		fprintf(stderr, "not 32-bit architecture object.\n");
		goto EXIT;
	}
	if (fread(&ehdr.e_type, sizeof(ehdr) - EI_NIDENT, 1, f) != 1) {
		fprintf(stderr, "fread failed.(%s)\n", strerror(errno));
		goto EXIT;
	}
	for (i = 0; i < ehdr.e_phnum; i++) {
		if (fread(&phdr, sizeof(phdr), 1, f) != 1) {
			fprintf(stderr, "fread failed.(%s)\n", strerror(errno));
			goto EXIT;
		}
		if (phdr.p_type != PT_LOAD || !(phdr.p_flags & PF_X))
			continue;
		bi->load_vaddr = phdr.p_vaddr;
		break;
	}
	rc = 0;
EXIT:
	if (f)
		fclose(f);
	return rc;
}

static asection* get_section_has_index(bfd *abfd, int index)
{
	asection *s;

	for (s = abfd->sections; s; s = s->next)
		if (s->index == index)
			return s;
	return NULL;
}

static int is_index_valid_sect(bfd *abfd, int *p, int n, int index)
{
	asection *s, *target_sect;

	target_sect = get_section_has_index(abfd, index);
	for (; n > 0; n--) {
		s = get_section_has_index(abfd, *p++);
		if (s->vma == target_sect->vma)
			return 0;
	}
	return 1;
}

static int chk_valid_code_sects(struct bfd_if *bi)
{
	int *top, *p;
	asection *s;


	top = calloc(bi->abfd->section_count, sizeof(int));
	if (!top) {
		fprintf(stderr, "calloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	p = top;
	for (s = bi->abfd->sections; s; s = s->next) {
		if (!(s->flags & SEC_CODE) || !(s->flags & SEC_ALLOC))
			continue;
		if (is_index_valid_sect(bi->abfd, p, p - top, s->index))
			*p++ = s->index;
	}
	bi->p_code_sects = top;
	bi->n_code_sects = p - top;
	/* for DEBUG
	printf("VALID CODE SECTS:");
	for (p = bi->p_code_sects, top = p + bi->n_code_sects; p < top; p++)
		printf(" %d,", *p);
	}
	printf("\n");
	*/
	return 0;
}

static int is_code_sect(struct bfd_if *bi, asection *sect)
{
	int *p, *p_max;

	for (p = bi->p_code_sects, p_max = p + bi->n_code_sects; p < p_max; p++)
		if (sect->index == *p)
			return 1;
	return 0;
}

static int read_all_symbols(bfd *abfd, int is_dynamic,
			    asymbol ***symbols, long *nsyms)
{
	long size;

	if (!(bfd_get_file_flags(abfd) & HAS_SYMS))
		return 0;

	if (is_dynamic)
		size = bfd_get_dynamic_symtab_upper_bound(abfd);
	else
		size = bfd_get_symtab_upper_bound(abfd);
	//printf("HERE: %d, %ld(%d)\n", is_dynamic, size, sizeof(asymbol*));

	if (size <= (long)sizeof(asymbol*)) {
		/*
		fprintf(stderr, "%s failed.\n",
			is_dynamic ? "bfd_get_dynamic_symtab_upper_bound" :
				     "bfd_get_symtab_upper_bound");
		return -1;
		*/
		return 0;
	}
	*symbols = malloc(size);
	if (!*symbols) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	if (is_dynamic)
		*nsyms = bfd_canonicalize_dynamic_symtab(abfd, *symbols);
	else
		*nsyms = bfd_canonicalize_symtab(abfd, *symbols);
	return 0;
}

static int __compare_symbols(const void *d1, const void *d2)
{
	asymbol *sym1 = *(asymbol**)d1, *sym2 = *(asymbol**)d2;
	bfd_vma a1, a2;

	a1 = bfd_asymbol_value(sym1);
	a2 = bfd_asymbol_value(sym2);
	if (a1 > a2)
		return 1;
	if (a1 < a2)
		return -1;
	return strcmp(sym1->name, sym2->name);
}

static long remove_useless_symbols(struct bfd_if *bi, asymbol **syms,
				   long count)
{
	asymbol **p_in = syms, **p_out = syms, *sym_prev = NULL;
	asection *sect;

	while (--count >= 0) {
		asymbol *sym = *p_in++;
		if (sym->name == NULL || sym->name[0] == '\0')
			continue;
		if (sym->flags & (BSF_DEBUGGING))
			continue;
		sect = sym->section;
		/*
		if (bfd_is_und_section(sym->section)
		    || bfd_is_com_section(sym->section))
			continue;
			*/
		if (bfd_is_com_section(sect))
			continue;
		if (!is_code_sect(bi, sect) && !bfd_is_und_section(sect))
			continue;
		*p_out++ = sym_prev = sym;
	}
	return p_out - syms;
}

static int __resolve_undsym(struct bfd_if *bi, char *sname, unsigned long addr)
{
	long i;
	asymbol *sym;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = bi->p_fsyms[i];
		if (sym->name[0] != sname[0] || strcmp(sym->name, sname) != 0)
			continue;
		sym->value = addr;
		return 0;
	}
	//fprintf(stdout, "WARN: symbol(%s:0x%08lx) not found.\n", sname, addr);
	return 0;
}

static int resolve_undsyms(struct bfd_if *bi, const char *obj_name,
			   const char *kallsyms)
{
	int rc, len, num;
	FILE *f = NULL;
	char *mod_name = strdup(obj_name);
	char buf[MAX_LINE_LEN + 1], sym[MAX_LINE_LEN], mod[MAX_LINE_LEN], *p;
	unsigned long addr;

	if (!kallsyms)
		return 0;
	rc = -1;
	mod_name = basename(mod_name);
	len = strlen(mod_name);
	if (len <= 3 || strcmp(&mod_name[len - 3], ".ko") != 0)
		return 0;	/* target object is not kernel module */
	mod_name[len - 3] = '\0';
	f = fopen(kallsyms, "r");
	if (!f) {
		fprintf(stderr, "%s open failed.\n", kallsyms);
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = '\0';
	while ((p = fgets(buf, MAX_LINE_LEN, f)) != NULL) {
		len = strlen(buf);
		if (buf[len - 2] != ']' || buf[9] != 'U')
			continue;
		num = sscanf(buf, "%lx %*c %s\t[%s]", &addr, sym, mod);
		if (num != 3 || strncmp(mod_name, mod, strlen(mod_name)) != 0)
			continue;
		__resolve_undsym(bi, sym, addr);
	}
	rc = 0;
EXIT:
	if (f)
		fclose(f);
	return rc;
}

static void chk_relocs_sect(bfd *abfd, asection *sect, void *obj)
{
	struct bfd_if *bi = (struct bfd_if*)obj;
	long size, n;

	if (!(is_code_sect(bi, sect) && (sect->flags & SEC_RELOC)))
		return;
	size = bfd_get_reloc_upper_bound(abfd, sect);
	if (size < 0 || size == sizeof(arelent*))
		return;
	n = size / sizeof(arelent*);
	if (bi->n_relocs)
		bi->p_relocs = realloc(bi->p_relocs,
				       (bi->n_relocs * sizeof(arelent*)) +size);
	else
		bi->p_relocs = malloc(size);
	if (!bi->p_relocs) {
		bi->n_relocs = 0;
		fprintf(stderr, "memory alloc failed.(%s)\n", strerror(errno));
		return;
	}
	memset(&bi->p_relocs[bi->n_relocs], 0, size);

	bfd_canonicalize_reloc(abfd, sect,
			       &bi->p_relocs[bi->n_relocs], bi->p_syms);
	bi->n_relocs += n - 1;
}

static int chk_relocs(struct bfd_if *bi)
{
	if (!(bfd_get_file_flags(bi->abfd) & HAS_RELOC))
		return 0;
	bfd_map_over_sections(bi->abfd, chk_relocs_sect, bi);
	return 0;
}

static int build_sorted_func_symbols(struct bfd_if *bi, const char *obj_name,
				     const char *kallsyms)
{
	long i;

	bi->n_syms = bi->n_dynsyms = bi->n_synsyms = 0;
	bi->p_syms = bi->p_dynsyms = NULL;
	bi->p_synsyms = NULL;

	if (read_all_symbols(bi->abfd, 0, &bi->p_syms, &bi->n_syms) < 0)
		goto EXIT;
	//dump_symbols("syms", bi->p_syms, bi->n_syms);
	if (read_all_symbols(bi->abfd, 1, &bi->p_dynsyms, &bi->n_dynsyms) < 0)
		goto EXIT;
	//dump_symbols("dynsyms", bi->p_dynsyms, bi->n_dynsyms);
	chk_relocs(bi);
	bi->n_synsyms = bfd_get_synthetic_symtab(bi->abfd,
						 bi->n_syms, bi->p_syms,
						 bi->n_dynsyms, bi->p_dynsyms,
						 &bi->p_synsyms);
	if (bi->n_synsyms < 0)
		goto EXIT;
	bi->n_fsyms = bi->n_syms ? bi->n_syms : bi->n_dynsyms;
	bi->p_fsyms = xmalloc((bi->n_fsyms + bi->n_synsyms) * sizeof(asymbol*));
	memcpy(bi->p_fsyms, bi->n_syms ? bi->p_syms : bi->p_dynsyms,
	       bi->n_fsyms * sizeof(asymbol*));
	bi->n_fsyms = remove_useless_symbols(bi, bi->p_fsyms, bi->n_fsyms);

	for (i = 0; i < bi->n_synsyms; i++) {
		bi->p_fsyms[bi->n_fsyms] = bi->p_synsyms + i;
		bi->n_fsyms++;
	}
	if (resolve_undsyms(bi, obj_name, kallsyms) < 0)
		goto EXIT;
	//dump_relocs(bi);
	qsort(bi->p_fsyms, bi->n_fsyms, sizeof(asymbol*), __compare_symbols);
	//dump_symbols("sorted-fsyms", bi->p_fsyms, bi->n_fsyms);
	return 0;
EXIT:
	if (bi->p_syms)
		free(bi->p_syms);
	if (bi->p_dynsyms)
		free(bi->p_dynsyms);
	if (bi->p_synsyms)
		free(bi->p_synsyms);
	if (bi->p_fsyms)
		free(bi->p_fsyms);
	return -1;
}

/* We check all section because HAS_DEBUG never set in abfd->flags */
static void chk_has_debuginfo(struct bfd_if *bi)
{
	asection *s;

	for (s = bi->abfd->sections; s; s = s->next)
		if (s->flags & SEC_DEBUGGING) {
			bi->has_debuginfo = 1;
			break;
		}
}

int init_bfd_if(struct bfd_if *bi, const char *obj_name, const char *kallsyms)
{
	struct disassemble_info *info = &bi->info;

	memset(bi, 0, sizeof(struct bfd_if));

	if (chk_elf_header(bi, obj_name) < 0)
		return -1;

	bi->abfd = bfd_openr(obj_name, NULL);
	if (!bi->abfd) {
		fprintf(stderr, "%s bfd_openr failed.\n", obj_name);
		return -1;
	}
	if (!bfd_check_format(bi->abfd, bfd_object)) {
		fprintf(stderr, "%s bfd_check_format failed.\n", obj_name);
		return -1;
	}
	chk_valid_code_sects(bi);
	if (build_sorted_func_symbols(bi, obj_name, kallsyms) < 0)
		return -1;

	init_disassemble_info(info, stdout, (fprintf_ftype)fprintf);
	info->flavour = bfd_get_flavour(bi->abfd);
	info->arch = bfd_get_arch(bi->abfd);
	info->mach = bfd_get_mach(bi->abfd);
	info->disassembler_options = NULL;
	info->octets_per_byte = bfd_octets_per_byte(bi->abfd);

	if (bfd_big_endian(bi->abfd))
		info->display_endian = info->endian = BFD_ENDIAN_BIG;
	else if (bfd_little_endian(bi->abfd))
		info->display_endian = info->endian = BFD_ENDIAN_LITTLE;
	else
		info->endian = BFD_ENDIAN_UNKNOWN;
	disassemble_init_for_target(info);
	chk_has_debuginfo(bi);
	return 0;
}

bfd_vma get_offset_addr(struct bfd_if *bi, bfd_vma begin)
{
	if (bfd_get_file_flags(bi->abfd) & HAS_RELOC)
		return begin;
	return begin - bi->load_vaddr;
}

/*-----------------------------------------------------------------------------
 *  misc function
 *-----------------------------------------------------------------------------
 */
void get_min_max_addr(struct bfd_if *bi, bfd_vma *min, bfd_vma *max)
{
	bfd_vma sect_max;
	asection *s;

	*min = 0xffffffff;
	*max = 0;
	for (s = bi->abfd->sections; s; s = s->next) {
		if (!is_code_sect(bi, s))
			return;
		if (s->vma < *min)
			*min = s->vma;
		sect_max = s->vma + bfd_get_section_size(s) - 1;
		if (sect_max > *max)
			*max = sect_max;
	}
	return;
}

static asection* get_sect_has_addr(struct bfd_if *bi, bfd_vma addr)
{
	asection *s;

	for (s = bi->abfd->sections; s; s = s->next) {
		if (is_code_sect(bi, s) &&
		    addr >= s->vma &&
		    addr < s->vma + bfd_get_section_size(s))
			return s;
	}
	return NULL;
}

static int prepare_print_insn(struct bfd_if *bi, bfd_vma addr)
{
	asection *sect;
	struct disassemble_info *info = &bi->info;
	struct sect_cache *cache = &bi->cache;

	if (cache->data) {
		if (addr >= cache->vma && addr < cache->vma + cache->size) {
			goto CACHE_FINISH;
		} else {
			free(cache->data);
			cache->data = NULL;
		}
	}
	sect = get_sect_has_addr(bi, addr);
	if (!sect)
		return -1;
		//last_error |= CHK_BTYPE_E_NOT_FOUND;

	cache->sect = sect;
	cache->vma = sect->vma;
	cache->size = bfd_get_section_size(sect);
	if (cache->size == 0)
		return 0;
	cache->data = xmalloc(cache->size);
	bfd_get_section_contents(bi->abfd, sect, cache->data, 0, cache->size);

CACHE_FINISH:
	info->buffer = cache->data;
	info->buffer_vma = cache->vma;
	info->buffer_length = cache->size;
	info->section = cache->sect;
	//info->symbols = NULL;
	//info->num_symbols = 0;
	info->insn_info_valid = 0;
	info->bytes_per_line = 0;
	info->bytes_per_chunk = 0;
	info->flags = 0;
	return 0;
}

int get_source_info(struct bfd_if *bi, bfd_vma addr,
		    const char **src_name, const char **func_name, int *lno)
{
	int rc;
	asection *sect;

#ifdef TEST
	return -1;
#endif
	if (!bi->has_debuginfo)
		return -1;
	sect = get_sect_has_addr(bi, addr);
	if (!sect || !is_code_sect(bi, sect))
		return -1;
	rc = bfd_find_nearest_line(bi->abfd, sect,
				   bi->n_syms ? bi->p_syms : bi->p_dynsyms,
				   addr - sect->vma, src_name, func_name, lno);
	return (rc ? 0 : -1);
}

/* This function nearly equal bsearch(3).
 * Except, this function also returns small index if not match found.
 */
static void* __bsearch(const void *key, const void *base, size_t nmemb,
		       size_t size, int (*comp)(const void*, const void*),
		       size_t *small_index)
{
	size_t min, max, mid;
	const void *p;
	int rc;

	min = 0;
	max = nmemb;
	while (min < max) {
		mid = (min + max) / 2;
		p = (void *)(((const char *)base) + (mid * size));
		rc = (*comp)(key, p);
		if (rc < 0)
			max = mid;
		else if (rc > 0)
			min = mid + 1;
		else {
			if (small_index)
				*small_index = mid;
			return (void*)p;
		}
	}
	if (small_index)
		*small_index = rc > 0 ? mid : mid - 1;
	return NULL;
}

static int __comp_fsym_addr(const void *key, const void *elem)
{
	bfd_vma addr = *(bfd_vma*)key;
	asymbol *sym = *(asymbol**)elem;
	bfd_vma sym_addr = bfd_asymbol_value(sym);

	if (addr > sym_addr)
		return 1;
	else if (addr < sym_addr)
		return -1;
	return 0;
}

int addr_to_func_name_and_offset(struct bfd_if *bi, bfd_vma addr,
				 const char **func_name, size_t *offset)
{
	long i;
	asymbol *sym, **p_sym;
	size_t sym_index;
	asection *sect;
	arelent *reloc;

	if (addr == UNKNOWN_BADDR)
		goto NOT_FOUND;
	if (bi->p_relocs) {
		for (i = 0; i < bi->n_relocs; i++) {
			reloc = bi->p_relocs[i];
			if (reloc->address == addr) {
				*func_name = (*reloc->sym_ptr_ptr)->name;
				*offset = 0;
				/* for DEBUG
				printf("[FOUND RELOC]", *func_name);
				*/
				return 0;
			}
		}
	}
	p_sym = __bsearch(&addr, bi->p_fsyms, bi->n_fsyms, sizeof(asymbol*),
			  __comp_fsym_addr, &sym_index);
	if (!p_sym) {
		if (sym_index < 0)
			goto NOT_FOUND;
		sym = bi->p_fsyms[sym_index];
	} else
		sym = *p_sym;
	sect = sym->section;
	/*
	if (sect->vma > addr || sect->vma + sect->size <= addr)
		goto NOT_FOUND;
		*/
	*func_name = sym->name;
	*offset = addr - bfd_asymbol_value(sym);
	return 0;

NOT_FOUND:
	*func_name = "";
	*offset = 0;
	return -1;
}

#define PRINT_BUF_MAX	128
static char print_buf[PRINT_BUF_MAX + 1];
static int seq_num;
static int offset;
static int baddr_offset;

static int dummy_sprintf(FILE *stream, const char *fmt, ...)
{
	va_list argp;
	int rc;

	va_start(argp, fmt);
	if (seq_num == 1)
		baddr_offset = offset;
	rc = vsnprintf(&print_buf[offset], PRINT_BUF_MAX - offset, fmt, argp);
	if (rc < 0)
		return -1;
	offset += rc;
	seq_num++;
	va_end(argp);
	return 0;
}

void printf_func_name(const char *func_name, size_t offset)
{
	if (offset)
		printf("<%s+0x%x>", func_name, offset);
	else
		printf("<%s>", func_name);
}

int printf_mnemonic(struct bfd_if *bi, bfd_vma addr, bfd_vma *p_baddr)
{
	int rc;
	char *p_tmp;
	struct disassemble_info *info = &bi->info;
	fprintf_ftype func_save;

	if (prepare_print_insn(bi, addr) < 0)
		return -1;

	seq_num = 0;
	offset = baddr_offset = 0;
	func_save = info->fprintf_func;
	info->fprintf_func = (fprintf_ftype)dummy_sprintf;
	rc = print_insn_i386_att(addr, &bi->info);
	info->fprintf_func = func_save;
	
	if (p_baddr) {
		*p_baddr = UNKNOWN_BADDR;
		if (baddr_offset > 0) {
			*p_baddr = strtoll(print_buf + baddr_offset, &p_tmp, 0);
			if (*p_tmp != '\0')
				*p_baddr = UNKNOWN_BADDR;
		}
	}
	if (rc >= 0)
		printf("%s", print_buf);
	return rc;
}

/*-----------------------------------------------------------------------------
 *  branch check
 *-----------------------------------------------------------------------------
 */
static int btype;
static bfd_vma baddr;

static int chk_btype_dummy_fprintf(FILE *stream, const char *fmt, ...)
{
	va_list argp;
	int rc;
	char *p_tmp;

	va_start(argp, fmt);
	rc = vsnprintf(print_buf, PRINT_BUF_MAX, fmt, argp);
	if (rc >= PRINT_BUF_MAX)
		last_error |= CHK_BTYPE_E_OVERFLOW;
	va_end(argp);

	//printf("DBG>%.*s<\n", rc, print_buf);
	switch (seq_num) {
	case 0:
		if (strncmp(print_buf, "jmp", 3) == 0 ||
		    strncmp(print_buf, "ljmp", 4) == 0) {
			btype = BTYPE_JMP;
		} else if (print_buf[0] == 'j' ||
			   strncmp(print_buf, "loop", 4) == 0) {
			btype = BTYPE_BRANCH;
		} else if (strncmp(print_buf, "call", 4) == 0) {
			btype = BTYPE_CALL;
		} else if (strncmp(print_buf, "ret", 3) == 0) {
			btype = BTYPE_RET;
		} else if (strncmp(print_buf, "iret", 4) == 0 ||
			   strncmp(print_buf, "sysexit", 7) == 0) {
			btype = BTYPE_IRET;
		} else if (strncmp(print_buf, "int", 3) == 0 ||
			   strncmp(print_buf, "hlt", 3) == 0 ||
			   strncmp(print_buf, "ud2", 3) == 0 ||
			   strncmp(print_buf, "sysenter", 8) == 0) {
			btype = BTYPE_INT;
		} else
			btype = BTYPE_OTHER;
		baddr = 0;
		break;
	case 1:
		switch (btype) {
		case BTYPE_JMP:
		case BTYPE_BRANCH:
		case BTYPE_CALL:
			baddr = strtoll(print_buf, &p_tmp, 0);
			if (p_tmp - print_buf != rc)
				baddr = UNKNOWN_BADDR;
				//last_error |= CHK_BTYPE_E_BADDR;
			break;
		}
		break;
	case 2:
		if (btype != BTYPE_OTHER && baddr != UNKNOWN_BADDR)
			last_error |= CHK_BTYPE_E_SEQNUM;
		break;
	}
	seq_num++;

	return rc;
}

int get_branch_info(struct bfd_if *bi, bfd_vma addr, int *type,
		    bfd_vma *addr_next, bfd_vma *addr_branch)
{
	int bytes;
	struct disassemble_info *info = &bi->info;
	fprintf_ftype func_save;

	if (prepare_print_insn(bi, addr) < 0)
		return -CHK_BTYPE_E_NOT_FOUND;

	seq_num = last_error = 0;
	baddr = 0;
	func_save = info->fprintf_func;
	info->fprintf_func = (fprintf_ftype)chk_btype_dummy_fprintf;
	//bytes = print_insn_i386_intel(addr, info);
	//bytes = print_insn_i386(addr, info);
	bytes = print_insn_i386_att(addr, info);
	if (bytes < 0) {
		last_error |= CHK_BTYPE_E_INTERNAL;
		goto EXIT;
	}
	info->fprintf_func = func_save;
	*addr_next = addr + bytes;
	*addr_branch = 0;
	switch (btype) {
	case BTYPE_JMP:
	case BTYPE_CALL:
	case BTYPE_BRANCH:
		*addr_branch = baddr;
		break;
	}
	*type = btype;
EXIT:
	return (last_error ? -last_error : 0);
}

/*-----------------------------------------------------------------------------
 *  path utility
 *-----------------------------------------------------------------------------
 */
static void printf_branch(struct branch *b)
{
	printf_bfd_vma(b->from);
	printf(" => ");
	printf_bfd_vma(b->to);
	printf(" (");
	printf_bfd_vma(b->end);
	printf(")\n");
}

void printf_path(struct path *p)
{
	struct branch *b;

	printf_bfd_vma(p->begin);
	printf("\n");
	for (b = p->branches; b; b = b->next) {
		printf("   B ");
		printf_branch(b);
	}
	if (p->call) {
		printf("   C ");
		printf_branch(p->call);
	}
	printf("   E ");
	printf_bfd_vma(p->end);
	printf("\n");
}

static int f_chk_range(void *__target, void *__dt)
{
	bfd_vma addr = *((bfd_vma*)__target);
	struct path *p = (struct path*)__dt;

	if (addr < p->begin)
		return -1;
	if (addr >= p->end)
		return 1;
	return 0;
}

static void f_one_path_free(void *__dt)
{
	struct path *p = (struct path*)__dt;
	struct branch *b, *tmp;

	if (p) {
		for (b = p->branches; b; b = tmp) {
			tmp = b->next;
			free(b);
		}
		free(p);
	}
}

struct path* find_path_from_addr(node *tree, bfd_vma addr)
{
	return (struct path*)search_tree(&addr, tree, f_chk_range);
}

node* delete_path_from_addr(node *tree, bfd_vma addr)
{
	return delete_tree(&addr, tree, f_chk_range, f_one_path_free);
}

/*-----------------------------------------------------------------------------
 *  path check
 *-----------------------------------------------------------------------------
 */
typedef struct __req_addrs {
	bfd_vma			addr;
	int			cnt;
	struct __req_addrs	*next;
} req_addrs;

typedef struct {
	struct bfd_if		*bi;
	node			**pt;
	req_addrs		*req_addrs;
} pack_bi_pt;

int f_print_path(void *__dt, void *user_data)
{
	struct path *p = (struct path*)__dt;

	printf_path(p);
	return 0;
}

void dump_path_tree(node *tree)
{
	for_each_node(tree, f_print_path, NULL);
}

static int add_req_addrs(node *pt, req_addrs **req, bfd_vma addr)
{
	req_addrs *r;

	if (addr == 0 || addr == UNKNOWN_BADDR || find_path_from_addr(pt, addr))
		return 0;

	r = malloc(sizeof(req_addrs));
	if (!r) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	r->addr = addr;
	r->cnt = 0;
	r->next = *req;
	*req = r;
#ifdef DBG_CHK_PATH
	printf("add_req_addrs:");
	printf_bfd_vma(addr);
	printf("\n");
#endif
	return 0;
}

static int chk_path_from_addr(struct bfd_if *bi, node **pt,
			      bfd_vma max, bfd_vma addr, bfd_vma *p_next,
			      req_addrs **req)
{
	bfd_vma begin, next, branch;
	int rc, type;
	struct path *p = NULL;
	struct branch *b, *b_prev = NULL;

	if (find_path_from_addr(*pt, addr))
		return 0;
#ifdef DBG_CHK_PATH
	printf("chk_path_from_addr called. ");
	printf_bfd_vma(addr);
	printf("<->");
	printf_bfd_vma(max);
	printf("\n");
#endif
	rc = 0;
	for (begin = addr; addr < max; addr = next) {
		rc = get_branch_info(bi, addr, &type, &next, &branch);
		if (rc < 0) {
			fprintf(stderr, "get_branch_info failed at ");
			fprintf_err_bfd_vma(addr);
			fprintf(stderr, ".(%d)\n", rc);
			return rc;
		}
		if (type == BTYPE_OTHER)
			continue;

		if (!p && !(p = calloc(1, sizeof(struct path)))) {
			fprintf(stderr, "calloc failed.(%s)\n",
				strerror(errno));
			return -1;
		}
		if (type == BTYPE_BRANCH || type == BTYPE_CALL) {
			if (!(b = calloc(1, sizeof(struct branch)))) {
				fprintf(stderr, "calloc failed.(%s)\n",
					strerror(errno));
				return rc;
			}
			b->from = addr;
			b->to = branch;
			b->end = next;

			if ((rc = add_req_addrs(*pt, req, branch)) < 0)
				return rc;
			if (type == BTYPE_BRANCH) {
				if (b_prev)
					b_prev->next = b;
				else
					p->branches = b;
				b_prev = b;
				continue;
			} else { // BTYPE_CALL
				p->call = b;
				if ((rc = add_req_addrs(*pt, req, next)) < 0)
					return rc;
			}
		} else if (type == BTYPE_JMP) {
			if ((rc = add_req_addrs(*pt, req, branch)) < 0)
				return rc;
		}
#ifdef CHK_LEFT_CODES
		/* check if left codes continue to checked range... */
		if (next == max && p_next) {
			asection *s;
			s = get_sect_has_addr(bi, begin);
			max = s->vma + bfd_get_section_size(s);
		}
#endif
		break;
	}
#ifdef CHK_LEFT_CODES
	if (p_next)
		*p_next = next;
#endif
	if (!p)
		return 0;
	if (next > max) {
		f_one_path_free(p);
		return 0;
	}
	p->begin = begin;
	p->end = next;

	/* if there is overcrossing address range, then delete old
	 * address range
	 */
	*pt = delete_path_from_addr(*pt, next - 1);
	*pt = insert_tree(p, *pt, f_chk_range);
	return 0;
}

#ifdef DBG_CHK_PATH
static void dbg_print_req_cnt(req_addrs *r)
{
	int cnt;

	for (cnt = 0; r; r = r->next, cnt++);
	printf("(req:%d)", cnt);
}
#endif

static int chk_req_addrs(struct bfd_if *bi, node **pt, asection *sect,
			 req_addrs *r, req_addrs **left)
{
	bfd *abfd = bi->abfd;
	asection *s;
	int code_sect_num = 0;
	bfd_vma max;
	req_addrs *r_sv;

	for (s = abfd->sections; s; s = s->next)
		if (is_code_sect(bi, s))
			code_sect_num++;

	max = sect->vma + bfd_get_section_size(sect);
	for (;;) {
		if (!r)
			break;
		if (r->addr < sect->vma || r->addr >= max) {
			r->cnt++;
			if (r->cnt >= code_sect_num) {
#ifdef DBG_CHK_PATH
				printf("WARN: none code-section addr(%d) ",
				       code_sect_num);
				printf_bfd_vma(r->addr);
				printf("\n");
#endif
				r_sv = r;
				r = r_sv->next;
				free(r_sv);
				continue;
			}
			r_sv = r;
			r = r->next;
			r_sv->next = *left;
			*left = r_sv;
		} else {
			r_sv = r;
			r = r_sv->next;
			if (chk_path_from_addr(bi, pt, max, r_sv->addr, NULL,
					       &r) < 0)
				return -1;
			free(r_sv);
		}
	}
	return 0;
}

static void chk_section_path(bfd *abfd, asection *sect, void *obj)
{
	bfd_vma max;
	pack_bi_pt *pack = (pack_bi_pt*)obj;
	struct bfd_if *bi = pack->bi;
	node **pt = pack->pt;
	req_addrs *r, **left = &pack->req_addrs;

	if (!is_code_sect(bi, sect))
		return;
	max = sect->vma + bfd_get_section_size(sect);

	/* first, left request check */
#ifdef DBG_CHK_PATH
	printf("==== check left");
	dbg_print_req_cnt(*left);
	printf("...\n");
#endif
	r = *left;
	*left = NULL;
	if (chk_req_addrs(bi, pt, sect, r, left) < 0) {
		last_error = 1;
		return;
	}
	/* start from section address */
#ifdef DBG_CHK_PATH
	printf("==== check from section start...\n");
#endif
	r = NULL;
	if (chk_path_from_addr(bi, pt, max, sect->vma, NULL, &r) < 0) {
		last_error = 1;
		return;
	}
	/* all request check */
#ifdef DBG_CHK_PATH
	printf("==== check all requests");
	dbg_print_req_cnt(r);
	printf("...\n");
#endif
	if (chk_req_addrs(bi, pt, sect, r, left) < 0)
		last_error = 1;
	return;
}

#ifdef CHK_LEFT_CODES
typedef struct __addr_range {
	bfd_vma			begin;
	bfd_vma			end;
	struct __addr_range	*next;
} addr_range;

addr_range	*left_ranges;

typedef struct {
	bfd_vma		last;
	struct bfd_if	*bi;
} pack_left;

int add_left_range(bfd_vma begin, bfd_vma end)
{
	addr_range *r;

#ifdef DBG_CHK_PATH
	printf("add_left_range b:");
	printf_bfd_vma(begin);
	printf(", e:");
	printf_bfd_vma(end);
	printf("\n");
#endif
	r = malloc(sizeof(addr_range));
	if (!r) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	r->begin = begin;
	r->end = end;
	r->next = left_ranges;
	left_ranges = r;
	return 0;
}

int f_chk_left_codes(void *__dt, void *user_data)
{
	struct path *p = (struct path*)__dt;
	pack_left *pleft = (pack_left*)user_data;
	bfd_vma last, *p_last = &pleft->last;
	asection *s;

	last = *p_last;
	*p_last = p->end;
	if (last == 0 || last == p->begin)
		return 0;
	s = get_sect_has_addr(pleft->bi, last);
	if (!s)
		return 0;
#ifdef DBG_CHK_PATH
	printf("f_chk_left_codes b:");
	printf_bfd_vma(last);
	printf(", e:");
	printf_bfd_vma(p->begin);
	printf("\n");
#endif
	if (p->begin < s->vma + bfd_get_section_size(s)) {
		if (add_left_range(last, p->begin) < 0)
			return -1;
	} else {
		if (add_left_range(last, s->vma + bfd_get_section_size(s)) < 0)
			return -1;
		s = get_sect_has_addr(pleft->bi, p->begin);
		if (s->vma == p->begin)
			return 0;
		if (add_left_range(s->vma, p->begin) < 0)
			return -1;
	}
	return 0;
}

int chk_one_range(struct bfd_if *bi, node **pt, addr_range *r, req_addrs **req)
{
	bfd_vma addr, next;

	for (addr = r->begin; addr < r->end; addr = next) {
		if (chk_path_from_addr(bi, pt, r->end, addr, &next, req) < 0)
			return -1;
		if (addr == next)
			break;
	}
	return 0;
}

int chk_left_codes(pack_bi_pt *pack)
{
	struct bfd_if *bi = pack->bi;
	node **pt = pack->pt;
	req_addrs **req = &pack->req_addrs;
	addr_range *r, *r_next;
	pack_left pleft = { 0, bi };

#ifdef DBG_CHK_PATH
	printf("**** check left codes ****\n");
#endif
	if (for_each_node(*pt, f_chk_left_codes, &pleft) < 0)
		return -1;
	for (r = left_ranges; r; r = r_next) {
		if (chk_one_range(bi, pt, r, req) < 0)
			return -1;
		r_next = r->next;
		free(r);
	}
	left_ranges = NULL;
	return 0;
}
#endif

#define GIVEUP_MAX	4
void loop_chk_req_addrs(bfd *abfd, pack_bi_pt *pack, int is_left_code_chk)
{
	long left_cnt, prev_left, giveup_cnt;
	req_addrs *r;

	prev_left = 0;
	for (giveup_cnt = 0; pack->req_addrs && giveup_cnt < GIVEUP_MAX;) {
		for (left_cnt = 0, r = pack->req_addrs;
		     r; r = r->next, left_cnt++);
		if (prev_left == left_cnt)
			giveup_cnt++;
		else
			giveup_cnt = 0;
		prev_left = left_cnt;

		if (is_left_code_chk) {
#ifdef CHK_LEFT_CODES
			if (chk_left_codes(pack) < 0) {
				last_error = 1;
				return;
			}
#else
			return;
#endif
		} else {
			bfd_map_over_sections(abfd, chk_section_path, pack);
			if (last_error)
				return;
		}
	}
	if (giveup_cnt >= GIVEUP_MAX) {
		printf("give up file analysis...\n");
		for (r = pack->req_addrs; r; r = r->next) {
			printf("    ");
			printf_bfd_vma(r->addr);
			printf("\n");
		}
		last_error = 1;
		return;
	}
}

struct bi_offset_pack {
	struct bfd_if	*bi;
	bfd_vma		offset;
};

int f_resolve_relocs_and_offset(void *__dt, void *user_data)
{
	struct path *p = (struct path*)__dt;
	struct bi_offset_pack *pack = (struct bi_offset_pack*)user_data;
	struct bfd_if *bi = pack->bi;
	bfd_vma offset = pack->offset;
	long i;
	struct branch *b;
	arelent *reloc;
	asymbol *sym;

	p->begin += offset;
	p->end += offset;
	for (b = p->branches; b; b = b->next) {
		b->from += offset;
		b->end += offset;
		if (b->to != UNKNOWN_BADDR)
			b->to += offset;
	}
	if ((b = p->call)) {
		b->from += offset;
		b->end += offset;
		if (b->to != UNKNOWN_BADDR) {
			for (i = 0; i < bi->n_relocs; i++) {
				reloc = bi->p_relocs[i];
				if (b->to != reloc->address)
					continue;
				sym = *reloc->sym_ptr_ptr;
				if (sym->flags & BSF_FUNCTION)
					b->to = bfd_asymbol_value(sym) + offset;
				else
					b->to = bfd_asymbol_value(sym);
				return 0;
			}
			b->to += offset;
		}
	}
	return 0;
}

void resolve_relocs_and_offset(struct bfd_if *bi, node *pt, bfd_vma offset)
{
	struct bi_offset_pack pack = { bi, offset };

	for_each_node(pt, f_resolve_relocs_and_offset, &pack);
}

int chk_path_tree(struct bfd_if *bi, node **pt, bfd_vma offset)
{
	int i;
	pack_bi_pt pack;
	req_addrs *r, *r_sv;
	asymbol *sym;

	last_error = 0;
	pack.bi = bi;
	pack.pt = pt;
	pack.req_addrs = NULL;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = bi->p_fsyms[i];
		if (!(sym->flags & BSF_FUNCTION))
			continue;
		if (add_req_addrs(*pt, &pack.req_addrs,
				  bfd_asymbol_value(sym)) < 0)
			break;
	}
	bfd_map_over_sections(bi->abfd, chk_section_path, &pack);
	if (last_error)
		goto EXIT;

	loop_chk_req_addrs(bi->abfd, &pack, 0);
	if (last_error)
		goto EXIT;

#ifdef CHK_LEFT_CODES
#ifdef DBG_CHK_PATH
	if (!last_error)
		dump_path_tree(*pt);
#endif
	if (chk_left_codes(&pack) < 0)
		last_error = 1;
	//loop_chk_req_addrs(bi->abfd, &pack, 1);
	/*
	if (last_error)
		goto EXIT;
		*/
#endif
EXIT:
	for (r = pack.req_addrs; r; r = r_sv) {
		r_sv = r->next;
		free(r);
	}
	if (!last_error)
		resolve_relocs_and_offset(bi, *pt, offset);
	/*
	add_offset_to_path_tree(*pt, offset);
	if (!last_error)
		resolve_relocs_to_path_tree(bi, *pt);
		*/
#ifdef DBG_CHK_PATH
	if (!last_error)
		dump_path_tree(*pt);
#endif
	return (last_error ? -last_error : 0);
}
