/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_ar_parse.c - /proc/modules and /proc/PID/maps parser                  */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#define eprintf(...)	fprintf(stderr, __VA_ARGS__)
#define dprintf(...)	if ((verbose)) printf(__VA_ARGS__)
#define ddprintf(...)	if ((verbose) >=2 ) printf(__VA_ARGS__)
int verbose;
#include "bt.h"

static struct pid_range *ranges;

int alloc_pid_range(pid_t pid)
{
	struct pid_range **p, *p_prev;

	p = &ranges;
	p_prev = *p;
	while (*p) {
		if ((*p)->pid == pid)
			return 0;
		p_prev = *p;
		p = &(*p)->next;
	}
	*p = malloc(sizeof(struct pid_range));
	if (!*p) {
		// output error message
		return -1;
	}
	(*p)->pid = pid;
	(*p)->range = NULL;
	(*p)->next = NULL;
	if (p_prev)
		p_prev->next = *p;
	return 0;
}

static struct pid_range* get_pid_range(pid_t pid)
{
	struct pid_range *p;

	p = ranges;
	while (p) {
		if (p->pid == pid)
			return p;
		p = p->next;
	}
	return p;
}

static void free_one_range(struct addr_range **p_ar)
{
	struct addr_range *p_tmp;

	if (!*p_ar)
		return;
	p_tmp = (*p_ar)->next;
	free(*p_ar);
	*p_ar = p_tmp;
}

void free_ranges(void)
{
	struct pid_range *p, *p_next;

	p = ranges;
	while (p) {
		while (p->range)
			free_one_range(&p->range);
		p_next = p->next;
		free(p);
		p = p_next;
	}
	ranges = NULL;
}

char* range2ulongs(char *p, unsigned long *begin, unsigned long *end)
{
	char *p_end;

	*begin = strtoul(p, &p_end, 0);
	if (*p_end != ':') {
		eprintf("begin address invalid(%s)\n", p);
		return NULL;
	}
	*end = strtoul(p_end + 1, &p_end, 0);
	if (*p_end != '\0' && *p_end != '\n' && *p_end != ' ') {
		eprintf("end address invalid(%s)\n", p);
		return NULL;
	}
	if (*begin >= *end) {
		eprintf("begin address greater or equal end address(%s)\n", p);
		return NULL;
	}
	return p_end;
}

static int is_overlap(unsigned long b1, unsigned long e1,
		      unsigned long b2, unsigned long e2, int continue_ok)
{
	unsigned long long len1, len2;

	len1 = e1 - b1;
	len2 = e2 - b2;

	if (continue_ok) {
		len1++;
		len2++;
	}
	if (b2 == b1 ||
	    (b2 > b1 && b2 - b1 <= len1) ||
	    (b2 < b1 && b1 - b2 <= len2))
		return 1;
	return 0;
}

int add_range(pid_t pid, unsigned long begin, unsigned long end)
{
	struct pid_range *p;
	struct addr_range **pp_ar, *p_ar, *p_tmp;

	ddprintf("------ ADD RANGE (0x%08lx, 0x%08lx) ------\n", begin, end);
	if (!(p = get_pid_range(pid))) {
		// error print
		return -1;
	}
	/* detect overlapped range */
	pp_ar = &p->range;
	while ((p_ar = *pp_ar)) {
		ddprintf("CHECK\n");
		/* overlapped ? */
		if (is_overlap(p_ar->begin, p_ar->end, begin, end, 1)) {
			ddprintf("OVERLAP DETECT\n");
			/* expand the range */
			if (begin < p_ar->begin)
				p_ar->begin = begin;
			if (end > p_ar->end) {
				p_ar->end = end;
				/* check upper ranges */
				p_tmp = p_ar->next;
				while (p_tmp) {
					if (p_tmp->begin <= p_ar->end ||
					    (p_tmp->begin > p_ar->end &&
					     p_tmp->begin - 1 == p_ar->end)){
						if (p_tmp->end > p_ar->end)
							p_ar->end = p_tmp->end;
						free_one_range(&p_ar->next);
						p_tmp = p_ar->next;
					} else {
						break;
					}
				}
			}
			return 0;
		}
		if (p_ar->begin > end) {
			ddprintf("BREAK\n");
			break;
		}
		pp_ar = &p_ar->next;
	}
	/* not overlapped */
	p_tmp = malloc(sizeof(struct addr_range));
	p_tmp->begin = begin;
	p_tmp->end = end;
	if (p->range) {
		p_tmp->next = *pp_ar;
		*pp_ar = p_tmp;
	} else {
		p_tmp->next = NULL;
		p->range = p_tmp;
	}
	return 0;
}

int del_range(pid_t pid, unsigned long begin, unsigned long end)
{
	struct pid_range *p;
	struct addr_range **pp_ar, *p_ar, *p_tmp;

	ddprintf("------ DEL RANGE (0x%08lx, 0x%08lx) ------\n", begin, end);
	if (!(p = get_pid_range(pid))) {
		// error print
		return -1;
	}
	/* detect overlapped range */
	pp_ar = &p->range;
	while ((p_ar = *pp_ar)) {
		ddprintf("CHECK\n");
		/* overlapped ? */
		if (is_overlap(p_ar->begin, p_ar->end, begin, end, 0)) {
			ddprintf("OVERLAP DETECT\n");
			if (p_ar->begin >= begin && p_ar->end <= end) {
				/* delete range include existance range */
				free_one_range(pp_ar);
			} else if (p_ar->begin < begin && p_ar->end > end){
				/* existence range include delete range */
				p_tmp = malloc(sizeof(struct addr_range));
				p_tmp->begin = end + 1;
				p_tmp->end = p_ar->end;
				p_ar->end = begin - 1;
				p_tmp->next = p_ar->next;
				p_ar->next = p_tmp;
				pp_ar = &p_ar->next;
			} else if (p_ar->begin < begin) {
				/* existence range overlapped (upper part) */
				p_ar->end = begin - 1;
				pp_ar = &p_ar->next;
			} else {
				/* existence range overlapped (lower part) */
				p_ar->begin = end + 1;
				pp_ar = &p_ar->next;
			}
		}
		if (p_ar->begin > end) {
			ddprintf("BREAK\n");
			break;
		}
		pp_ar = &p_ar->next;
	}
	return 0;
}

void dump_ranges(void)
{
	struct pid_range *p;
	struct addr_range *p_ar;

	p = ranges;
	while (p) {
		printf("------ pid: %d ------\n", p->pid);
		p_ar = p->range;
		while (p_ar) {
			printf("0x%08lx:0x%08lx\n", p_ar->begin, p_ar->end);
			p_ar = p_ar->next;
		}
		p = p->next;
	}
}

struct pid_range *get_all_ranges(void)
{
	return ranges;
}

struct addr_range *get_pid_addr_range(pid_t pid)
{
	struct pid_range *p;

	p = ranges;
	while (p) {
		if (p->pid == pid)
			return p->range;
		p = p->next;
	}
	return NULL;
}

/* for speed up, this routine should be convert to the macro... */
int is_addr_match(struct bt_record *p, struct addr_range *r)
{
	while (r) {
		if ((p->from >= r->begin && p->from <= r->end) ||
		    (p->to >= r->begin && p->to <= r->end))
			return 1;
		r = r->next;
	}
	return 0;
}

/*----------------------------------------------------------------------------*/
/*  utility functions                                                         */
/*----------------------------------------------------------------------------*/
int open_mmapfile(char *path, int *fd, void **p, size_t *size)
{
	struct stat st;

	*fd = -1;
	*p = MAP_FAILED;
	*size = 0;

	if ((*fd = open(path, O_RDONLY)) < 0) {
		fprintf(stderr, "can't open %s.(%s)\n", path,
			strerror(errno));
		return -1;
	}
	if (lstat(path, &st) < 0) {
		fprintf(stderr, "%s lstat failed.(%s)\n", path,strerror(errno));
		return -1;
	}
	*size = st.st_size;
	*p = mmap(NULL, *size, PROT_READ, MAP_PRIVATE, *fd, 0);
	if (*p == MAP_FAILED) {
		fprintf(stderr, "mmap failed.(%s)\n", strerror(errno));
		return -1;
	}
	return 0;
}

void close_mmapfile(int fd, void *p, size_t size)
{
	if (p != MAP_FAILED)
		munmap(p, size);
	if (fd >= 0)
		close(fd);
}

static int find_under_path(char *dirname, int max_len, const char *name,
			   int down)
{
	DIR *dir;
	struct dirent *d;
	int dlen, rc;
	struct stat stat;

	//printf("chk dir =>%s< (for >%s<)\n", dirname, name);
	rc = -1;
	if (!(dir = opendir(dirname)))
		return -1;
	while ((d = readdir(dir)) != NULL) {
		if (strcmp(".", d->d_name) == 0 || strcmp("..", d->d_name) == 0)
			continue;
		//printf("chk child =>%s<\n", d->d_name);
		dlen = strlen(dirname);
		if (strcmp(name, d->d_name) == 0) {
			snprintf(dirname + dlen, max_len - dlen,
				 "/%s", d->d_name);
			rc = 0;
			goto EXIT;
		}
		if (!down)
			continue;
		snprintf(dirname + dlen, max_len - dlen, "/%s", d->d_name);
		if (lstat(dirname, &stat) < 0)
			goto EXIT;
		if (S_ISDIR(stat.st_mode)) {
			if (find_under_path(dirname, max_len, name, down) == 0){
				rc = 0;
				goto EXIT;
			}
		}
		dirname[dlen] = '\0';
	}
EXIT:
	closedir(dir);
	return rc;
}

static int get_module_full_path(char *buf, int buf_len, const char *name)
{
	struct utsname utsn;

	if (uname(&utsn) < 0)
		return -1;
	if (buf[0] == '\0')
		snprintf(buf, buf_len, "/lib/modules/%s", utsn.release);
	return find_under_path(buf, buf_len, name, 1);
}

static int __get_vmlinux_full_path(char *buf, int buf_len)
{
	int rc;

	rc = find_under_path(buf, buf_len, "vmlinux", 0);
	if (rc >= 0)
		return rc;
	return find_under_path(buf, buf_len, "vmlinux", 1);
}

static int get_vmlinux_full_path(char *buf, int buf_len)
{
	struct utsname utsn;

	if (buf[0] != '\0') {	/* search start from "buf" path */
		if (__get_vmlinux_full_path(buf, buf_len) >= 0)
			return 0;
	}
	if (uname(&utsn) < 0)
		return -1;
	snprintf(buf, buf_len, "/usr/lib/debug/lib/modules/%s", utsn.release);
	if (__get_vmlinux_full_path(buf, buf_len) >= 0)
		return 0;
	snprintf(buf, buf_len, "/lib/modules/%s/build", utsn.release);
	if (__get_vmlinux_full_path(buf, buf_len) >= 0)
		return 0;
	return -1;
}

//#define HOME_DEBUG
#ifdef HOME_DEBUG
#define DBG_MOD_PATH	"/home/tetsu/siken/to/copied_objcts/modules"
#endif

int parse_modules(t_func_r2n func, char *dir)
{
	FILE *fd;
	int rc, num, len;
	char path[128], kallsyms[128];
	char buf[MAX_LINE_LEN + 1], *p, fullname[MAX_LINE_LEN + 1];
	struct addr_range *range;
	struct bt_record rec;

	rc = -1;
	snprintf(kallsyms, sizeof(kallsyms), "%s/kallsyms", dir);
	snprintf(path, sizeof(path), "%s/modules", dir);
	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = fullname[MAX_LINE_LEN] = '\0';
	range = get_pid_addr_range(ALL_PID);
	while ((p = fgets(buf, MAX_LINE_LEN, fd)) != NULL) {
		len = strlen(p) - 1;
		buf[len] = '\0';
		num = sscanf(buf, "%s %ld %*d %*s %*s %lx",
			     fullname, &rec.to, &rec.from);
		if (num != 3) {
			fprintf(stderr, "modules format error(%d)\n", num);
			goto EXIT;
		}
		rec.to += rec.from;
		if (!is_addr_match(&rec, range))
			continue;
		snprintf(buf, MAX_LINE_LEN, "%s.ko", fullname);
#ifdef HOME_DEBUG
		sprintf(fullname, "%s", DBG_MOD_PATH);
#else
		fullname[0] = '\0';
#endif
		if (get_module_full_path(fullname, MAX_LINE_LEN, buf) < 0) {
			printf("WARN: %s not found.\n", buf);
			continue;
		}
		if (func(rec.from, rec.to, fullname, kallsyms, 1) < 0)
			goto EXIT;
	}
#ifdef HOME_DEBUG
	sprintf(fullname, "%s", DBG_MOD_PATH);
#else
	fullname[0] = '\0';
#endif
	if (get_vmlinux_full_path(fullname, MAX_LINE_LEN) < 0)
		printf("WARN: vmlinux not found.\n");
	else {
		if (func(0, 0, fullname, NULL, 0) <0)
			goto EXIT;
	}
	rc = 0;
EXIT:
	if (fd)
		fclose(fd);
	return rc;
}

int parse_maps(t_func_r2n func, char *dir, char *fname)
{
	FILE *fd;
	int rc, num, len;
	char path[128];
	char mode[6], buf[MAX_LINE_LEN + 1], *p;
	struct addr_range *range;
	struct bt_record rec;
	long size;

	rc = -1;
	snprintf(path, sizeof(path), "%s/%s.maps", dir, fname);
	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = '\0';
	range = get_pid_addr_range(ALL_PID);
	while ((p = fgets(buf, MAX_LINE_LEN, fd)) != NULL) {
		len = strlen(p) - 1;
		buf[len] = '\0';
		num = sscanf(buf, "%lx-%lx %s %*s %*x %*s %ld",
			     &rec.from, &rec.to, mode, &size);
		if (num != 4) {
			fprintf(stderr, "map format error(%d)\n", num);
			goto EXIT;
		}
		if (size == 0 || mode[2] != 'x')
			continue;
		for (; *p != '/' && *p != '\0'; p++);
		if (*p != '/' || buf[len-1] == ')')
			continue;	// e.g. /SYSV00000000 (deleted)
		rec.to -= 1;
		if (!is_addr_match(&rec, range))
			continue;
		if (func(rec.from, rec.to, p, NULL, 0) < 0)
			goto EXIT;
	}
	rc = 0;
EXIT:
	if (fd)
		fclose(fd);
	return rc;
}
