/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_irq.c - branch trace module (interrupt handler)                       */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <linux/interrupt.h>
#include <linux/irq.h>
#include <asm/apic.h>
#include "bt.h"

extern struct ds_manage *ds_manage[NR_CPUS];
extern cpumask_t bt_enabled;
extern unsigned long irq_desc_p;
extern unsigned long no_irq_type_p;

static char *irq_name = "bts";
static int pmi_irq = -1;

void write_bt_records(void*, size_t);

void bts_log_write(void);

static void do_nothing(unsigned int irq)
{
}

static unsigned int startup_pmi(unsigned int irq)
{
	return 0;
}

static void set_affinity_pmi(unsigned int irq, cpumask_t dest) {
}

static void irq_mask_clear(void) {
	unsigned long dt;

	dt = apic_read(APIC_LVTPC);
	dt &= ~APIC_LVT_MASKED;
	apic_write(APIC_LVTPC, dt);
}

static void ack(unsigned int irq)
{
	int cpu;
	unsigned int low, high;

	ack_APIC_irq();

	cpu = smp_processor_id();

	if (!ds_manage[cpu] || !cpu_isset(cpu, bt_enabled)) {
		return;
	}
	rdmsr(MSR_DEBUGCTLA, low, high);
	wrmsr(MSR_DEBUGCTLA, low & ~MSR_DEBUGCTLA_TR, high);

	/* for DEBUG
	{
		irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
		serial_prints("ack(cpu:%d) (pend:%d)\n", smp_processor_id(),
			      desc[pmi_irq].status & IRQ_PENDING);
	}
	*/
	bts_log_write();
	irq_mask_clear();

	wrmsr(MSR_DEBUGCTLA, low, high);
}

static void end(unsigned int irq)
{
	/* for DEBUG
	irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
	serial_prints("end(cpu:%d) (pend:%d)\n", smp_processor_id(),
		      desc[pmi_irq].status & IRQ_PENDING);
		      */
}

static struct hw_interrupt_type	bt_int_type = {
	.typename = "BT",
	.startup = startup_pmi,
	.shutdown = do_nothing,
	.enable = do_nothing,
	.disable = do_nothing,
	.ack = ack,
	.end = end,
	.set_affinity = set_affinity_pmi
};

irqreturn_t bt_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	return IRQ_HANDLED;
}

static int __init init_isr_per_system(void)
{
	int irq;
	int rc;
	irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
	struct hw_interrupt_type *type
		= (struct hw_interrupt_type*)no_irq_type_p;

	for (irq = 95; irq > 15; irq--) {
		if (desc[irq].handler == type)
			break;
	}
	if (irq == 15)
		return -EBUSY;

	desc[irq].handler = &bt_int_type;
	rc = request_irq(irq, bt_interrupt, SA_INTERRUPT | SA_SHIRQ,
			 irq_name, irq_name);
	if (rc) {
		printk("%s: cannot setup irq %d for BTS\n", MOD_NAME, irq);
		desc[irq].handler = type;
		return rc;
	}
	pmi_irq = irq;

	return 0;
}

static void __init init_isr_per_cpu(void* data)
{
	int vector;
	unsigned long dt, current_vector;
	int cpu;
	cpumask_t *failed_map = data;

	cpu = smp_processor_id();
	vector = pmi_irq + FIRST_EXTERNAL_VECTOR;

#if 1
	dt = apic_read(APIC_LVTPC);
	if (dt & APIC_DM_SMI) {
		printk("%s: (%d)BTS facility handled by SMI\n", MOD_NAME, cpu);
		return;
	}
#endif
	current_vector = dt & APIC_VECTOR_MASK;
	/* check whether a vector already exists, temporarily masked? */	
	if (current_vector != 0 && current_vector != APIC_VECTOR_MASK) {
		printk("%s: (%d)BTS LVT vector (0x%016lx) already installed\n",
		       MOD_NAME, cpu, (dt & APIC_VECTOR_MASK));
		return;
	}
	apic_write(APIC_LVTPC, vector | APIC_DM_FIXED | APIC_LVT_MASKED);

	dt = apic_read(APIC_LVTPC);
	apic_write_around(APIC_LVTPC, dt & ~APIC_LVT_MASKED);
	serial_prints("CPU%d: BTS overflow interrupt enabled\n", cpu);
	serial_prints("\t(0x%016lx)\n", apic_read(APIC_LVTPC));
	cpu_clear(cpu, *failed_map);
}

static void cleanup_isr_per_system(void)
{
	irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
	struct hw_interrupt_type *type
		= (struct hw_interrupt_type*)no_irq_type_p;

	free_irq(pmi_irq, irq_name);
	desc[pmi_irq].handler = type;

	return;
}

static void cleanup_isr_per_cpu(void *data)
{
	unsigned long dt;
	int cpu;

	cpu = smp_processor_id();
	dt = apic_read(APIC_LVTPC);
	if ((dt & APIC_VECTOR_MASK) != pmi_irq + FIRST_EXTERNAL_VECTOR)
		return;

	apic_write(APIC_LVTPC, APIC_LVT_MASKED | APIC_VECTOR_MASK);
	serial_prints("CPU:%d LVTPC cleanup(0x%016lx) exit\n", cpu,
		      apic_read(APIC_LVTPC));

	return;
}

int setup_isr(void)
{
	int rc;
	cpumask_t failed_map = cpu_online_map;

	if ((rc = init_isr_per_system()) < 0)
		return rc;
	on_each_cpu(init_isr_per_cpu, &failed_map, 1, 1);
	if (!cpus_empty(failed_map))
		return -EBUSY;
	return 0;
}

void cleanup_isr(void)
{
	if (pmi_irq < 0)
		return;
	cleanup_isr_per_system();
	on_each_cpu(cleanup_isr_per_cpu, NULL, 1, 1);
	pmi_irq = -1;
}
