/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_main.h - branch trace module header                                   */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/smp.h>
#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <asm/pgtable.h>
#include "bt.h"

cpumask_t bt_enabled;
pid_t *pid_tbl;
int pid_max = 256;
int pid_cnt;

unsigned long irq_desc_p;
unsigned long no_irq_type_p;
unsigned long switch_to_addr;
int switch_to_size;

size_t btsbuf_size = BTS_BUF_MIN_SIZE;
size_t subbuf_size = 2 * 1024 * 1024;
size_t subbuf_num = 16;
size_t subbuf_sleep_threshold = 8;
int do_not_overwrite = 0;

struct ds_manage *ds_manage[NR_CPUS];
struct pid_manage pid_manage[NR_CPUS];

/* prototypes */
int reg_probe(void);
void unreg_probe(void);

int proc_init(void);
void proc_cleanup(void);
static void bt_mod_cleanup(void);

void write_pid_record(pid_t);
void write_bt_records(void*, size_t);
int relfs_init(void);
void relfs_cleanup(void);
void relfs_flush(void);

int setup_isr(void);
void cleanup_isr(void);

#ifndef DEBUG
int serial_init(int ttyS) { return 0; }
asmlinkage int serial_prints(const char *fmt, ...) { return 0; }
#endif

void bts_log_write(void)
{
	int cpu = smp_processor_id();
	struct ds_manage *ds_mng;
	struct pid_manage *pid_mng;

	ds_mng = ds_manage[cpu];
	pid_mng = &pid_manage[cpu];
	if (!ds_mng || ds_mng->bts_index <= ds_mng->bts_base)
		return;

	if (is_app_mode && !pid_mng->is_wrote) {
		write_pid_record(pid_mng->pid);
		pid_mng->is_wrote = 1;
	}
	write_bt_records((void*)ds_mng->bts_base,
			 ds_mng->bts_index - ds_mng->bts_base);
	ds_mng->bts_index = ds_mng->bts_base;
}

int is_trace_pid(pid_t pid)
{
	int i;

	for (i = 0; i < pid_cnt; i++)
		if (pid_tbl[i] == pid)
			return 1;
	return 0;
}

void bts_facility_on(void)
{
	int cpu = smp_processor_id();
	unsigned int low, high;

	if (!ds_manage[cpu])
		return;

	rdmsr(MSR_DEBUGCTLA, low, high);
	low |= MSR_DEBUGCTLA_BTS | MSR_DEBUGCTLA_TR | MSR_DEBUGCTLA_BTINT;
	wrmsr(MSR_DEBUGCTLA, low, high);
}

void bts_facility_off(void)
{
	int cpu = smp_processor_id();
	unsigned int low, high;

	if (!ds_manage[cpu])
		return;

	rdmsr(MSR_DEBUGCTLA, low, high);
	low &= ~(MSR_DEBUGCTLA_BTS | MSR_DEBUGCTLA_TR |
		 MSR_DEBUGCTLA_BTINT | MSR_DEBUGCTLA_LBR);
	wrmsr(MSR_DEBUGCTLA, low, high);
}

static void bt_enable_per_cpu(void *data)
{
	int cpu;
	pid_t pid;
	struct pid_manage *pid_mng;

	serial_prints("bt_enable_per_cpu(cpu:%d, preemptible:%d)\n",
		      smp_processor_id(), preemptible());
	cpu = smp_processor_id();
	if (!ds_manage[cpu])
		return;

	if (is_app_mode) {
		pid = current->pid;
		if (!is_trace_pid(pid))
			goto EXIT;
		pid_mng = &pid_manage[cpu];
		pid_mng->pid = pid;
		pid_mng->is_wrote = 0;
	}
	bts_facility_on();
EXIT:
	cpu_set(cpu, bt_enabled);
	return;
}

static void bt_disable_per_cpu(void *data)
{
	int cpu;

	serial_prints("bt_disable_per_cpu(cpu:%d, preemptible:%d)\n",
		      smp_processor_id(), preemptible());
	cpu = smp_processor_id();
	if (!ds_manage[cpu])
		return;

	cpu_clear(cpu, bt_enabled);
	bts_facility_off();

	bts_log_write();
}

void bt_enable(void)
{
	on_each_cpu(bt_enable_per_cpu, NULL, 1, 0);
}

void bt_disable(void)
{
	int need_flush = !cpus_empty(bt_enabled);

	on_each_cpu(bt_disable_per_cpu, NULL, 1, 1);
	if (need_flush)
		relfs_flush();
}

EXPORT_SYMBOL_GPL(bt_enable);
EXPORT_SYMBOL_GPL(bt_disable);

static int check_bts_availability(void)
{
	unsigned int eax, ebx, ecx, edx;
	
	cpuid(0x01, &eax, &ebx, &ecx, &edx);
	if (!(edx & (1 << 21))) {
		printk("%s: BTS facility is not available in this processor.\n",
		       MOD_NAME);
		return -ENOSYS;
	}
	
	rdmsr(MSR_DEBUGCTLA, eax, edx);
	if (eax & MSR_DEBUGCTLA_BTS) {
		printk("%s: BTS facility is already used.\n", MOD_NAME);
		return -EBUSY;
	}
	
	return 0;
}

static void setup_ds_area_per_cpu(void *data)
{
	int cpu = smp_processor_id();
	struct ds_manage *ds_mng;
	unsigned int low, high;

	ds_mng = ds_manage[cpu];
	if (!ds_mng)
		return;

	rdmsr(MSR_IA32_DS_AREA, low, high);
	wrmsr(MSR_IA32_DS_AREA, ds_mng, high);
}

static int setup_ds_area(void)
{
	int i;
	struct ds_manage *ds_mng;

	btsbuf_size -= btsbuf_size % sizeof(struct bt_record); 
	for (i = 0; i < NR_CPUS; i++) {
		if (!cpu_isset(i, cpu_online_map))
			continue;

		ds_mng = (struct ds_manage*)vmalloc(sizeof(struct ds_manage)
						    + btsbuf_size);
		ds_manage[i] = ds_mng;
		if (!ds_mng) {
			printk("%s: BTS buffer cannot allocate.\n", MOD_NAME);
			return -ENOMEM;
		}

		ds_mng->bts_base = (unsigned long)&ds_mng[1];
		ds_mng->bts_index = ds_mng->bts_base;
		ds_mng->bts_max = ds_mng->bts_base + btsbuf_size;
		ds_mng->bts_threshold =
			ds_mng->bts_max
				- THRESHOLD_RECS * sizeof(struct bt_record);
	}
	on_each_cpu(setup_ds_area_per_cpu, NULL, 1, 1);
	return 0;
}

static void cleanup_ds_area(void)
{
	int i;

	for (i = 0; i < NR_CPUS; i++)
		if (ds_manage[i]) {
			vfree(ds_manage[i]);
			ds_manage[i] = NULL;
		}
}

static int bt_mod_init(void)
{
	int rc = 0;

	cpus_clear(bt_enabled);

	/* check module parameters */
	if (!irq_desc_p || !no_irq_type_p || !switch_to_addr || !switch_to_size
	    || !subbuf_num || subbuf_size < BTS_BUF_MIN_SIZE
	    || !subbuf_sleep_threshold || btsbuf_size < BTS_BUF_MIN_SIZE) {
		rc = -EINVAL;
		goto ERROR;
	}
	serial_init(1);
	if ((rc = reg_probe()) < 0)
		goto ERROR;
	if ((rc = check_bts_availability()) < 0)
		goto ERROR;
	if ((rc = setup_ds_area()) < 0)
		goto ERROR;
	if ((rc = proc_init()) < 0)
		goto ERROR;
	if ((rc = relfs_init()) < 0)
		goto ERROR;
	if ((rc = setup_isr()) < 0)
		goto ERROR;
	return rc;
ERROR:
	bt_mod_cleanup();
	return rc;
}

static void bt_mod_cleanup(void)
{
	bt_disable();
	cleanup_isr();
	relfs_cleanup();
	proc_cleanup();
	cleanup_ds_area();
	unreg_probe();

	return;
}

module_init(bt_mod_init);
module_exit(bt_mod_cleanup);

module_param(btsbuf_size, int, 0);
MODULE_PARM_DESC(btsbuf_size, "BTS buffer size.");
module_param(subbuf_size, int, 0);
MODULE_PARM_DESC(subbuf_size, "relayfs buffer size.");
module_param(subbuf_num, int, 0);
MODULE_PARM_DESC(subbuf_num, "relayfs buffer number.");
module_param(subbuf_sleep_threshold, int, 0);
MODULE_PARM_DESC(subbuf_sleep_threshold,
		 "target process sleep threshold of relayfs buffers.");
module_param(do_not_overwrite, int, 0);
MODULE_PARM_DESC(do_not_overwrite, "do not overwrite relayfs buffer.");
module_param(irq_desc_p, ulong, 0);
module_param(no_irq_type_p, ulong, 0);
module_param(switch_to_addr, ulong, 0);
module_param(switch_to_size, int, 0);

MODULE_AUTHOR("Yumiko Sugita <sugita@sdl.hitachi.co.jp>,\n" \
	      "\t\tSatoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>");
MODULE_DESCRIPTION("Hardware BTS facility controller");
MODULE_LICENSE("GPL");
