/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_hconv.c - coverage output to html converter                           */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <sys/stat.h>
#include "bt_ar_parse.h"
#include "bt_hconv.h"

/*
 * When the object contains debugging information, we need to create the source
 * code html.
 * The source line number of the result of the coverage check has the
 * possibility of appearing at random when it compiled with optimization.
 * Moreover, there is a possibility that two or more branches are included in
 * the same source line, too.
 * Therefore, we first create the execute-type per line array, then create the
 * source html by merging it with source file.
 */

#define T_NONE		0
#define T_NOT_EXECUTED	1
#define T_EXECUTED	2

struct src_info {
	char	path[PATH_MAX + 1];	/* absolute path */
	long	ln_max;
	char	*exec_types;		/* execute-type per line */

	/* cache data that is used for convert abs-path to html-path */
	int	is_ref;
	char	html_out_path[PATH_MAX + 1];
};

static struct src_info **src_info;
static long src_info_num;

/*-----------------------------------------------------------------------------
 *  output html
 *-----------------------------------------------------------------------------
 */
#define COLOR_OK	"#00ff00"
#define COLOR_HT	"#ffff00"
#define COLOR_NT	"#ff0000"

static void out_html_header(FILE *f, char *doctype, char *title)
{
	fprintf(f,
		"%s\n" \
		"<html lang=\"en\">\n" \
		"<head>\n" \
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=ISO-8859-1\">\n" \
		"<meta http-equiv=\"Content-Style-Type\" content=\"text/css\">\n" \
		"<title>%s</title>\n" \
		"</head>\n",
		doctype, title);
}

static void out_html_frame_header(FILE *f, char *title)
{
	out_html_header(f,
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\">",
			title);
}

static void out_html_normal_header(FILE *f, char *title)
{
	out_html_header(f,
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">",
			title);
}

static int out_top_html(struct html_out_data *dt)
{
	char buf[PATH_MAX + 1];
	FILE *f;

	buf[PATH_MAX] = '\0';
	snprintf(buf, PATH_MAX, "%s/top.html", dt->outdir);
	f = fopen(buf, "w");
	if (!f) {
		fprintf(stderr, "%s can't open.(%s)\n", buf, strerror(errno));
		return -1;
	}
	out_html_frame_header(f, "BTRAX - COVERAGE RESULT");
	fprintf(f,
		"<frameset title=\"all\" rows=\"40,*\">\n" \
		"  <frame title=\"title\" src=\"./title.html\" scrolling=\"no\" noresize>\n" \
		"  <frameset title=\"right\" cols=\"50%%,*\">\n" \
		"    <frameset title=\"left\" rows=\"40%%,*\">\n" \
		"      <frame title=\"summary\" name=\"summary\" src=\"./summary.html\">\n" \
		"      <frame title=\"each\" name=\"each\">\n" \
		"    </frameset>\n" \
		"    <frame title=\"src\" name=\"src\">\n" \
		"  </frameset>\n" \
		"  <noframes><body><p>use frame supported browser</p></body></noframes>\n" \
		"</frameset>\n" \
		"</html>\n");
	fclose(f);
	return 0;
}

static int out_title_html(struct html_out_data *dt)
{
	char buf[PATH_MAX + 1];
	FILE *f;

	buf[PATH_MAX] = '\0';
	snprintf(buf, PATH_MAX, "%s/title.html", dt->outdir);
	f = fopen(buf, "w");
	if (!f) {
		fprintf(stderr, "%s can't open.(%s)\n", buf, strerror(errno));
		return -1;
	}
	out_html_normal_header(f, "title");
	fprintf(f,
		"<body style=\"background-color:silver\">\n" \
		"<p style=\"font-size:16pt; font-weight:bold; text-align:center\">\n" \
		"<span style=\"color:#0000ff\">BTRAX</span> Coverage\n" \
		"<span style=\"color:#ff0000\">R</span>e" \
		"<span style=\"color:#ffff00\">s</span>u" \
		"<span style=\"color:#00ff00\">l</span>t\n" \
		"</p>\n" \
		"</body></html>\n");
	fclose(f);
	return 0;
}

static char* get_rel_path(struct html_out_data *dt, char cov_type)
{
	snprintf(dt->abs_path, PATH_MAX, "%s/each/%s.%c.html",
		 dt->outdir, dt->name, cov_type);
	return dt->abs_path + strlen(dt->outdir) + 1;
}

static void open_html(struct html_out_data *dt)
{
	dt->cur_out = fopen(dt->abs_path, "w");
	if (!dt->cur_out) {
		fprintf(stderr, "%s can't open.(%s)\n",
			dt->abs_path, strerror(errno));
		exit(1);
	}
}

static void close_html(struct html_out_data *dt)
{
	if (dt->cur_out) {
		fclose(dt->cur_out);
		dt->cur_out = NULL;
	}
}

static void out_summary_html_start(struct html_out_data *dt)
{
	out_html_normal_header(dt->summary_out, "summary");
	fprintf(dt->summary_out,
		"<body>\n" \
		"<table summary=\"summary\" border=\"1\">\n" \
		"  <tr><th rowspan=\"2\">name</th><th colspan=\"3\">coverage</th></tr>\n" \
		"  <tr><th>function</th><th>branch</th><th>state</th></tr>\n");
}

void out_summary_html_name(struct html_out_data *dt)
{
	fprintf(dt->summary_out, "  <tr><td>%s</td>\n", dt->name);
}

void out_summary_html_func(struct html_out_data *dt,
			   long n_func, long n_func_all)
{
	char *rel_path;

	rel_path = get_rel_path(dt, 'f');
	fprintf(dt->summary_out,
		"    <td align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a></td>\n",
		rel_path, rel_path, (double)n_func * 100 / n_func_all);
}

void out_summary_html_branch(struct html_out_data *dt,
			     long n_br_ok, long n_br_uk,
			     long n_br_ht, long n_br_nt, long n_br_all)
{
	char *rel_path;

	rel_path = get_rel_path(dt, 'b');
	if (n_br_all)
		fprintf(dt->summary_out,
			"    <td align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a></td>\n",
			rel_path, rel_path,
			(double)(n_br_ok * 2 + n_br_uk + n_br_ht) * 100
								/ n_br_all);
	else
		fprintf(dt->summary_out,
			"    <td align=\"right\">----</td>\n");
}

void out_summary_html_state(struct html_out_data *dt, long n_ok, long n_states)
{
	char *rel_path;

	rel_path = get_rel_path(dt, 's');
	fprintf(dt->summary_out,
		"    <td align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a></td></tr>\n",
		rel_path, rel_path, (double)n_ok * 100 / n_states);
}

static void out_summary_html_end(struct html_out_data *dt)
{
	fprintf(dt->summary_out,
		"</table>\n" \
		"</body></html>\n");
}

static struct src_info* chk_src_line_type(struct html_out_data*, const char*,
					  long, char);
static char* get_src_html_path(struct html_out_data*, struct src_info*, int);

void out_func_html_start(struct html_out_data *dt)
{
	char title[MAX_LINE_LEN + 1];

	open_html(dt);
	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s function coverage", dt->name);
	out_html_normal_header(dt->cur_out, title);
	fprintf(dt->cur_out,
		"<body>\n" \
		"<table summary=\"%s\" border=\"1\"><tr><th>function</th><th>count</th></tr>\n",
		title);
}

void out_func_html_each(struct html_out_data *dt, struct range_to_name *r2n,
			unsigned long addr, long cnt)
{
	const char *src, *func, *tmp;
	int ln, rc;
	size_t offset;
	struct src_info *info;
	char *color, *ref;
	char type;

	if (cnt) {
		type = T_EXECUTED;
		color = COLOR_OK;
	} else {
		type = T_NOT_EXECUTED;
		color = COLOR_NT;
	}
	fprintf(dt->cur_out, "<tr><td bgcolor=\"%s\">", color);
	rc = addr_to_func_name_and_offset(&r2n->bi, addr, &func, &offset);
	if (rc >= 0) {
		rc = get_source_info(&r2n->bi, addr, &src, &tmp, &ln);
		if (src && (info = chk_src_line_type(dt, src, ln, type))) {
			ref = get_src_html_path(dt, info, 1);
			if (offset)
				fprintf(dt->cur_out,
					"<a href=\"%s#%d\" target=\"src\">%s+0x%x</a></td><td align=\"right\">",
					ref, ln, func, offset);
			else
				fprintf(dt->cur_out,
					"<a href=\"%s#%d\" target=\"src\">%s</a></td><td align=\"right\">",
					ref, ln, func);
		} else {
			if (offset)
				fprintf(dt->cur_out,
					"%s+0x%x</td><td align=\"right\">",
					func, offset);
			else
				fprintf(dt->cur_out,
					"%s</td><td align=\"right\">", func);
		}
	} else
		fprintf(dt->cur_out,
			"0x%08lx</td><td align=\"right\">", addr);
	fprintf(dt->cur_out, "%ld</td></tr>\n", cnt);
}

void out_func_html_end(struct html_out_data *dt)
{
	fprintf(dt->cur_out,
		"</table>\n" \
		"</body></html>\n");
	close_html(dt);
}

void out_branch_html_start(struct html_out_data *dt)
{
	char title[MAX_LINE_LEN + 1];

	open_html(dt);
	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s branch coverage", dt->name);
	out_html_normal_header(dt->cur_out, title);
	fprintf(dt->cur_out,
		"<body>\n" \
		"<table summary=\"%s\" border=\"1\"><tr>\n" \
		"  <th rowspan=\"2\">base</th>\n" \
		"  <th colspan=\"2\">branch</th>\n" \
		"  <th colspan=\"2\">next</th></tr>\n" \
		"  <tr>\n" \
		"  <th>address</th><th>cnt</th>\n" \
		"  <th>address</th><th>cnt</th>\n" \
		"  </tr>\n", title);
}

static struct src_info* chk_branch_point(struct html_out_data *dt,
					 struct range_to_name *r2n,
					 unsigned long addr, long cnt,
					 char **__src, int *ln)
{
	const char *src, *func;
	char type;
	int rc;

	*__src = NULL;
	rc = get_source_info(&r2n->bi, addr, &src, &func, ln);
	if (rc >= 0 && src) {
		*__src = strdup(src);
		type = cnt ? T_EXECUTED : T_NOT_EXECUTED;
		return chk_src_line_type(dt, src, *ln, type);
	}
	return NULL;
}

void out_branch_html_each(struct html_out_data *dt, struct range_to_name *r2n,
			  unsigned long base, unsigned long branch, long b_cnt,
			  unsigned long fall, long f_cnt)
{
	char *src, *color, *ref = "";
	int ln;
	struct src_info *info;

	color = COLOR_NT;
	if (b_cnt && f_cnt)
		color = COLOR_OK;
	else if (b_cnt || f_cnt)
		color = COLOR_HT;

	info = chk_branch_point(dt, r2n, base, b_cnt + f_cnt, &src, &ln);
	if (info) {
		ref = get_src_html_path(dt, info, 1);
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\"><a href=\"%s#%d\" target=\"src\">%s,%d</a></td>\n",
			color, ref, ln, basename(src), ln);
		free(src);
	} else if (src && ln) {
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\">%s,%d</td>\n",
			color, basename(src), ln);
	} else {
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\">0x%08lx</td>\n", color, base);
	}
	if (branch == UNKNOWN_BADDR)
		fprintf(dt->cur_out,
			"    <td>----------</td><td align=\"right\">%ld</td>\n",
			b_cnt);
	else {
		chk_branch_point(dt, r2n, branch, b_cnt, &src, &ln);
		if (src && ln) {
			fprintf(dt->cur_out,
				"    <td>%s,%d</td><td align=\"right\">%ld</td>\n",
				basename(src), ln, b_cnt);
			free(src);
		} else {
			fprintf(dt->cur_out,
				"    <td>0x%08lx</td><td align=\"right\">%ld</td>\n",
				branch, b_cnt);
		}
	}
	if (fall == UNKNOWN_BADDR)
		fprintf(dt->cur_out,
			"    <td></td><td align=\"right\"></td></tr>\n");
	else {
		chk_branch_point(dt, r2n, fall, f_cnt, &src, &ln);
		if (src && ln) {
			fprintf(dt->cur_out,
				"    <td>%s,%d</td><td align=\"right\">%ld</td></tr>\n",
				basename(src), ln, f_cnt);
			free(src);
		} else {
			fprintf(dt->cur_out,
				"    <td>0x%08lx</td><td align=\"right\">%ld</td></tr>\n",
				fall, f_cnt);
		}
	}
}

void out_branch_html_end(struct html_out_data *dt)
{
	fprintf(dt->cur_out,
		"</table>\n" \
		"</body></html>\n");
	close_html(dt);
}

void out_state_html_start(struct html_out_data *dt)
{
	char title[MAX_LINE_LEN + 1];

	open_html(dt);
	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s state coverage", dt->name);
	out_html_normal_header(dt->cur_out, title);
	fprintf(dt->cur_out,
		"<body>\n" \
		"<table summary=\"%s\" border=\"1\"><tr><th>state</th><th>exec</th></tr>\n",
		title);
}

void out_state_html_each(struct html_out_data *dt, struct range_to_name *r2n,
			 int is_exec, unsigned long addr)
{
	int ln;
	const char *src, *func;
	struct src_info *info;
	char *color, *mark, *ref, type, *p;

	if (is_exec) {
		type = T_EXECUTED;
		color = COLOR_OK;
		mark = "Y";
	} else {
		type = T_NOT_EXECUTED;
		color = COLOR_NT;
		mark = "N";
	}
	get_source_info(&r2n->bi, addr, &src, &func, &ln);
	if (ln && (info = chk_src_line_type(dt, src, ln, type))) {
		ref = get_src_html_path(dt, info, 1);
		p = strdup(src);
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\"><a href=\"%s#%d\" target=\"src\">%s,%d</a></td><td align=\"center\">%s</td></tr>\n",
			color, ref, ln, basename(p), ln, mark);
		free(p);
	} else if (src && ln) {
		p = strdup(src);
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\">%s,%d</td><td align=\"center\">%s</td></tr>\n",
			color, basename(p), ln, mark);
		free(p);
	} else {
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\">0x%08lx</td><td align=\"center\">%s</td></tr>\n",
			color, addr, mark);
	}
}

static char* get_src_html_path(struct html_out_data *dt, struct src_info *info,
			       int is_ref)
{
	int rc;

	if (info->html_out_path[0] != '\0' && info->is_ref == is_ref)
		return info->html_out_path;
	if (is_ref)
		rc = snprintf(info->html_out_path, PATH_MAX,
			      "../src/%s.html", info->path + 1);
	else
		rc = snprintf(info->html_out_path, PATH_MAX,
			      "%s/src/%s.html", dt->outdir, info->path + 1);
	if (rc > PATH_MAX)
		fprintf(stderr, "WARN: PATH_MAX is too short.\n");
	info->is_ref = is_ref;
	return info->html_out_path;
}

static struct src_info* chk_src_line_type(struct html_out_data *dt,
					  const char *src, long ln, char type)
{
	FILE *f = NULL;
	char path[PATH_MAX + 1];
	struct src_info *info = NULL;
	long i;
	char buf[MAX_LINE_LEN + 1];

	if (!ln)
		return NULL;

	/* get absolute path */
	if (src[0] == '/')
		strcpy(path, src);
	else {
		if (!dt->srcdir)
			return NULL;
		sprintf(path, "%s/%s", dt->srcdir, src);
	}

	/* search into src_info structures */
	for (i = src_info_num - 1; i >= 0; i--) {
		info = src_info[i];
		if (strcmp(info->path, path) == 0) {
			if (ln > info->ln_max)
				return NULL;
			goto CHK_LINE_TYPE;
		}
	}
	/* check source file existance and max line number */
	f = fopen(path, "r");
	if (!f)
		return NULL;
	for (i = 0; fgets(buf, MAX_LINE_LEN, f); i++);
	fclose(f);
	if (ln > i)
		return NULL;

	info = xcalloc(1, sizeof(struct src_info));
	strcpy(info->path, path);
	info->ln_max = i;
	info->exec_types = xcalloc(i, sizeof(char));
	if (src_info_num) {
		src_info = xrealloc(src_info, (src_info_num + 1) *
				    		sizeof(struct src_info*));
	} else {
		src_info = xmalloc(sizeof(struct src_info*));
	}
	src_info[src_info_num++] = info;
CHK_LINE_TYPE:
	if (info->exec_types[ln - 1] == T_NONE ||
	    (info->exec_types[ln - 1] != type && type == T_EXECUTED))
		info->exec_types[ln - 1] = type;
	return info;
}

static char escape_need_chars[] = "\t<>&\"";
static char *escape_htmls[] = {
	"        ",
	"&lt;",
	"&gt;",
	"&amp;",
	"&quot;",
};

static void escape_for_html(char *buf)
{
	int len_max = strlen(buf), len, n;
	char *from, *to, *p, tmp[MAX_LINE_LEN + 1];

	tmp[MAX_LINE_LEN] = '\0';
	len = 0;
	for (from = buf, to = tmp; from - buf < len_max;) {
		p = strchr(escape_need_chars, *from);
		if (p) {
			from++;
			p = escape_htmls[p - escape_need_chars];
			n = strlen(p);
			if (len + n > MAX_LINE_LEN) {
				n = MAX_LINE_LEN - (len + n);
				snprintf(to, n, "%s", p);
			} else
				sprintf(to, "%s", p);
			to += n;
			len += n;
		} else {
			*to++ = *from++;
			len++;
		}
	}
	*to = '\0';
	strcpy(buf, tmp);
}

static int get_ln_cols(struct src_info *info)
{
	int cols = 0;
	long left;

	for (left = info->ln_max + 1; left; left = left / 10)
		cols++;
	return cols;
}

static char* type2color(char type)
{
	switch (type) {
	case T_NOT_EXECUTED:
		return COLOR_NT;
	case T_EXECUTED:
		return COLOR_OK;
	default:
		return "";
	}
}

static void out_src_html(struct html_out_data *dt)
{
	long i, ln;
	struct src_info *info;
	int ln_cols;
	FILE *r, *w;
	char type, *p, *path, buf[MAX_LINE_LEN + 1];

	buf[MAX_LINE_LEN] = '\0';
	for (i = 0; i < src_info_num; i++) {
		info = src_info[i];
		r = fopen(info->path, "r");
		if (!r) {
			fprintf(stdout, "%s can't open.(%s)\n",
				info->path, strerror(errno));
			break;
		}
		path = get_src_html_path(dt, info, 0);
		p = strdup(path);
		if (dir_chk_and_create(dirname(p), 0) < 0) {
			fclose(r);
			break;
		}
		free(p);
		w = fopen(path, "w");
		if (!w) {
			fprintf(stdout, "%s can't open.(%s)\n",
				path, strerror(errno));
			break;
		}
		ln_cols = get_ln_cols(info);
		out_html_normal_header(w, "src");
		fprintf(w, "<body style=\"font-size:8pt\"><pre>\n");
		type = T_NONE;
		for (ln = 0; fgets(buf, MAX_LINE_LEN, r); ln++) {
			//printf("%*ld: %d\n", ln_cols, ln + 1, info->exec_types[ln]);
			fprintf(w, "%*ld: ", ln_cols, ln + 1);
			if (info->exec_types[ln] != T_NONE)
				type = info->exec_types[ln];
			buf[strlen(buf) - 1] = '\0';
			escape_for_html(buf);
			fprintf(w, "<a name=\"%ld\" style=\"background:%s\">%s</a>\n",
				ln + 1, type2color(type), buf);

			/* '}' of a line-head is regarded as the end of a
			 * function.
			 */
			if (buf[0] == '}')
				type = T_NONE;
		}
		fprintf(w, "</pre></body></html>\n");
		fclose(r);
		fclose(w);
	}
}

/*-----------------------------------------------------------------------------
 *  initialize
 *-----------------------------------------------------------------------------
 */
int dir_chk_and_create(char *path, int err_on_exists)
{
	struct stat st;
	char buf[MAX_LINE_LEN + 1];

	if (stat(path, &st) < 0) {
		if (errno == ENOENT) {
			buf[MAX_LINE_LEN] = '\0';
			snprintf(buf, MAX_LINE_LEN, "mkdir -p \"%s\"", path);
			if (system(buf) < 0) {
				fprintf(stderr, "%s can't create.\n", path);
				return -1;
			}
			return 0;
		}
		fprintf(stderr, "%s can't get stat.(%s)\n",
			path, strerror(errno));
		return -1;
	}
	if (err_on_exists) {
		fprintf(stderr, "%s already exists.\n", path);
		return -1;
	}
	if (!S_ISDIR(st.st_mode)) {
		fprintf(stderr, "%s is not directory.\n", path);
		return -1;
	}
	return 0;
}

int init_html_output(struct html_out_data *dt)
{
	char path[PATH_MAX + 1];

	path[PATH_MAX] = '\0';
	snprintf(path, PATH_MAX, "%s/each", dt->outdir);
	if (dir_chk_and_create(path, 1) < 0)
		return -1;

	path[PATH_MAX] = '\0';
	snprintf(path, PATH_MAX, "%s/summary.html", dt->outdir);
	dt->summary_out = fopen(path, "w");
	if (!dt->summary_out) {
		fprintf(stderr, "%s can't open.(%s)\n", path,
			strerror(errno));
		return -1;
	}
	if (out_top_html(dt) < 0)
		return -1;
	if (out_title_html(dt) < 0)
		return -1;
	out_summary_html_start(dt);
	return 0;
}

int exit_html_output(struct html_out_data *dt)
{
	out_src_html(dt);
	out_summary_html_end(dt);
	if (dt->summary_out)
		fclose(dt->summary_out);
	return 0;
}

