/*
 call table redirection
 Copyright (c) 2005,2007 Hitachi,Ltd.,
 Created by Satoru Moriya <satoru.moriya.br@hitachi.com>
 Updated by Masami Hiramatsu <masami.hiramatsu.pt@hitachi.com>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <linux/module.h>
#include <linux/mm.h>
#include <linux/highmem.h>
#include <asm/unistd.h>
#include "ctr.h"

#define KSTRAX_PAGE_RO 0
#define KSTRAX_PAGE_RW 1

/*
 *  global variable
 */
static syscall_handler_t pre_handler = NULL;
static syscall_handler_t post_handler = NULL;
static call_table_t *true_sys_call_addr = NULL;
call_table_t true_sys_call_table[512] = { NULL, };

/*
 *  global variable and parameter
 */
static unsigned long addr;
module_param(addr, ulong, 0444);
MODULE_PARM_DESC(addr, "Address of system call table(sys_call_table).");

extern int sys_call_stub(void);

#ifdef CONFIG_DEBUG_RODATA

static void change_kernel_linear_page_attr(int is_rw,
					   call_table_t *addr)
{
	int i;
	struct page *p, *p_processed = NULL;
	pgprot_t prot = is_rw ? PAGE_KERNEL : PAGE_KERNEL_RO;

	for (i = 0; i < NR_syscalls; i++) {
		p = virt_to_page(&addr[i]);
		if (p == p_processed)
			continue;
		change_page_attr(p, 1, prot);
		p_processed = p;
	}
	global_flush_tlb();
}


#else  /* CONFIG_DEBUG_RODATA */

static inline void change_kernel_linear_page_attr(int is_rw,
						  call_table_t *addr)
{
}

#endif /* CONFIG_DEBUG_RODATA */

void pre_sys_call(const ctr_arg_t args)
{
	if (pre_handler)
		pre_handler(args);
}

void post_sys_call(const ctr_arg_t args)
{
	if (post_handler)
		post_handler(args);
}

static int change_sys_call_table(call_table_t *saddr)
{
	int i;
	
	if (true_sys_call_addr != NULL)
		return -1;
	true_sys_call_addr = saddr;
	change_kernel_linear_page_attr(KSTRAX_PAGE_RW, true_sys_call_addr);
	for (i = 0; i < NR_syscalls; i++) {
		true_sys_call_table[i] = saddr[i];
		saddr[i] = (call_table_t)sys_call_stub;
	}
	change_kernel_linear_page_attr(KSTRAX_PAGE_RO, true_sys_call_addr);
	return 0;
}

static void restore_sys_call_table(void)
{
	int i;
	call_table_t *sys_call_table;

	if (true_sys_call_addr == NULL)
		return;
	
	change_kernel_linear_page_attr(KSTRAX_PAGE_RW, true_sys_call_addr);
	sys_call_table = (void *)true_sys_call_addr;

	for (i = 0; i < NR_syscalls; i++)
		sys_call_table[i] = true_sys_call_table[i];

	change_kernel_linear_page_attr(KSTRAX_PAGE_RO, true_sys_call_addr);
	true_sys_call_addr = NULL;
}

/*------------------------------------------------------------------------
 *  module interface
 */
int register_ctr(syscall_handler_t pre, syscall_handler_t post)
{
	if (pre == NULL && post == NULL)
		return -EINVAL;
	pre_handler = pre;
	post_handler = post;
	return 0;
}

void unregister_ctr(void)
{
	pre_handler = NULL;
	post_handler = NULL;
}

/*------------------------------------------------------------------------
 *  module init/exit
 */
static void exit_ctr(void)
{
	restore_sys_call_table();
}

static int __init init_ctr(void)
{
	/* check parameter */
	if ((void *)addr == NULL) {
		printk(KERN_ERR "error(init):Invalid parameter\n");
		return -EINVAL;
	}

	/* call table redirection */
	if (change_sys_call_table((void *)addr) != 0) {
		exit_ctr();
		printk(KERN_ERR "error(init):Failed to change system call table\n");
		return -EINVAL;
	}
	try_module_get(THIS_MODULE);

	return 0;
}

module_init(init_ctr);
module_exit(exit_ctr);
MODULE_AUTHOR("S.Moriya <satoru.moriya.br@hitachi.com>");
MODULE_LICENSE("GPL");
EXPORT_SYMBOL_GPL(register_ctr);
EXPORT_SYMBOL_GPL(unregister_ctr);
