/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_hconv.c - coverage output to html converter                           */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <sys/stat.h>
#include "bt_ar_parse.h"
#include "bt_hconv.h"

/*
 * When the object contains debugging information, we need to create the source
 * code html.
 * The source line number of the result of the coverage check has the
 * possibility of appearing at random when it compiled with optimization.
 * Moreover, there is a possibility that two or more branches are included in
 * the same source line, too.
 * Therefore, we first create the execute-type per line array, then create the
 * source html by merging it with source file.
 */

#define T_NONE		0
#define T_NOT_EXECUTED	1
#define T_EXECUTED	2
#define T_UNKNOWN	3

/*-----------------------------------------------------------------------------
 *  output html
 *-----------------------------------------------------------------------------
 */
#define TREE_HTML_PATH	"each/__btrax_ftree__.html"

#define COLOR_OK	"#00ff00"
#define COLOR_HT	"#ffff00"
#define COLOR_NT	"#ff0000"
#define COLOR_ALREADY	"#a9a9a9"
#define COLOR_UN	COLOR_HT

#define COLOR_BOTH_OK	"#ddffdd"
#define COLOR_BOTH_HT	"#fffacd"
#define COLOR_BOTH_NT	"#ffcccc"
#define COLOR_DBL_NAME	"#999999"

#define JS_TREE_MENU	"tree_menu.js"
#define JS_BOTH_OPEN	"both_open_and_scrl.js"

#define BLANK_SRC1	"src/__s1.html"
#define BLANK_SRC2	"src/__s2.html"

enum {
	HEAD_TYPE_NORMAL,
	HEAD_TYPE_JS,
};

typedef int (*f_out_html_file)(struct cov_out_data *dt, FILE *f);

static int out_html_file(struct cov_out_data *dt, char *path,
			 f_out_html_file func)
{
	char buf[PATH_MAX + 1];
	FILE *f;
	int rc;

	buf[PATH_MAX] = '\0';
	snprintf(buf, PATH_MAX, "%s/%s", dt->outdir, path);
	f = fopen(buf, "w");
	if (!f) {
		fprintf(stderr, "%s can't open.(%s)\n", buf, strerror(errno));
		return -1;
	}
	rc = func(dt, f);
	if (rc < 0)
		return rc;
	fclose(f);
	return 0;
}

static int __out_tree_menu_js(struct cov_out_data *dt, FILE *f)
{
	fprintf(f,
		"function treeMenu(tName) {\n" \
		"\ttMenu = document.all[tName].style;\n" \
		"\tif (tMenu.display == 'none') tMenu.display = 'block';\n" \
		"\telse tMenu.display = 'none'\n" \
		"}\n");
	return 0;
}
#define out_tree_menu_js(dt) \
	out_html_file((dt), JS_TREE_MENU, __out_tree_menu_js)

static int __out_both_open_and_scrl_js(struct cov_out_data *dt, FILE *f)
{
	fprintf(f,
		"function bothOpen(x1, x2) {\n" \
	        "\twindow.open(x1, \"s1\");\n" \
	        "\twindow.open(x2, \"s2\");\n" \
	        "\tparent.s1.startSetTop();\n" \
	        "\tparent.s2.startSetTop();\n" \
	        "}\n" \
		"function syncScroll(e) {\n" \
	        "\tif (e.charCode == 85 || e.charCode == 117) {\n" \
	        "\t\tparent.s1.scrollBy(0,10);\n" \
	        "\t\tparent.s2.scrollBy(0,10);\n" \
	        "\t} else if (e.charCode == 68 || e.charCode == 100) {\n" \
	        "\t\tparent.s1.scrollBy(0,-10);\n" \
	        "\t\tparent.s2.scrollBy(0,-10);\n" \
	        "\t}\n" \
	        "}\n");
	return 0;
}
#define out_both_open_and_scrl_js(dt) \
	out_html_file((dt), JS_BOTH_OPEN, __out_both_open_and_scrl_js)

static void out_html_header(FILE *f, char *doctype, char *title, int type,
			    char *js_path)
{
	fprintf(f,
		"%s\n" \
		"<html lang=\"en\">\n" \
		"<head>\n" \
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=ISO-8859-1\">\n" \
		"<meta http-equiv=\"Content-Style-Type\" content=\"text/css\">\n",
		doctype);
	if (type == HEAD_TYPE_JS) {
		fprintf(f,
			"<meta http-equiv=\"Content-Script-Type\" content=\"text/css\">\n" \
			"<script type=\"text/javascript\" src=\"%s\"></script>\n",
			js_path);
	}
	fprintf(f,
		"<title>%s</title>\n" \
		"</head>\n",
		title);
}

static void out_html_frame_header(FILE *f, char *title)
{
	out_html_header(f,
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\">",
			title, HEAD_TYPE_NORMAL, NULL);
}

static void out_html_normal_header(FILE *f, char *title, int type,
				   char *js_path)
{
	out_html_header(f,
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">",
			title, type, js_path);
}

static int __out_top_html(struct cov_out_data *dt, FILE *f)
{
	out_html_frame_header(f, "BTRAX - COVERAGE RESULT");
	fprintf(f,
		"<frameset title=\"all\" rows=\"40,*\">\n" \
		"  <frame title=\"title\" src=\"./title.html\" scrolling=\"no\" noresize>\n");
	if (dt->chk_type == CHK_DIFF_KERNEL)
		fprintf(f,
			"  <frameset title=\"right\" cols=\"30%%,*\">\n" \
			"    <frameset title=\"left\" rows=\"40%%,*\">\n" \
			"      <frame title=\"summary\" name=\"summary\" src=\"./summary.html\">\n" \
			"      <frame title=\"each\" name=\"each\">\n" \
			"    </frameset>\n" \
			"    <frameset title=\"src\" cols=\"50%%,*\">\n" \
			"      <frame title=\"s1\" name=\"s1\">\n" \
			"      <frame title=\"s2\" name=\"s2\" noresize>\n" \
			"    </frameset>\n" \
			"  </frameset>\n");
	else
		fprintf(f,
			"  <frameset title=\"right\" cols=\"50%%,*\">\n" \
			"    <frameset title=\"left\" rows=\"40%%,*\">\n" \
			"      <frame title=\"summary\" name=\"summary\" src=\"./summary.html\">\n" \
			"      <frame title=\"each\" name=\"each\">\n" \
			"    </frameset>\n" \
			"    <frame title=\"src\" name=\"src\">\n" \
			"  </frameset>\n");
	fprintf(f,
		"  <noframes><body><p>use frame supported browser</p></body></noframes>\n" \
		"</frameset>\n" \
		"</html>\n");
	return 0;
}
#define out_top_html(dt) \
	out_html_file((dt), "top.html", __out_top_html)

static int __out_title_html(struct cov_out_data *dt, FILE *f)
{
	out_html_normal_header(f, "title", HEAD_TYPE_NORMAL, NULL);
	fprintf(f,
		"<body style=\"background-color:silver\">\n" \
		"<p style=\"font-size:16pt; font-weight:bold; text-align:center\">\n" \
		"<span style=\"color:#0000ff\">BTRAX</span> Coverage\n" \
		"<span style=\"color:#ff0000\">R</span>e" \
		"<span style=\"color:#ffff00\">s</span>u" \
		"<span style=\"color:#00ff00\">l</span>t\n" \
		"</p>\n" \
		"</body></html>\n");
	return 0;
}
#define out_title_html(dt) \
	out_html_file((dt), "title.html", __out_title_html)

static int __out_blank_html(struct cov_out_data *dt, FILE *f, int kern)
{
	out_html_normal_header(f, "title", HEAD_TYPE_NORMAL, NULL);
	fprintf(f,
		"<body style=\"font-size:8pt\"><pre>\n"\
		"</pre>\n" \
		"</body></html>\n");
	return 0;
}
static int __out_blank_html1(struct cov_out_data *dt, FILE *f)
{
	return __out_blank_html(dt, f, 0);
}
static int __out_blank_html2(struct cov_out_data *dt, FILE *f)
{
	return __out_blank_html(dt, f, 1);
}
#define out_blank_html1(dt) \
	out_html_file((dt), BLANK_SRC1, __out_blank_html1)
#define out_blank_html2(dt) \
	out_html_file((dt), BLANK_SRC2, __out_blank_html2)

static char* get_rel_path(struct cov_out_data *dt, int kern, char cov_type)
{
	if (cov_type == 'f' || dt->chk_type != CHK_DIFF_KERNEL)
		snprintf(dt->abs_path, PATH_MAX, "%s/each/%s.%c.html",
			 dt->outdir, dt->name, cov_type);
	else
		snprintf(dt->abs_path, PATH_MAX, "%s/each/%s.%c.%s.html",
			 dt->outdir, dt->name, cov_type,
			 dt->r2p[kern]->r2i.uname_r);
	return dt->abs_path + strlen(dt->outdir) + 1;
}

static void open_html(struct cov_out_data *dt)
{
	dt->cur_out = fopen(dt->abs_path, "w");
	if (!dt->cur_out) {
		fprintf(stderr, "%s can't open.(%s)\n",
			dt->abs_path, strerror(errno));
		exit(1);
	}
}

static void close_html(struct cov_out_data *dt)
{
	if (dt->cur_out) {
		fclose(dt->cur_out);
		dt->cur_out = NULL;
	}
}

static void out_summary_html_start(struct cov_out_data *dt)
{
	out_html_normal_header(dt->summary_out, "summary", HEAD_TYPE_NORMAL,
			       NULL);
	fprintf(dt->summary_out, "<body>\n");
	if (dt->chk_type == CHK_SINGLE)
		fprintf(dt->summary_out,
			"<table summary=\"summary\" border=\"1\">\n" \
			"  <tr><th rowspan=\"2\">name</th><th colspan=\"3\">coverage</th></tr>\n" \
			"  <tr><th>function</th><th>branch</th><th>state</th></tr>\n");
	else
		fprintf(dt->summary_out,
			"<b>Coverage compare:<br></b>\n" \
			"<b>L1(%s)</b> and <b>L2(%s)</b><br>\n" \
			"<table summary=\"summary\" border=\"1\">\n" \
			"  <tr><th rowspan=\"3\">name</th><th colspan=\"6\">coverage</th></tr>\n" \
			"  <tr><th colspan=\"2\">function</th><th colspan=\"2\">branch</th><th colspan=\"2\">state</th></tr>\n" \
			"  <tr><th>L1</th><th>L2</th><th>L1</th><th>L2</th><th>L1</th><th>L2</th></tr>\n",
			dt->r2p[0]->r2i.uname_r, dt->r2p[1]->r2i.uname_r);
}

void out_summary_html_name(struct cov_out_data *dt)
{
	fprintf(dt->summary_out, "  <tr><td>%s</td>\n", dt->name);
}

void out_summary_html_func(struct cov_out_data *dt, int kern,
			   long n_func, long n_func_all)
{
	char *rel_path;

	rel_path = get_rel_path(dt, kern, 'f');
	fprintf(dt->summary_out,
		"    <td nowrap align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a><br>(%ld/%ld)</td>\n",
		rel_path, rel_path, get_percent(n_func, n_func_all),
		n_func, n_func_all);
}

void out_summary_html_branch(struct cov_out_data *dt, int kern,
			     long n_br_ok, long n_br_uk,
			     long n_br_ht, long n_br_nt,
			     long n_br_all, long n_uk_all)
{
	char *rel_path;

	rel_path = get_rel_path(dt, kern, 'b');
	if (n_br_all)
		fprintf(dt->summary_out,
			"    <td nowrap align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%(%.2f%%)</a><br>(OK:%ld,HT:%ld,NT:%ld/%ld (UK:%ld/%ld))</td>\n",
			rel_path, rel_path,
			get_percent(n_br_ok * 2 + n_br_ht, n_br_all),
			get_percent(n_br_uk, n_uk_all),
			n_br_ok, n_br_ht, n_br_nt, n_br_all,
			n_br_uk, n_uk_all);
	else
		fprintf(dt->summary_out,
			"    <td nowrap align=\"right\">----</td>\n");
}

void out_summary_html_state(struct cov_out_data *dt, int kern,
			    long n_ok, long n_states)
{
	char *rel_path;

	rel_path = get_rel_path(dt, kern, 's');
	fprintf(dt->summary_out,
		"    <td nowrap align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a><br>(%ld/%ld)</td>",
		rel_path, rel_path, get_percent(n_ok, n_states),
		n_ok, n_states);
	if (kern != 0 || dt->chk_type == CHK_SINGLE)
		fprintf(dt->summary_out, "</tr>\n");
}

static void out_summary_html_end(struct cov_out_data *dt, int limit_by_funcs)
{
	fprintf(dt->summary_out, "</table>\n");
	if (limit_by_funcs && dt->chk_type == CHK_SINGLE)
		fprintf(dt->summary_out,
			"<a href=\"%s\" target=\"each\" title=\"ftree\">View function tree.</a>",
			TREE_HTML_PATH);
	fprintf(dt->summary_out, "</body></html>\n");
}

static struct src_info* chk_src_line_type(struct cov_out_data*, int kern,
					  const char*, long, char);
static char* get_src_html_path(struct cov_out_data*, struct src_info*, int);

static int get_fname_and_ref_str(struct cov_out_data *dt, int kern,
				 unsigned long addr, int type,
				 char buf[MAX_LINE_LEN],
				 char **p_fname, char **p_ref)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;
	const char *src, *func, *tmp;
	int ln, rc, len, left;
	size_t offset;
	struct src_info *info;
	char *ref = NULL, *p;

	rc = addr_to_func_name_and_offset(&r2n->bi, addr, &func, &offset);
	if (rc >= 0) {
		rc = get_source_info(&r2n->bi, addr, &src, &tmp, &ln);
		if (src && ln &&
		    (info = chk_src_line_type(dt, kern, src, ln, type)))
			ref = get_src_html_path(dt, info, 1);
		if (offset)
			len = snprintf(buf, MAX_LINE_LEN, "%s+0x%x",
				       func, offset);
		else
			len = snprintf(buf, MAX_LINE_LEN, "%s", func);
	} else
		len = snprintf(buf, MAX_LINE_LEN, "0x%08lx", addr);
	if (len >= MAX_LINE_LEN)
		goto ERR;
	*p_fname = buf;
	*p_ref = NULL;
	if (!ref)
		return 0;
	p = buf + len + 1;
	left = MAX_LINE_LEN - len - 1;
	len = snprintf(p, left, "%s#%d", ref, ln);
	if (len >= left)
		goto ERR;
	*p_ref = p;
	return 0;
ERR:
	fprintf(stderr, "MAX_LINE_LEN too short.\n");
	return -1;
}

static void get_func_each_init_value(struct cov_out_data *dt,
				     struct func_chk *fc, struct func_chk *fc2,
				     unsigned long *addr, unsigned long *addr2,
				     unsigned long *cnt, unsigned long *cnt2,
				     int *is_double_name)
{
	if (fc) {
		*addr = fc->addr - dt->r2p[0]->r2n->offset;
		*cnt = fc->cnt;
		*is_double_name = IS_DOUBLE_NAME_FC(fc);
	} else {
		*addr = UNKNOWN_BADDR;
		*cnt = 0;
	}
	if (fc2) {
		*addr2 = fc2->addr - dt->r2p[1]->r2n->offset;
		*cnt2 = fc2->cnt;
		*is_double_name = IS_DOUBLE_NAME_FC(fc2);
	} else {
		*addr2 = UNKNOWN_BADDR;
		*cnt2 = 0;
	}
}

void __out_func_html_each(struct cov_out_data *dt,
			  struct func_chk *fc, struct func_chk *fc2,
			  long *same, long *diff)
{
	FILE *out = dt->cur_out;
	char *color, buf[2][MAX_LINE_LEN], *p_fname, *p_ref, *p_ref2;
	int is_exec, is_exec2, type, type2 = T_NONE, is_double_name = 0;
	unsigned long addr, addr2, cnt, cnt2;

	get_func_each_init_value(dt, fc, fc2, &addr, &addr2, &cnt, &cnt2,
				 &is_double_name);
	is_exec = cnt != 0;
	type = is_exec ? T_EXECUTED : T_NOT_EXECUTED;
	if (dt->chk_type == CHK_SINGLE)
		color = is_exec ? COLOR_OK : COLOR_NT;
	else {
		is_exec2 = cnt2 != 0;
		if (same && diff) {
			if (is_exec != is_exec2)
				(*diff)++;
			else
				(*same)++;
			return;
		}
		if (is_double_name)
			color = COLOR_DBL_NAME;
		else if (is_exec == is_exec2)
			color = is_exec2 ? COLOR_BOTH_OK : COLOR_BOTH_NT;
		else
			color = is_exec2 ? COLOR_OK : COLOR_NT;
		type2 = is_exec2 ? T_EXECUTED : T_NOT_EXECUTED;
	}
	p_ref = p_ref2 = NULL;
	if (fc) {
		if (get_fname_and_ref_str(dt, 0, addr, type, buf[0],
					  &p_fname, &p_ref) < 0)
			return;
	}
	if (dt->chk_type != CHK_SINGLE && fc2) {
		if (get_fname_and_ref_str(dt, 1, addr2, type2, buf[1], &p_fname,
					  &p_ref2) < 0)
			return;
	}
	fprintf(out, "<tr><td bgcolor=\"%s\">", color);
	if (dt->chk_type == CHK_DIFF_KERNEL) {
		if (!p_ref && !p_ref2)
			fprintf(out, "%s", p_fname);
		else
			fprintf(out, "<a href=\"javascript:bothOpen('%s', '%s')\">%s</a>",
				p_ref ? p_ref : "../" BLANK_SRC1,
				p_ref2 ? p_ref2 : "../" BLANK_SRC2,
			       	p_fname);
	} else {
		if (p_ref)
			fprintf(out, "<a href=\"%s\" target=\"src\">%s</a>",
				p_ref, p_fname);
		else
			fprintf(out, "%s", p_fname);
	}
	fprintf(out, "</td>");
	if (fc)
		fprintf(out, "<td align=\"right\">%ld</td>", cnt);
	else
		fprintf(out, "<td align=\"right\"></td>");
	if (dt->chk_type == CHK_SINGLE) {
		if (dt->limit_by_funcs)
			fprintf(out, "<td align=\"right\">%ld</td>",
				fc->excluded_tree_cnt);
	} else {
		if (fc2)
			fprintf(out, "<td align=\"right\">%ld</td>", cnt2);
		else
			fprintf(out, "<td align=\"right\"></td>");
	}
	fprintf(out, "</tr>\n");
}

void out_func_html_each(struct cov_out_data *dt, struct func_chk *fc)
{
	__out_func_html_each(dt, fc, NULL, NULL, NULL);
}

void out_func_html_each2(struct cov_out_data *dt,
			 struct func_chk *fc1, struct func_chk *fc2,
			 long *same, long *diff)
{
	__out_func_html_each(dt, fc1, fc2, same, diff);
}

void out_func_html_end(struct cov_out_data *dt)
{
	fprintf(dt->cur_out,
		"</table>\n" \
		"</body></html>\n");
	close_html(dt);
}

static inline void out_diff_same_diff(struct cov_out_data *dt,
				      unsigned long same, unsigned long diff)
{
	fprintf(dt->cur_out, "<b>same:</b> %.2f%%, <b>diff:</b> %.2f%%<br>\n",
		get_percent(same, same + diff), get_percent(diff, same + diff));
}

void out_branch_html_start(struct cov_out_data *dt, long same, long diff)
{
	char title[MAX_LINE_LEN + 1];

	open_html(dt);
	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s branch coverage", dt->name);
	out_html_normal_header(dt->cur_out, title, HEAD_TYPE_NORMAL, NULL);
	fprintf(dt->cur_out, "<body>\n");
	if (dt->chk_type == CHK_SAME_KERNEL) {
		out_diff_same_diff(dt, same, diff);
		fprintf(dt->cur_out,
			"<table summary=\"%s\" border=\"1\"><tr>\n" \
			"  <th rowspan=\"3\">base</th>\n" \
			"  <th colspan=\"3\">branch</th>\n" \
			"  <th colspan=\"3\">next</th></tr>\n" \
			"  <tr>\n" \
			"  <th rowspan=\"2\">address</th><th colspan=\"2\">cnt</th>\n" \
			"  <th rowspan=\"2\">address</th><th colspan=\"2\">cnt</th>\n" \
			"  <tr>\n" \
			"  <th>L1</th><th>L2</th><th>L1</th><th>L2</th>\n" \
			"  </tr>\n", title);
	} else {
		fprintf(dt->cur_out,
			"<table summary=\"%s\" border=\"1\"><tr>\n" \
			"  <th rowspan=\"2\">base</th>\n" \
			"  <th colspan=\"2\">branch</th>\n" \
			"  <th colspan=\"2\">next</th></tr>\n" \
			"  <tr>\n" \
			"  <th>address</th><th>cnt</th>\n" \
			"  <th>address</th><th>cnt</th>\n" \
			"  </tr>\n", title);
	}
}

static struct src_info* chk_branch_point(struct cov_out_data *dt, int kern,
					 unsigned long addr, long cnt,
					 char **__src, int *ln)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;
	const char *src, *func;
	char type;
	int rc;

	*__src = NULL;
	rc = get_source_info(&r2n->bi, addr, &src, &func, ln);
	if (rc >= 0 && src && ln) {
		*__src = strdup(src);
		type = cnt ? T_EXECUTED : T_NOT_EXECUTED;
		return chk_src_line_type(dt, kern, src, *ln, type);
	}
	return NULL;
}

enum {
	BRANCH_INFO_OK,
	BRANCH_INFO_BRANCH,
	BRANCH_INFO_FALLTHROUGH,
	BRANCH_INFO_NT,
};

static int get_bi_type(struct branch_info *bi)
{
	if (bi->b_cnt && bi->f_cnt)
		return BRANCH_INFO_OK;
	else if (bi->b_cnt)
		return BRANCH_INFO_BRANCH;
	else if (bi->f_cnt)
		return BRANCH_INFO_FALLTHROUGH;
	return BRANCH_INFO_NT;
}

static inline char* get_bi_color(struct branch_info *bi)
{
	if (bi->b_cnt && bi->f_cnt)
		return COLOR_OK;
	else if (bi->b_cnt || bi->f_cnt)
		return COLOR_HT;
	return COLOR_NT;
}

static inline char *get_bi2_color(struct branch_info *bi1,
				  struct branch_info *bi2)
{
	int t1, t2;

	if (!bi2)
		return get_bi_color(bi1);
	t1 = get_bi_type(bi1);
	t2 = get_bi_type(bi2);
	if (t1 == t2) {
		switch (t1) {
		case BCOV_TYPE_OK:
			return COLOR_BOTH_OK;
		case BCOV_TYPE_HT:
			return COLOR_BOTH_HT;
		default:
			return COLOR_BOTH_NT;
		}
	} else
		return get_bi_color(bi2);
}

static int is_bi2_diff(struct branch_info *bi1, struct branch_info *bi2)
{
	int t1, t2;

	if (!bi2)
		return 1;
	t1 = get_bi_type(bi1);
	t2 = get_bi_type(bi2);
	return (t1 != t2);
}

#define get_target_frame(dt, kern) \
	(dt)->chk_type == CHK_DIFF_KERNEL ? \
		((kern) == 0 ? "s1" : "s2") : "src"

static inline void __out_branch_html_each(FILE *out, struct cov_out_data *dt,
					  int kern, struct branch_info *bi,
					  struct branch_info *bi2,
					  long *same, long *diff)
{
	char *src, *color, *ref = "";
	int ln;
	struct src_info *info;

	if (same && diff) {
		if (is_bi2_diff(bi, bi2))
			(*diff)++;
		else
			(*same)++;
		return;
	}
	color = get_bi2_color(bi, bi2);
	fprintf(out, "<td bgcolor=\"%s\">", color);
	info = chk_branch_point(dt, kern, bi->base, bi->b_cnt + bi->f_cnt,
				&src, &ln);
	if (info) {
		ref = get_src_html_path(dt, info, 1);
		if (dt->chk_type == CHK_SAME_KERNEL)
			chk_branch_point(dt, 1, bi2->base,
					 bi2->b_cnt + bi2->f_cnt, &src, &ln);
		fprintf(out,
			"<a href=\"%s#%d\" target=\"%s\">%s,%d</a></td>\n",
			ref, ln, get_target_frame(dt, kern), basename(src), ln);
		free(src);
	} else if (src && ln) {
		fprintf(out, "%s,%d</td>\n", basename(src), ln);
	} else {
		fprintf(out, "0x%08lx</td>\n", bi->base);
	}
	if (bi->branch == UNKNOWN_BADDR)
		fprintf(out, "    <td>----------</td>");
	else {
		chk_branch_point(dt, kern, bi->branch, bi->b_cnt, &src, &ln);
		if (src && ln) {
			if (dt->chk_type == CHK_SAME_KERNEL)
				chk_branch_point(dt, 1, bi2->branch, bi2->b_cnt,
						 &src, &ln);
			fprintf(out, "    <td>%s,%d</td>", basename(src), ln);
			free(src);
		} else {
			fprintf(out, "    <td>0x%08lx</td>", bi->branch);
		}
	}
	fprintf(out, "<td align=\"right\">%ld</td>\n", bi->b_cnt);
	if (dt->chk_type == CHK_SAME_KERNEL)
		fprintf(out, "<td align=\"right\">%ld</td>\n", bi2->b_cnt);
	if (bi->fall == UNKNOWN_BADDR)
		fprintf(out, "    <td></td><td align=\"right\"></td>");
	else {
		chk_branch_point(dt, kern, bi->fall, bi->f_cnt, &src, &ln);
		if (src && ln) {
			if (dt->chk_type == CHK_SAME_KERNEL)
				chk_branch_point(dt, 1, bi2->fall, bi2->f_cnt,
						 &src, &ln);
			fprintf(out, "    <td>%s,%d</td>", basename(src), ln);
			free(src);
		} else {
			fprintf(out, "    <td>0x%08lx</td>", bi->fall);
		}
		fprintf(out, "<td align=\"right\">%ld</td>", bi->f_cnt);
		if (dt->chk_type == CHK_SAME_KERNEL)
			fprintf(out, "<td align=\"right\">%ld</td>",
				bi2->f_cnt);
	}
}

void out_branch_html_each(struct cov_out_data *dt, int kern,
			  struct branch_info *bi)
{
	fprintf(dt->cur_out, "<tr>");
	__out_branch_html_each(dt->cur_out, dt, kern, bi, NULL, NULL, NULL);
	fprintf(dt->cur_out, "</tr>\n");
}

void out_branch_html_each2(struct cov_out_data *dt,
			   struct branch_info *bi, struct branch_info *bi2,
			   long *same, long *diff)
{
	if (!(same || diff))
		fprintf(dt->cur_out, "<tr>");
	__out_branch_html_each(dt->cur_out, dt, 0, bi, bi2, same, diff);
	if (!(same || diff))
		fprintf(dt->cur_out, "</tr>\n");
}

void out_branch_html_end(struct cov_out_data *dt)
{
	fprintf(dt->cur_out,
		"</table>\n" \
		"</body></html>\n");
	close_html(dt);
}

void out_state_html_start(struct cov_out_data *dt, long same, long diff)
{
	char title[MAX_LINE_LEN + 1];

	open_html(dt);
	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s state coverage", dt->name);
	out_html_normal_header(dt->cur_out, title, HEAD_TYPE_NORMAL, NULL);
	fprintf(dt->cur_out, "<body>\n");
	if (dt->chk_type == CHK_SAME_KERNEL) {
		out_diff_same_diff(dt, same, diff);
		fprintf(dt->cur_out,
			"<table summary=\"%s\" border=\"1\">" \
			"<tr><th rowspan=\"2\">state</th><th colspan=\"2\">exec</th></tr>\n" \
			"<tr><th>L1</th><th>L2</th></tr>\n", title);
	} else {
		fprintf(dt->cur_out,
			"<table summary=\"%s\" border=\"1\">" \
			"<tr><th>state</th><th>exec</th></tr>\n", title);
	}
}

#define get_state_type_and_mark(is_exec, type, mark) \
	do { \
		if (is_exec) { \
			type = T_EXECUTED; \
			mark = "Y"; \
		} else { \
			type = T_NOT_EXECUTED; \
			mark = "N"; \
		} \
	} while (0)

static inline void __out_state_html_each(FILE *out, struct cov_out_data *dt,
					 int kern, int is_exec, int is_exec2,
					 unsigned long addr,
					 unsigned long call_addr, int uk_cnt)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;
	int ln;
	const char *src, *func;
	struct src_info *info;
	char *color, *mark, *mark2 = "", *ref, type, type2 = T_NONE, *p;

	if (dt->chk_type == CHK_SAME_KERNEL) {
		if (is_exec == is_exec2)
			color = is_exec2 ? COLOR_BOTH_OK : COLOR_BOTH_NT;
		else
			color = is_exec2 ? COLOR_OK : COLOR_NT;
		get_state_type_and_mark(is_exec2, type2, mark2);
		kern = 0;
	} else {
		color = is_exec ? COLOR_OK : COLOR_NT;
	}
	get_state_type_and_mark(is_exec, type, mark);
	if (uk_cnt > 1)
		fprintf(out, "<td bgcolor=\"%s\" rowspan=\"%d\">",
			color, uk_cnt);
	else
		fprintf(out, "<td bgcolor=\"%s\">", color);
	get_source_info(&r2n->bi, addr, &src, &func, &ln);
	if (ln && (info = chk_src_line_type(dt, kern, src, ln, type))) {
		ref = get_src_html_path(dt, info, 1);
		if (dt->chk_type == CHK_SAME_KERNEL)
			chk_src_line_type(dt, 1, src, ln, type2);
		p = strdup(src);
		fprintf(out,
			"<a href=\"%s#%d\" target=\"%s\">%s,%d</a>",
			ref, ln, get_target_frame(dt, kern), basename(p), ln);
		free(p);
	} else if (src && ln) {
		p = strdup(src);
		fprintf(out, "%s,%d", basename(p), ln);
		free(p);
	} else {
		fprintf(out, "0x%08lx", addr);
	}

	/* to display the function call line as 'executed' in html colored
	 * source file.
	 */
	if (call_addr != UNKNOWN_BADDR) {
		get_source_info(&r2n->bi, call_addr, &src, &func, &ln);
		if (ln) {
			chk_src_line_type(dt, kern, src, ln, type);
			if (dt->chk_type == CHK_SAME_KERNEL)
				chk_src_line_type(dt, 1, src, ln, type2);
		}
	}

	fprintf(out, "</td>");
	if (uk_cnt > 1)
		fprintf(out, "<td align=\"center\" rowspan=\"%d\">%s</td>",
			uk_cnt, mark);
	else
		fprintf(out, "<td align=\"center\">%s</td>", mark);
	if (dt->chk_type == CHK_SAME_KERNEL)
		fprintf(out, "<td align=\"center\">%s</td>", mark2);
	return;
}

void out_state_html_each(struct cov_out_data *dt, int kern, int is_exec,
			 struct path *p, unsigned long offset)
{
	unsigned long call_addr;

	call_addr = p->type == BTYPE_CALL ? p->base - offset: UNKNOWN_BADDR;
	fprintf(dt->cur_out, "<tr>");
	__out_state_html_each(dt->cur_out, dt, kern, is_exec, 0,
			      p->addr - offset, call_addr, 0);
	fprintf(dt->cur_out, "</tr>\n");
}

void out_state_html_each2(struct cov_out_data *dt, int is_exec, int is_exec2,
			  struct path *p, unsigned long offset)
{
	unsigned long call_addr;

	call_addr = p->type == BTYPE_CALL ? p->base - offset: UNKNOWN_BADDR;
	fprintf(dt->cur_out, "<tr>");
	__out_state_html_each(dt->cur_out, dt, 0, is_exec, is_exec2,
			      p->addr - offset, call_addr, 0);
	fprintf(dt->cur_out, "</tr>\n");
}

#define TREE_MARGIN_VAL		2.0
#define TREE_CTRL_CHILD_CHAR	'T'
#define TREE_CTRL_COV_CHAR	'C'

void out_func_tree_html_start(struct cov_out_data *dt,
			      char *s_inc, char *s_exc)
{
	char path[PATH_MAX + 1];
	char title[MAX_LINE_LEN + 1];

	path[PATH_MAX] = '\0';
	snprintf(path, PATH_MAX, "%s/%s", dt->outdir, TREE_HTML_PATH);
	dt->ftree_out = fopen(path, "w");
	if (!dt->ftree_out) {
		fprintf(stderr, "%s can't open.(%s)\n", path,
			strerror(errno));
		return;
	}
	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "function tree");
	out_html_normal_header(dt->ftree_out, title, HEAD_TYPE_JS,
			       "../" JS_TREE_MENU);
	fprintf(dt->ftree_out, "<body>\n");
	fprintf(dt->ftree_out, "<b>includes: </b>%s<br>\n", s_inc);
	fprintf(dt->ftree_out, "<b>excludes: </b>%s<br>\n", s_exc);
	fprintf(dt->ftree_out,
		"<a href=\"javascript:treeMenu('cov_format')\">" \
		"View coverage table format</a>\n" \
		"<div id=\"cov_format\" style=\"display:none\">" \
		"<table summary=\"cov_format\" border=\"1\">" \
		"<tr>" \
		"<th colspan=\"2\">state coverage</th>" \
		"<th colspan=\"5\">branch coverage</th>" \
		"</tr>" \
		"<tr>" \
		"<th>state</th><th>exec</th>" \
		"<th>base</th><th>branch</th><th>cnt</th>" \
		"<th>next</th><th>cnt</th>" \
		"</tr>" \
		"</table></div><br><br>\n");
}

static inline void out_ftree_html_each_bcov(struct cov_out_data *dt, int kern,
					    int bcov_type,
					    struct branch_info *bi)
{
	if (bi->uk_id)
		fprintf(dt->ftree_out, "<tr>");
	__out_branch_html_each(dt->ftree_out, dt, kern, bi, NULL, NULL, NULL);
	if (bi->uk_id)
		fprintf(dt->ftree_out, "</tr>\n");
}

static void out_ftree_html_each_cov(struct cov_out_data *dt, int kern,
				    char *fname, struct func_chk *fc, int nest,
				    int type, int has_child)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;
	struct path *p;
	long i;

	fprintf(dt->ftree_out,
		"<div id=\"%s_%c\" style=\"display:none\">\n" \
		"<table summary=\"%s\" border=\"1\" style=\"margin-left: %.1fem\">\n",
		fname, TREE_CTRL_COV_CHAR, fname, TREE_MARGIN_VAL);
	i = r2n->cnt;
	find_path_from_addr(r2n->pt, &i, fc->addr);
	for (; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if (p->next > fc->end)
			break;
		fprintf(dt->ftree_out, "<tr>");
		__out_state_html_each(dt->ftree_out, dt, kern, p->cnt > 0, 0,
				      p->addr - r2n->offset, 0,
				      get_unknown_bcnt(p));
		if (p->type == BTYPE_BRANCH || IS_SWITCH_JMP(p))
			do_one_branch_coverage(dt, kern, p,
					       out_ftree_html_each_bcov);
		else
			fprintf(dt->ftree_out,
				"<td></td><td></td><td></td><td></td><td></td>");
		fprintf(dt->ftree_out, "</tr>\n");
	}
	fprintf(dt->ftree_out, "</table><br></div>");
}

void out_func_tree_html_each_enter(struct cov_out_data *dt, int kern,
				   struct func_chk *fc, int nest, int type,
				   int has_child)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;
	unsigned long addr = fc->addr - r2n->offset;
	char *color, buf[MAX_LINE_LEN], *p_fname, *p_ref;
	int exec_type;

	switch (type) {
	case BCOV_TYPE_OK:
		exec_type = T_EXECUTED;
		color = COLOR_OK;
		break;
	case BCOV_TYPE_NT:
		exec_type = T_NOT_EXECUTED;
		color = COLOR_NT;
		break;
	case BCOV_TYPE_HT:	/* already displayed function */
		exec_type = T_NOT_EXECUTED;
		color = COLOR_ALREADY;
		break;
	default:
		/* not reached */
		return;
	}
	if (get_fname_and_ref_str(dt, kern, addr, exec_type, buf,
				  &p_fname, &p_ref) < 0)
		return;

	fprintf(dt->ftree_out,
		"<div style=\"margin-left: %.1fem\"><nobr>",
		nest ? TREE_MARGIN_VAL : 0);
	if (!has_child)
		fprintf(dt->ftree_out, "%c ", TREE_CTRL_CHILD_CHAR);
	else
		fprintf(dt->ftree_out,
			"<a href=\"javascript:treeMenu('%s_%c')\">%c</a> ",
			p_fname, TREE_CTRL_CHILD_CHAR, TREE_CTRL_CHILD_CHAR);
	if (type == BCOV_TYPE_HT)
		fprintf(dt->ftree_out, "%c ", TREE_CTRL_COV_CHAR);
	else
		fprintf(dt->ftree_out,
			"<a href=\"javascript:treeMenu('%s_%c')\">%c</a> ",
			p_fname, TREE_CTRL_COV_CHAR, TREE_CTRL_COV_CHAR);
	fprintf(dt->ftree_out, "<span style=\"background:%s\">", color);
	if (p_ref)
		fprintf(dt->ftree_out, "<a href=\"%s\" target=\"src\">%s</a>",
			p_ref, p_fname);
	else
		fprintf(dt->ftree_out, "%s", p_fname);
	fprintf(dt->ftree_out, "</span> %ld (F:%ld)</nobr>",
		fc->cnt, fc->excluded_tree_cnt);

	if (type != BCOV_TYPE_HT)
		out_ftree_html_each_cov(dt, kern, p_fname, fc, nest, type,
					has_child);
	if (has_child)
		fprintf(dt->ftree_out,
			"<div id=\"%s_%c\" style=\"display:%s\">\n",
			p_fname, TREE_CTRL_CHILD_CHAR,
			(nest > 0 ? "none" : "block"));
}

void out_func_tree_html_each_exit(struct cov_out_data *dt, int nest,
				  int has_child)
{
	if (has_child)
		fprintf(dt->ftree_out, "</div>");
	fprintf(dt->ftree_out, "</div>\n");
}

void out_func_tree_html_each_invalid(struct cov_out_data *dt, int kern,
				     struct func_chk *fc)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;
	unsigned long addr = fc->addr - r2n->offset;
	char buf[MAX_LINE_LEN], *p_fname, *p_ref;

	if (get_fname_and_ref_str(dt, kern, addr, T_EXECUTED, buf,
				  &p_fname, &p_ref) < 0)
		return;
	fprintf(dt->ftree_out, "UT <span style=\"background:%s\">", COLOR_OK);
	if (p_ref)
		fprintf(dt->ftree_out, "<a href=\"%s\" target=\"src\">%s</a>",
			p_ref, p_fname);
	else
		fprintf(dt->ftree_out, "%s", p_fname);
	fprintf(dt->ftree_out, "</span> %ld</nobr><br>",
		fc->cnt);
}

void out_func_tree_html_end(struct cov_out_data *dt)
{
	fprintf(dt->ftree_out,
		"</body></html>\n");
	fclose(dt->ftree_out);
}

void out_func_html_start(struct cov_out_data *dt, long same, long diff)
{
	char title[MAX_LINE_LEN + 1], *js_path;
	int head_type;

	open_html(dt);
	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s function coverage", dt->name);
	if (dt->chk_type == CHK_DIFF_KERNEL) {
		head_type = HEAD_TYPE_JS;
		js_path = "../" JS_BOTH_OPEN;
	} else {
		head_type = HEAD_TYPE_NORMAL;
		js_path = NULL;
	}
	out_html_normal_header(dt->cur_out, title, head_type, js_path);
	if (dt->chk_type == CHK_DIFF_KERNEL)
		fprintf(dt->cur_out,
			"<body onKeyPress=\"syncScroll(event)\">\n" \
			"Press 'u', 'd' key for sync-scroll both sources.<br>\n");
	else
		fprintf(dt->cur_out, "<body>\n");
	if (dt->chk_type == CHK_SINGLE) {
		fprintf(dt->cur_out,
			"<table summary=\"%s\" border=\"1\">" \
			"<tr><th>function</th><th>count</th>%s</tr>\n",
			title, dt->limit_by_funcs ? "<th>n-function</th>" : "");
	} else {
		if (dt->chk_type == CHK_SAME_KERNEL)
			out_diff_same_diff(dt, same, diff);
		fprintf(dt->cur_out,
			"<table summary=\"%s\" border=\"1\">" \
			"<tr><th rowspan=\"2\">function</th><th colspan=\"2\">count</th></tr>\n" \
			"<tr><th>L1</th><th>L2</th></tr>\n", title);
	}
}

static char* get_src_html_path(struct cov_out_data *dt, struct src_info *info,
			       int is_ref)
{
	int rc;

	if (info->html_out_path[0] != '\0' && info->is_ref == is_ref)
		return info->html_out_path;
	if (is_ref)
		rc = snprintf(info->html_out_path, PATH_MAX,
			      "../src/%s.html", info->path + 1);
	else
		rc = snprintf(info->html_out_path, PATH_MAX,
			      "%s/src/%s.html", dt->outdir, info->path + 1);
	if (rc > PATH_MAX)
		printf("WARN: PATH_MAX is too short.\n");
	info->is_ref = is_ref;
	return info->html_out_path;
}

static struct src_info* chk_src_line_type(struct cov_out_data *dt, int kern,
					  const char *src, long ln, char type)
{
	struct r2i_pack *r2p = dt->r2p[kern];
	FILE *f = NULL;
	char path[PATH_MAX + 1];
	struct src_info *info = NULL;
	long i;
	char buf[MAX_LINE_LEN + 1];

	if (!ln)
		return NULL;

	/* get absolute path */
	if (src[0] == '/')
		sprintf(path, "%s%s", get_elf_path_prefix(), src);
	else {
		if (!r2p->srcdir)
			return NULL;
		sprintf(path, "%s%s/%s",
			get_elf_path_prefix(), r2p->srcdir, src);
	}

	/* search into src_info structures */
	for (i = r2p->src_info_num - 1; i >= 0; i--) {
		info = r2p->src_info[i];
		if (strcmp(info->path, path) == 0) {
			if (ln > info->ln_max)
				return NULL;
			goto CHK_LINE_TYPE;
		}
	}
	/* check source file existance and max line number */
	f = fopen(path, "r");
	if (!f)
		return NULL;
	for (i = 0; fgets(buf, MAX_LINE_LEN, f); i++);
	fclose(f);
	if (ln > i)
		return NULL;

	info = xcalloc(1, sizeof(struct src_info));
	strcpy(info->path, path);
	info->ln_max = i;
	info->exec_types = xcalloc(i, sizeof(char));
	if (r2p->src_info_num) {
		r2p->src_info = xrealloc(r2p->src_info,
					 (r2p->src_info_num + 1) *
					 	sizeof(struct src_info*));
	} else {
		r2p->src_info = xmalloc(sizeof(struct src_info*));
	}
	r2p->src_info[r2p->src_info_num++] = info;
CHK_LINE_TYPE:
	if (info->exec_types[ln - 1] == T_NONE ||
	    (info->exec_types[ln - 1] != type && type == T_EXECUTED))
		info->exec_types[ln - 1] = type;
	return info;
}

static char escape_need_chars[] = "\t<>&\"";
static char *escape_htmls[] = {
	"        ",
	"&lt;",
	"&gt;",
	"&amp;",
	"&quot;",
};

static void escape_for_html(char *buf)
{
	int len_max = strlen(buf), len, n;
	char *from, *to, *p, tmp[MAX_LINE_LEN + 1];

	tmp[MAX_LINE_LEN] = '\0';
	len = 0;
	for (from = buf, to = tmp; from - buf < len_max;) {
		p = strchr(escape_need_chars, *from);
		if (p) {
			from++;
			p = escape_htmls[p - escape_need_chars];
			n = strlen(p);
			if (len + n > MAX_LINE_LEN) {
				n = MAX_LINE_LEN - (len + n);
				snprintf(to, n, "%s", p);
			} else
				sprintf(to, "%s", p);
			to += n;
			len += n;
		} else {
			*to++ = *from++;
			len++;
		}
	}
	*to = '\0';
	strcpy(buf, tmp);
}

static int get_ln_cols(struct src_info *info)
{
	int cols = 0;
	long left;

	for (left = info->ln_max + 1; left; left = left / 10)
		cols++;
	return cols;
}

static inline char *__type2color(char type)
{
	switch (type) {
	case T_NOT_EXECUTED:
		return COLOR_NT;
	case T_EXECUTED:
		return COLOR_OK;
	case T_UNKNOWN:
		return COLOR_HT;
	default:
		return "";
	}
}

static char* type2color(int chk_type, char type, char type2)
{
	if (chk_type == CHK_SAME_KERNEL) {
		if (type == type2) {
			switch (type2) {
			case T_NOT_EXECUTED:
				return COLOR_BOTH_NT;
			case T_EXECUTED:
				return COLOR_BOTH_OK;
			case T_UNKNOWN:
				return COLOR_BOTH_HT;
			default:
				return "";
			}
		} else
			return __type2color(type2);
	} else
		return __type2color(type);
}

#define CHK_EXEC_TYPE(info, ln, type) \
	do { \
		if ((info)->exec_types[(ln)] != T_NONE) \
			type = (info)->exec_types[(ln)]; \
		else if (type != T_NONE) \
			type = T_UNKNOWN; \
	} while (0)

static void out_src_html(struct cov_out_data *dt, int kern)
{
	struct r2i_pack *r2p = dt->r2p[kern], *r2p2 = NULL;
	long i, j, ln;
	struct src_info *info = NULL, *info2 = NULL;
	int ln_cols;
	FILE *r, *w;
	char type, type2, *p, *path, *color, buf[MAX_LINE_LEN + 1];

	if (dt->chk_type == CHK_SAME_KERNEL)
		r2p2 = dt->r2p[1];
	buf[MAX_LINE_LEN] = '\0';
	for (i = 0; i < r2p->src_info_num; i++) {
		info = r2p->src_info[i];
		if (dt->chk_type == CHK_SAME_KERNEL) {
			for (j = 0; j < r2p2->src_info_num; j++) {
				info2 = r2p2->src_info[j];
				if (strcmp(info->path, info2->path) == 0)
					break;
			}
			if (j == r2p2->src_info_num) {
				fprintf(stderr, "Source file (%s) not found.\n",
					info->path);
				continue;
			}
		}
		snprintf(buf, MAX_LINE_LEN - 1, "%s", info->path);
		r = fopen(buf, "r");
		if (!r) {
			fprintf(stdout, "%s can't open.(%s)\n",
				buf, strerror(errno));
			break;
		}
		path = get_src_html_path(dt, info, 0);
		p = strdup(path);
		if (dir_chk_and_create(dirname(p), 0) < 0) {
			fclose(r);
			break;
		}
		free(p);
		w = fopen(path, "w");
		if (!w) {
			fprintf(stdout, "%s can't open.(%s)\n",
				path, strerror(errno));
			break;
		}
		ln_cols = get_ln_cols(info);
		out_html_normal_header(w, "src", HEAD_TYPE_NORMAL, NULL);
		fprintf(w, "<body style=\"font-size:8pt\"><pre>\n");
		type = type2 = T_NONE;
		for (ln = 0; fgets(buf, MAX_LINE_LEN, r); ln++) {
			/* for DEBUG
			printf("%*ld: %d\n",
			       ln_cols, ln + 1, info->exec_types[ln]);
			       */
			CHK_EXEC_TYPE(info, ln, type);
			/*
			if (info->exec_types[ln] != T_NONE)
				type = info->exec_types[ln];
			else if (type != T_NONE)
				type = T_UNKNOWN;
			else
				type = T_NONE;
				*/
			if (dt->chk_type == CHK_SAME_KERNEL) {
				CHK_EXEC_TYPE(info2, ln, type2);
				/*
				if (info2->exec_types[ln] != T_NONE)
					type2 = info2->exec_types[ln];
					*/
			}
			fprintf(w, "%*ld: ", ln_cols, ln + 1);
			buf[strlen(buf) - 1] = '\0';
			escape_for_html(buf);
			color = type2color(dt->chk_type, type, type2);
			if (color == "")
				fprintf(w, "%s\n", buf);
			else
				fprintf(w, "<a name=\"%ld\" style=\"background:%s\">%s</a>\n",
					ln + 1, color, buf);
			/* '}' of a line-head is regarded as the end of a
			 * function.
			 */
			if (buf[0] == '}')
				type = type2 = T_NONE;
		}
		fprintf(w, "</pre></body></html>\n");
		fclose(r);
		fclose(w);
	}
}

/*-----------------------------------------------------------------------------
 *  initialize
 *-----------------------------------------------------------------------------
 */
int dir_chk_and_create(char *path, int err_on_exists)
{
	struct stat st;
	char buf[MAX_LINE_LEN + 1];

	if (stat(path, &st) < 0) {
		if (errno == ENOENT) {
			buf[MAX_LINE_LEN] = '\0';
			snprintf(buf, MAX_LINE_LEN, "mkdir -p \"%s\"", path);
			if (system(buf) < 0) {
				fprintf(stderr, "%s can't create.\n", path);
				return -1;
			}
			return 0;
		}
		fprintf(stderr, "%s can't get stat.(%s)\n",
			path, strerror(errno));
		return -1;
	}
	if (err_on_exists) {
		fprintf(stderr, "%s already exists.\n", path);
		return -1;
	}
	if (!S_ISDIR(st.st_mode)) {
		fprintf(stderr, "%s is not directory.\n", path);
		return -1;
	}
	return 0;
}

int init_html_output(struct cov_out_data *dt)
{
	char path[PATH_MAX + 1];

	path[PATH_MAX] = '\0';
	snprintf(path, PATH_MAX, "%s/each", dt->outdir);
	if (dir_chk_and_create(path, 1) < 0)
		return -1;

	path[PATH_MAX] = '\0';
	snprintf(path, PATH_MAX, "%s/summary.html", dt->outdir);
	dt->summary_out = fopen(path, "w");
	if (!dt->summary_out) {
		fprintf(stderr, "%s can't open.(%s)\n", path,
			strerror(errno));
		return -1;
	}
	if (out_top_html(dt) < 0)
		return -1;
	if (out_title_html(dt) < 0)
		return -1;
	if (dt->chk_type == CHK_DIFF_KERNEL) {
		path[PATH_MAX] = '\0';
		snprintf(path, PATH_MAX, "%s/src", dt->outdir);
		if (dir_chk_and_create(path, 1) < 0)
			return -1;
		out_both_open_and_scrl_js(dt);
		out_blank_html1(dt);
		out_blank_html2(dt);
	} else if (dt->chk_type == CHK_SINGLE && dt->limit_by_funcs)
		out_tree_menu_js(dt);
	out_summary_html_start(dt);
	return 0;
}

int exit_html_output(struct cov_out_data *dt, int limit_by_funcs)
{
	out_src_html(dt, 0);
	if (dt->chk_type == CHK_DIFF_KERNEL)
		out_src_html(dt, 1);
	out_summary_html_end(dt, limit_by_funcs);
	if (dt->summary_out)
		fclose(dt->summary_out);
	return 0;
}

