/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <time.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"clock_getres"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	NORMAL,
	NULL_POINTER,
};


/*
 * Data Structure
 */
struct test_case {
	clockid_t clk_id;
	int ttype;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EPERM		can't check because clock_getres not requires permission
 *   EINVAL		v (not supported clk_id)
 */
static struct test_case tcase[] = {
	{ // case00
		.clk_id		= CLOCK_REALTIME,
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.clk_id		= CLOCK_MONOTONIC,
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.clk_id		= CLOCK_PROCESS_CPUTIME_ID,
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.clk_id		= CLOCK_THREAD_CPUTIME_ID,
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.clk_id		= -1,
		.ttype		= NORMAL,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case05
		.clk_id		= CLOCK_REALTIME,
		.ttype		= NULL_POINTER,
		.ret		= 0,
		.err		= 0,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct timespec res;

	/*
	 * Execute system call
	 */
	errno = 0;
	if (tc->ttype == NULL_POINTER)
		sys_ret = clock_getres(tc->clk_id, NULL);
	else
		sys_ret = clock_getres(tc->clk_id, &res);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

