/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <time.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"clock_settime"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	NORMAL,
	NULL_POINTER,
};


/*
 * Data Structure
 */
struct test_case {
	clockid_t clk_id;
	int ttype;
	char *user;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EPERM		v (not have permission)
 *   EINVAL		v (not supported clk_id)
 *   EFAULT		can't check because glibc generates SEGV
 */
static struct test_case tcase[] = {
	{ // case00
		.clk_id		= CLOCK_REALTIME,
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.clk_id		= CLOCK_MONOTONIC,
		.ttype		= NORMAL,
		.ret		= -1,
		.err		= EINVAL,	// documented in man page
	},
	{ // case02
		.clk_id		= CLOCK_PROCESS_CPUTIME_ID,
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.clk_id		= CLOCK_THREAD_CPUTIME_ID,
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.clk_id		= -1,
		.ttype		= NORMAL,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case05
		.clk_id		= CLOCK_REALTIME,
		.ttype		= NORMAL,
		.user		= "nobody",
		.ret		= -1,
		.err		= EPERM,
	},
#if 0	// glibc generates SEGV error (RHEL4U1 + 2.6.18)
	{ // caseXX
		.clk_id		= CLOCK_REALTIME,
		.ttype		= NULL_POINTER,
		.ret		= -1,
		.err		= EFAULT,
	},
#endif
};

/*
 * chk_difftime()
 *   Return : OK(0), NG(-1)
 */
#define MAX_MSEC_DIFF	1000
static int chk_difftime(struct timespec *t1, struct timespec *t2)
{
	struct timespec t;

	if (t1->tv_sec > t2->tv_sec) {
		t.tv_sec = t1->tv_sec - t2->tv_sec;
		t.tv_nsec = t1->tv_nsec - t2->tv_nsec;
	} else {
		t.tv_sec = t2->tv_sec - t1->tv_sec;
		t.tv_nsec = t2->tv_nsec - t1->tv_nsec;
	}
	if (t.tv_nsec < 0) {
		t.tv_sec -= 1;
		t.tv_nsec += 1000000000;
	}
	if ((t.tv_sec * 1000) + (t.tv_nsec / 1000000) > MAX_MSEC_DIFF)
		return -1;
	return 0;
}

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct timespec oldtp, newtp, tp;
	int rc;
	uid_t old_uid;
	int cmp_ok = 1;

	/*
	 * Change effective user id
	 */
	if (tc->user != NULL) {
		rc = setup_euid(tc->user, &old_uid);
		if (rc < 0)
			return 1;
	}

	/*
	 * Save time
	 */
	rc = clock_gettime(tc->clk_id, &oldtp);
	if (rc < 0 && tc->clk_id != -1) {
		EPRINTF("clock_gettime failed.\n");
		result = 1;
		goto EXIT;
	}

	/*
	 * Execute system call
	 */
	tp.tv_sec = 1000;
	tp.tv_nsec = 1000;
	errno = 0;
	if (tc->ttype == NULL_POINTER)
		sys_ret = clock_settime(tc->clk_id, NULL);
	else
		sys_ret = clock_settime(tc->clk_id, &tp);
	sys_errno = errno;

	/*
	 * Check got-time and restore time
	 */
	if (sys_ret >= 0) {
		rc = clock_gettime(tc->clk_id, &newtp);

		errno = 0;
		sys_ret = clock_settime(tc->clk_id, &oldtp);
		sys_errno = errno;

		if (rc >= 0)
			cmp_ok = chk_difftime(&tp, &newtp) == 0;
	}

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);
EXIT:
	/*
	 * Restore effective user id
	 */
	if (tc->user != NULL) {
		rc = cleanup_euid(old_uid);
		if (rc < 0)
			return 1;
	}
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

