/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include "numaif.h"

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"get_mempolicy"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	DEFAULT,	// get default policy
	ADDR,		// get policy of memory which include mapped address
	INVALID_POINTER,
	INVALID_FLAGS,
};

enum from_node {
	NONE,
	SELF,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	int policy;
	int from_node;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   (NONE)		man page hadn't been completed.
 *
 *   test status of errors NOT on man page
 *
 *   EFAULT		v (invalid address)
 *   EINVAL		v (invalid parameters)
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= DEFAULT,
		.policy		= MPOL_DEFAULT,
		.from_node	= NONE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= DEFAULT,
		.policy		= MPOL_BIND,
		.from_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= DEFAULT,
		.policy		= MPOL_INTERLEAVE,
		.from_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= DEFAULT,
		.policy		= MPOL_PREFERRED,
		.from_node	= NONE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.ttype		= DEFAULT,
		.policy		= MPOL_PREFERRED,
		.from_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case05
		.ttype		= ADDR,
		.policy		= MPOL_DEFAULT,
		.from_node	= NONE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case06
		.ttype		= ADDR,
		.policy		= MPOL_BIND,
		.from_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case07
		.ttype		= ADDR,
		.policy		= MPOL_INTERLEAVE,
		.from_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case08
		.ttype		= ADDR,
		.policy		= MPOL_PREFERRED,
		.from_node	= NONE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case09
		.ttype		= ADDR,
		.policy		= MPOL_PREFERRED,
		.from_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case10
		.ttype		= INVALID_POINTER,
		.policy		= MPOL_DEFAULT,
		.from_node	= NONE,
		.ret		= -1,
		.err		= EFAULT,
	},
	{ // case11
		.ttype		= INVALID_FLAGS,
		.policy		= MPOL_DEFAULT,
		.from_node	= NONE,
		.ret		= -1,
		.err		= EINVAL,
	},
};

#define MEM_LENGTH		(4 * 1024 * 1024)
/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int rc, policy, flags, cmp_ok = 1;
	nodemask_t nodemask, getnodemask;
	unsigned long maxnode = NUMA_NUM_NODES;
	char *p = NULL;
	unsigned long len = MEM_LENGTH;

	/* We assume that there is only one node(node0). */
	nodemask_zero(&nodemask);
	nodemask_set(&nodemask, 0);
	nodemask_zero(&getnodemask);

	switch (tc->ttype) {
	case DEFAULT:
		flags = 0;
		p = NULL;
		// set memory policy
		if (tc->from_node == NONE)
			rc = syscall(__NR_set_mempolicy, tc->policy, NULL, 0);
		else
			rc = syscall(__NR_set_mempolicy, tc->policy,
				     &nodemask, maxnode);
		if (rc < 0) {
			EPRINTF("set_mempolicy failed.\n");
			result = 1;
			goto EXIT;
		}
		break;
	default:
		flags = MPOL_F_ADDR;
		// mmap memory
		p = mmap(NULL, len, PROT_READ|PROT_WRITE,
			 MAP_PRIVATE|MAP_ANONYMOUS, 0, 0);
		if (p == (void*)-1) {
			EPRINTF("malloc failed.\n");
			result = 1;
			goto EXIT;
		}
		// set memory policy
		if (tc->from_node == NONE)
			rc = syscall(__NR_mbind, p, len, tc->policy,
				     NULL, 0, 0);
		else
			rc = syscall(__NR_mbind, p, len, tc->policy,
				     &nodemask, maxnode, 0);
		if (tc->ttype == INVALID_POINTER)
			p = NULL;
		if (tc->ttype == INVALID_FLAGS)
			flags = -1;
		break;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_get_mempolicy, &policy, &getnodemask, maxnode,
			  p, flags);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

	// When policy equals MPOL_DEFAULT, then get_mempolicy not return node
	if (tc->policy == MPOL_DEFAULT)
		nodemask_zero(&nodemask);
	cmp_ok = tc->policy == policy && nodemask_equal(&nodemask,&getnodemask);
	if (opt_debug) {
		nodemask_dump("nodemask Expect:", &nodemask);
		nodemask_dump("nodemask Result:", &getnodemask);
		EPRINTF("policy E:%d R:%d\n", tc->policy, policy);
	}

TEST_END:
	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(0, tc->ret, tc->err, sys_ret, sys_errno, cmp_ok);
EXIT:
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

