/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include "numaif.h"

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"migrate_pages"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	NORMAL,
	INVALID_MAXNODE,
	NO_SUCH_PROC,
};

enum node_type {
	NONE,
	SELF,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	char *user;
	int from_node;
	int to_node;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   (NONE)		man page doesn't exist
 *
 *   test status of errors NOT on man page
 *
 *   EINVAL		v (invalid argument)
 *   ESRCH		v (not exist process)
 *   EPERM		v (permission denied)
 */
static struct test_case tcase[] = {
	{ // case00
		.from_node	= NONE,
		.to_node	= NONE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.from_node	= NONE,
		.to_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.from_node	= SELF,
		.to_node	= NONE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.from_node	= SELF,
		.to_node	= SELF,
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.ttype		= INVALID_MAXNODE,
		.from_node	= NONE,
		.to_node	= SELF,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case05
		.ttype		= NO_SUCH_PROC,
		.from_node	= NONE,
		.to_node	= NONE,
		.ret		= -1,
		.err		= ESRCH,
	},
	{ // case05
		.user		= "nobody",
		.from_node	= NONE,
		.to_node	= NONE,
		.ret		= -1,
		.err		= EPERM,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int rc;
	nodemask_t nodemask, *from, *to;
	unsigned long maxnode = NUMA_NUM_NODES;
	pid_t pid;

	/* We assume that there is only one node(node0). */
	nodemask_zero(&nodemask);
	nodemask_set(&nodemask, 0);

	switch (tc->ttype) {
	case NO_SUCH_PROC:
		pid = get_unexist_pid();
		if (pid < 0) {
			result = 1;
			goto EXIT;
		}
		break;
	case INVALID_MAXNODE:
		maxnode = 0;
		/* fallthrough */
	default:
		pid = getpid();
		break;
	}

	if (tc->user != NULL) {
		// It's no use for changing the euid. We should change the uid.
		rc = setup_uid("nobody");
		if (rc < 0) {
			result = 1;
			goto EXIT;
		}
		pid = 1; // set the target process ('init'), so always fail.
	}

	from = tc->from_node == NONE ? NULL : &nodemask;
	to = tc->to_node == NONE ? NULL : &nodemask;

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_migrate_pages, pid, maxnode, from, to);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

TEST_END:
	/*
	 * Check results
	 */
	result |= sys_errno != tc->err;
	PRINT_RESULT(0, tc->ret, tc->err, sys_ret, sys_errno);
EXIT:
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

