/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <asm/errno.h>
#include "../../include/socketcall.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"accept"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	INVALID_POINTER,
	NOT_SOCK_STREAM,
	NOT_LISTEN,
	NOT_SUPPORT_PROTO,
//	NOT_CONN,
	WOULD_BLOCK,
	NO_FILE,
	NORMAL,
};


/*
 * Data Structure
 */
struct test_case {
	int domain;
	int ttype;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EAGAIN/EWOULDBLOCK	v (would block)
 *   EBADF		v (not a valid descriptor)
 *   ENOTSOCK		v (file descriptor)
 *   EOPNOTSUPP		v (socket type is not SOCK_STREAM)
 *   EINTR		--- (signal occur)
 *   ECONNABORTED	--- (connection has been aborted)
 *   EINVAL		v (not listening)
 *   EMFILE		v (process file table overflow)
 *   ENFILE		can't check because it's difficult to create file limit
 *   EFAULT		v (points to not process address space)
 *   ENOBUFS/ENOMEM	can't check because it's difficult to create no-memory
 *   EPROTO		can't check because protocol error generates EOPNOTSUPP
 *   EPERM		can't check because it's difficult to create firewall
 *                      rules
 */
static struct test_case tcase[] = {
	// -------- PF_UNIX --------
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_NONE,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_NOT_EXIST,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_FILE,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= INVALID_POINTER,
		.ret	= -1,
		.err	= EFAULT,	// RHEL4U1 + 2.6.18 returns EINVAL
	},
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= NOT_SOCK_STREAM,
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= NOT_LISTEN,
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= WOULD_BLOCK,
		.ret	= -1,
		.err	= EAGAIN,
	},
#if 0					// RHEL4U0 is locked
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= NO_FILE,
		.ret	= -1,
		.err	= EMFILE,
	},
#endif
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.ret	= 0,
		.err	= 0,
	},
	// -------- PF_INET --------
	{ // case00
		.domain	= PF_INET,
		.ttype	= FD_NONE,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case00
		.domain	= PF_INET,
		.ttype	= FD_NOT_EXIST,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case00
		.domain	= PF_INET,
		.ttype	= FD_FILE,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case00
		.domain	= PF_INET,
		.ttype	= INVALID_POINTER,
		.ret	= -1,
		.err	= EFAULT,	// RHEL4U1 + 2.6.18 returns EINVAL
	},
	{ // case00
		.domain	= PF_INET,
		.ttype	= NOT_SOCK_STREAM,
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
	{ // case00
		.domain	= PF_INET,
		.ttype	= NOT_LISTEN,
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case00
		.domain	= PF_INET,
		.ttype	= WOULD_BLOCK,
		.ret	= -1,
		.err	= EAGAIN,
	},
#if 0					// RHEL4U0 is locked
	{ // case00
		.domain	= PF_INET,
		.ttype	= NO_FILE,
		.ret	= -1,
		.err	= EMFILE,
	},
#endif
	{ // case00
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.ret	= 0,
		.err	= 0,
	},
};

#define ULIMIT_FNUM	0

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int fd = -1, rc;
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct sockaddr_in in_addr, cin_addr;
	struct sockaddr_un un_addr, cun_addr;
	struct sockaddr *addr, *caddr;
	int len, clen, type;
	pid_t cl_pid = 0;
	rlim_t oldlim;
	int st, cl_type;

	// initialize 'xn_addr' and 'len' and create server process if necessary
	if (tc->domain == PF_UNIX) {
		un_addr.sun_family = AF_UNIX;
		strcpy(un_addr.sun_path, "");
		len = clen = sizeof(un_addr);
		addr = (struct sockaddr*)&un_addr;
		caddr = (struct sockaddr*)&cun_addr;
	} else {
		in_addr.sin_family = AF_INET;
		in_addr.sin_port = 11000;	// this port must be unused
		in_addr.sin_addr.s_addr = INADDR_ANY;
		len = clen = sizeof(in_addr);
		addr = (struct sockaddr*)&in_addr;
		caddr = (struct sockaddr*)&cin_addr;
	}

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			return 1;
		}
		break;
	case INVALID_POINTER:
		caddr = NULL;
		/* fallthrough */
	default:
		type = tc->ttype == NOT_SOCK_STREAM ? SOCK_DGRAM : SOCK_STREAM;
		fd = socket(tc->domain, type, 0);
		if (fd < 0) {
			EPRINTF("can't create socket.\n");
			result = 1;
			goto EXIT;
		}
		rc = bind(fd, addr, len);
		if (rc < 0) {
			EPRINTF("bind failed.\n");
			result = 1;
			goto EXIT;
		}
		switch (tc->ttype) {
		case INVALID_POINTER:
		case NOT_SOCK_STREAM:
		case NOT_LISTEN:
			break;
		default:
			rc = listen(fd, 1);
			if (rc < 0) {
				EPRINTF("listen failed.\n");
				result = 1;
				goto EXIT;
			}
			if (tc->ttype == WOULD_BLOCK) {
				int flval;
				flval = fcntl(fd, F_GETFL, 0);
				fcntl(fd, F_SETFL, flval | O_NONBLOCK);
			} else if (tc->ttype == NO_FILE) {
				rc = setup_ulimit_fnum(ULIMIT_FNUM, &oldlim);
				if (rc < 0)
					return 1;
			} else {
				cl_type = CL_NORMAL;
				cl_pid = create_client_proc(SOCK_STREAM, 0,
							    addr, len, cl_type);
				if (cl_pid < 0) {
					result = 1;
					goto EXIT;
				}
			}
			break;
		}
		break;
	}

	// do accept
	errno = 0;
	sys_ret = accept(fd, caddr, &clen);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);
EXIT:
	if (tc->ttype == NO_FILE)
		cleanup_ulimit_fnum(oldlim);

	if (fd >= 0)
		close(fd);
	if (cl_pid > 0) {
		kill(cl_pid, SIGTERM);
		wait(&st);
	}
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

