/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <asm/errno.h>
#include "../../include/socketcall.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"recvmsg"

#define LEN_PER_VEC	4
#define MBUF_LEN	((UIO_MAXIOV + 1) * LEN_PER_VEC)


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static int port;
static struct sockaddr_in in_from;
static struct sockaddr_un un_from;

// default is 'connected socket'
enum test_type {
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	NULL_MSG,
	NOT_CONNECTED,
	SERVER_DISCONNECT,
	SERVER_SEND_SIGINT,
	WOULD_BLOCK,
	NORMAL,
};


/*
 * Data Structure
 */
struct test_case {
	int domain;
	int ttype;
	char *path;	// for PF_UNIX
	u_int32_t addr;	// for PF_INET
	size_t iovlen;
	struct sockaddr* from;
	int fromlen;
	int flag;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   ECONNREFUSED	can't check because connection error generates ENOTCONN
 *   ENOTCONN		v (not connected)
 *   ENOTSOCK		v (file descriptor)
 *   EAGAIN		v (would block)
 *   EINTR		can't check because EINTR never happen even though
 *   			SIGINT was received
 *   EFAULT		v (points to not process address space)
 *   EMSGSIZE		v (invalid message size)
 *   EINVAL		v (invalid argument)
 *   ENOMEM		can't check because it's difficult to create no-memory
 */
static struct test_case tcase[] = {
	// -------- PF_UNIX --------
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_NONE,
		.path	= "",
		.iovlen	= 1,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case01
		.domain	= PF_UNIX,
		.ttype	= FD_NOT_EXIST,
		.path	= "",
		.iovlen	= 1,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case02
		.domain	= PF_UNIX,
		.ttype	= FD_FILE,
		.path	= "",
		.iovlen	= 1,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case03
		.domain	= PF_UNIX,
		.ttype	= NULL_MSG,
		.path	= "",
		.iovlen	= 1,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case04
		.domain	= PF_UNIX,
		.ttype	= NOT_CONNECTED,
		.path	= "",
		.iovlen	= 1,
		.flag	= MSG_NOSIGNAL,
		.ret	= -1,
		.err	= ENOTCONN,	// RHEL4U1 + 2.6.18 returns EINVAL
	},
	{ // case05
		.domain	= PF_UNIX,
		.ttype	= SERVER_DISCONNECT,
		.path	= "",
		.iovlen	= 1,
		.flag	= MSG_NOSIGNAL,
		.ret	= 0,
		.err	= 0,
	},
#if 0
	{ // caseXX
		.domain	= PF_UNIX,
		.ttype	= SERVER_SEND_SIGINT,
		.path	= "",
		.iovlen	= 1,
		.ret	= -1,
		.err	= EINTR,	// RHEL4U1 + 2.6.18 returns SUCCESS
	},
#endif
	{ // case06
		.domain	= PF_UNIX,
		.ttype	= WOULD_BLOCK,
		.path	= "",
		.iovlen	= 1,
		.ret	= -1,
		.err	= EAGAIN,
	},
	{ // case07
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.iovlen	= 1,
		.from	= (struct sockaddr*)&un_from,
		.fromlen= sizeof(un_from),
		.ret	= 0,
		.err	= 0,
	},
	{ // case08
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.iovlen	= 1,
		.ret	= 0,
		.err	= 0,
	},
	{ // case09
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.iovlen	= UIO_MAXIOV,
		.ret	= 0,
		.err	= 0,
	},
	{ // case10
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.iovlen	= UIO_MAXIOV + 1,
		.ret	= -1,
		.err	= EMSGSIZE,
	},
	// -------- PF_INET --------
	{ // case11
		.domain	= PF_INET,
		.ttype	= FD_NONE,
		.iovlen	= 1,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case12
		.domain	= PF_INET,
		.ttype	= FD_NOT_EXIST,
		.iovlen	= 1,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case13
		.domain	= PF_INET,
		.ttype	= FD_FILE,
		.iovlen	= 1,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case14
		.domain	= PF_INET,
		.ttype	= NULL_MSG,
		.addr	= INADDR_ANY,
		.iovlen	= 1,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case15
		.domain	= PF_INET,
		.ttype	= NOT_CONNECTED,
		.addr	= INADDR_ANY,
		.iovlen	= 1,
		.flag	= MSG_NOSIGNAL,
		.ret	= -1,
		.err	= ENOTCONN,
	},
	{ // case16
		.domain	= PF_INET,
		.ttype	= SERVER_DISCONNECT,
		.addr	= INADDR_ANY,
		.iovlen	= 1,
		.flag	= MSG_NOSIGNAL,
		.ret	= 0,
		.err	= 0,
	},
#if 0
	{ // caseXX
		.domain	= PF_INET,
		.ttype	= SERVER_SEND_SIGINT,
		.addr	= INADDR_ANY,
		.iovlen	= 1,
		.ret	= -1,
		.err	= EINTR,	// RHEL4U1 + 2.6.18 returns SUCCESS
	},
#endif
	{ // case17
		.domain	= PF_INET,
		.ttype	= WOULD_BLOCK,
		.addr	= INADDR_ANY,
		.iovlen	= 1,
		.ret	= -1,
		.err	= EAGAIN,
	},
	{ // case18
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.iovlen	= 1,
		.from	= (struct sockaddr*)&in_from,
		.fromlen= sizeof(in_from),
		.ret	= 0,
		.err	= 0,
	},
	{ // case19
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.iovlen	= 1,
		.ret	= 0,
		.err	= 0,
	},
	{ // case20
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.iovlen	= UIO_MAXIOV,
		.ret	= 0,
		.err	= 0,
	},
	{ // case21
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.iovlen	= UIO_MAXIOV + 1,
		.ret	= -1,
		.err	= EMSGSIZE,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int fd = -1, rc, type;
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct sockaddr_in in_addr;
	struct sockaddr_un un_addr;
	struct sockaddr *addr;
	int i, j, len, st, send_successed, cmp_ok;
	pid_t sv_pid = 0;
	struct msghdr msg;
	struct iovec miovec[UIO_MAXIOV + 1];
	char *p, mbuf[MBUF_LEN], rmbuf[MBUF_LEN];
	int svtype;

	type = SOCK_STREAM;
	if (tc->domain == PF_UNIX) {
		un_addr.sun_family = AF_UNIX;
		strcpy(un_addr.sun_path, tc->path);
		len = sizeof(un_addr);
		addr = (struct sockaddr*)&un_addr;
	} else {
		if (port == 0)
			port = (getpid() % 1024) * MAX_SOCK_TESTCASES + 11000;
		else
			port++;
		in_addr.sin_family = AF_INET;
		in_addr.sin_port = htons(port);
		in_addr.sin_addr.s_addr = tc->addr;
		len = sizeof(in_addr);
		addr = (struct sockaddr*)&in_addr;
	}
	switch (tc->ttype) {
	case SERVER_DISCONNECT:
		svtype = SV_DISCONN_BY_RECV;
		break;
	case SERVER_SEND_SIGINT:
		svtype = SV_SEND_SIGINT_BY_RECV;
		break;
	default:
		svtype = SV_NORMAL;
		break;
	}
	sv_pid = create_server_proc(type, 0, addr, len, svtype);
	if (sv_pid < 0) {
		result = 1;
		goto EXIT;
	}

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			return 1;
		}
		break;
	default:
		// do socket
		fd = socket(tc->domain, type, 0);
		if (fd < 0) {
			EPRINTF("can't create socket.\n");
			result = 1;
			goto EXIT;
		}

		// do connect
		if (tc->ttype != NOT_CONNECTED) {
			rc = connect(fd, addr, len);
			if (rc < 0) {
				EPRINTF("connect failed.%d %d %s\n", rc, errno,
					strerror(errno));
				result = 1;
				goto EXIT;
			}
		}
		if (tc->ttype == WOULD_BLOCK) {
			int flval;
			flval = fcntl(fd, F_GETFL, 0);
			fcntl(fd, F_SETFL, flval | O_NONBLOCK);
		}
		break;
	}

	// do sendmsg
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = miovec;
	msg.msg_iovlen = tc->iovlen;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_flags = 0;
	p = mbuf;
	for (i = 0; i < UIO_MAXIOV + 1; i++) {
		miovec[i].iov_base = p;
		miovec[i].iov_len = LEN_PER_VEC;
		for (j = 0; j < LEN_PER_VEC; j++) {
			*p = p - mbuf;
			p++;
		}
	}

	rc = sendmsg(fd, &msg, tc->flag);
	send_successed = rc > 0;

	switch (tc->ttype) {
	case SERVER_DISCONNECT:
	////case SERVER_SEND_SIGINT:
		usleep(100000);	// timing for server action
		break;
	}

	// check whether sendmsg data is really sended one
	cmp_ok = 1;
	msg.msg_name = tc->from;
	msg.msg_namelen = tc->fromlen;
	msg.msg_iov = miovec;
	msg.msg_iovlen = tc->iovlen;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_flags = 0;
	p = rmbuf;
	for (i = 0; i < UIO_MAXIOV + 1; i++) {
		miovec[i].iov_base = p;
		miovec[i].iov_len = LEN_PER_VEC;
		for (j = 0; j < LEN_PER_VEC; j++) {
			*p = 0;
			p++;
		}
	}
	if (send_successed && tc->ttype != WOULD_BLOCK)
		usleep(100000);	// timing for server action
	errno = 0;
	if (tc->ttype == NULL_MSG)
		sys_ret = recvmsg(fd, NULL, 0);
	else
		sys_ret = recvmsg(fd, &msg, 0);
	sys_errno = errno;

	if (sys_ret > 0 && send_successed) {
		if (sys_ret != tc->iovlen * LEN_PER_VEC)
			cmp_ok = 0;
		for (i = 0; i < tc->iovlen * LEN_PER_VEC; i++) {
			if (opt_debug)
				EPRINTF("(E) %d <=> (R) %d\n",
					mbuf[i], rmbuf[i]);
			if (mbuf[i] != rmbuf[i]) {
				cmp_ok = 0;
				//break;
			}
		}
	}

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(tc->ttype != SERVER_DISCONNECT && sys_ret >= 0,
			 tc->ret, tc->err, sys_ret, sys_errno, cmp_ok);
EXIT:
	if (fd >= 0)
		close(fd);
	if (sv_pid > 0) {
		kill(sv_pid, SIGTERM);
		wait(&st);
	}
	return result;
}


/*
 * sighandler()
 */
void sighandler(int sig)
{
	if (sig == SIGINT || sig == SIGPIPE)
		return;
	// NOTREACHED
	return;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	signal(SIGINT, sighandler);
	signal(SIGPIPE, sighandler);
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

