/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <dirent.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <libgen.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"utimes"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;


enum test_type {
	NORMAL,
	FILE_NOT_EXIST,
	NO_FNAME,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	long a_sec;
	long m_sec;
	char *user;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EACCES		v (permission denied)
 *   ENOENT		v (file does not exist)
 *
 *   test status of errors on man page
 *
 *   EFAULT		v (points to not process address space) 
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.a_sec		= 0,
		.m_sec		= 1000,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.a_sec		= 1000,
		.m_sec		= 0,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= NORMAL,
		.a_sec		= 1000,
		.m_sec		= 2000,
		.user		= "nobody",
		.ret		= -1,
		.err		= EACCES, // RHEL4U1 + 2.6.18 returns EPERM
	},
	{ // case03
		.ttype		= FILE_NOT_EXIST,
		.a_sec		= 1000,
		.m_sec		= 2000,
		.ret		= -1,
		.err		= ENOENT,
	},
	{ // case04
		.ttype		= NO_FNAME,
		.a_sec		= 1000,
		.m_sec		= 2000,
		.ret		= -1,
		.err		= EFAULT,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct timeval tv[2];
	char fpath[PATH_MAX], c = '\0';
	int rc, len, cmp_ok = 1;
	struct stat st;
	uid_t old_uid;

	rc = setup_file(progdir, "test.file", fpath);
	if (rc < 0)
		return 1;

	/*
	 * Change effective user id
	 */
	if (tc->user != NULL) {
		rc = setup_euid(tc->user, &old_uid);
		if (rc < 0)
			goto EXIT2;
	}

	/*
	 * Execute system call
	 */
	tv[0].tv_sec = tc->a_sec;
	tv[1].tv_sec = tc->m_sec;
	len = strlen(fpath);
	if (tc->ttype == FILE_NOT_EXIST) {
		c = fpath[len - 1];
		fpath[len - 1] = '\0';
	}
	errno = 0;
	if (tc->ttype == NO_FNAME)
		sys_ret = utimes(NULL, tv);
	else
		sys_ret = utimes(fpath, tv);
	sys_errno = errno;
	if (tc->ttype == FILE_NOT_EXIST)
		fpath[len - 1] = c;
	if (sys_ret < 0)
		goto TEST_END;

	/*
	 * Check test file's time stamp
	 */
	rc = stat(fpath, &st);
	if (rc < 0) {
		EPRINTF("stat failed.\n");
		result = 1;
		goto EXIT1;
	}
	PRINTF("E:%ld,%ld <=> R:%ld,%ld\n",
	       tv[0].tv_sec, tv[1].tv_sec, st.st_atime, st.st_mtime);
	cmp_ok = st.st_atime == tv[0].tv_sec && st.st_mtime == tv[1].tv_sec;

	/*
	 * Check results
	 */
TEST_END:
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);

	/*
	 * Restore effective user id
	 */
EXIT1:
	if (tc->user != NULL) {
		rc = cleanup_euid(old_uid);
		if (rc < 0)
			return 1;
	}
EXIT2:
	cleanup_file(fpath);

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

