# BTS/FileLocker.pm

package BTS::FileLocker;
use strict;
my $retry_count = 3;
my $wait_time = 0.3;
my $timeout = 60;

use File::Spec;
use Time::HiRes qw(sleep);
use FindBin;

sub new {
	my $pkg = shift;
	my $hash = {
		locker_temp_dir => shift
	};

	bless $hash, $pkg;
}

sub lock {
	my $self = shift;
	my $filename = shift;

	my $locker_temp_dir = $self->{locker_temp_dir};
	if(!-e $locker_temp_dir){
		return 1;
	}

	if($locker_temp_dir eq ''){return 0;}
	#$real_filename =  File::Spec->rel2abs($filename);

	if(!-e $locker_temp_dir){
		mkdir $locker_temp_dir, 0666;
	}

	my $real_filename = $locker_temp_dir . '/' . &filename_convert($filename);
	my $proc_filename = "$locker_temp_dir/$$\.tmp";

	my $opened = 0;
	my $canopen = 0;

	# normal wait
	foreach (my $i=0;$i<$retry_count;$i++){
		unless (-f $real_filename) { $canopen = 1; last; }
		sleep($wait_time);
	}

	# force unlock
	if(!$canopen){
		my @filestat = stat $real_filename;
		my $mtime = &format_time(localtime($filestat[9]));
		my $timelimit = &format_time(localtime(time + ($timeout * -1)));
		if($mtime < $timelimit){
			unlink $real_filename;
			$canopen = 1;
		}
	}


	if($canopen){
		if(open(IO, ">$proc_filename")){
			close(IO);
			
			foreach (my $i=0;$i<$retry_count;$i++){
				unless (-f $real_filename) {
					rename($proc_filename, $real_filename);
					$opened = 1;
					last;
				}
			}
		}
	}

	if($opened eq 1){
		return 1;
	}else{
		return undef;
	}
}

sub unlock {
	my $self = shift;
	my $filename = shift;

	my $locker_temp_dir = $self->{locker_temp_dir};
	my $real_filename = $locker_temp_dir . '/' . &filename_convert($filename);

	for(my $i=0;$i<$retry_count;$i++){
		unless(-f $real_filename){
			last;
		}
		if(unlink($real_filename) eq 1){
			last;
		}
		sleep($wait_time);
	}
}

sub filename_convert {
	my $filename = @_[0];
	$filename =~ s/\//_/g;
	$filename =~ s/\\/_/g;
	$filename =~ s/\./_/g;
	$filename =~ s/\:/_/g;
	$filename .= ".lock";
	return $filename;
}

sub format_time {
	my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = @_;
	my $fmt0 = "%04d%02d%02d%02d%02d%02d";
	my $time = sprintf($fmt0, $year+1900,$mon+1,$mday,$hour,$min,$sec);
	return $time;
}

1;
