/*
 * DebugMutex.h
 *
 * Copyright (C) 2001 by Ito Yoshiichi.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef debug_DebugMutex_h
#define debug_DebugMutex_h

#include "dgconfig.h"

#if defined(dg_USE_WIN32_THREADS)
#  include <windows.h>
#elif defined(dg_USE_POSIX_THREADS)
#  include <pthread.h>
#elif defined(dg_USE_VXW_THREADS)
#  include <semLib.h>
#endif

dg_NAMESPACE_BEGIN(debug)

/**
 * ȈՔrZ}tHB
 */
class DebugMutex
{
#if defined(dg_USE_WIN32_THREADS)
  public:
	DebugMutex()  { InitializeCriticalSection(&m_mutex); }
	~DebugMutex() { DeleteCriticalSection(&m_mutex); }
	void lock()   { EnterCriticalSection(&m_mutex); }
	void unlock() { LeaveCriticalSection(&m_mutex); }

  private:
	CRITICAL_SECTION m_mutex;

#elif defined(dg_USE_POSIX_THREADS)
  public:
	DebugMutex()  { pthread_mutex_init(&m_mutex, 0); }
	~DebugMutex() { pthread_mutex_destroy(&m_mutex); }
	void lock()   { pthread_mutex_lock(&m_mutex);   }
	void unlock() { pthread_mutex_unlock(&m_mutex); }

  private:
	pthread_mutex_t m_mutex;

#elif defined(dg_USE_VXW_THREADS)
  public:
	DebugMutex() : m_mutex(semMCreate(SEM_Q_PRIORITY | SEM_INVERSION_SAFE)) {}
	~DebugMutex() { semDelete(m_mutex); }
	void lock()   { semTake(m_mutex, WAIT_FOREVER); }
	void unlock() { semGive(m_mutex); }

  private:
	SEM_ID m_mutex;

#else /* no threads */
  public:
	DebugMutex()  {}
	~DebugMutex() {}
	void lock()   {}
	void unlock() {}

#endif
  private:
	DebugMutex(const DebugMutex& rhs);
	DebugMutex& operator=(const DebugMutex& rhs);

  public:
	/** DebugMutex bNB */
	class AutoLock {
	  public:
		/// bN𐶐B
		explicit AutoLock(DebugMutex *mutex)
			: m_mutex(mutex) { m_mutex->lock(); }
		/// bNB
		~AutoLock() { m_mutex->unlock(); }

	  private:
		AutoLock(const AutoLock& rhs);
		AutoLock& operator=(const AutoLock& rhs);
		DebugMutex* m_mutex;
	};
};

dg_NAMESPACE_END

#endif /* !debug_DebugMutex_h */
