(*
 * Copyright (C) 2002 Jun Kikuchi <kikuchi@bonnou.com>
 *
 * This file is part of ÍobN.
 *
 * ÍobN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ÍobN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *)
unit MainFormUnit;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, Buttons, ComCtrls, Menus, gcryptUnit, KeyUnit,
  ActnList;

type
  TMainForm = class(TForm)
    ExitButton: TButton;
    Panel1: TPanel;
    Image2: TImage;
    Bevel1: TBevel;
    StatusBar: TStatusBar;
    ExportBagButton: TSpeedButton;
    OpenDialog: TOpenDialog;
    DecryptButton: TSpeedButton;
    Bevel2: TBevel;
    Label1: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    UserIDLabel: TLabel;
    DateLabel: TLabel;
    KeyIDLabel: TLabel;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    OpenKeyMenu: TMenuItem;
    N1: TMenuItem;
    ExitMenu: TMenuItem;
    SaveDialog: TSaveDialog;
    Help1: TMenuItem;
    About1: TMenuItem;
    ActionList: TActionList;
    ExitAction: TAction;
    LoadDecodeKeyAction: TAction;
    ExportEncryptBagAction: TAction;
    AboutAction: TAction;
    EncryptFileButton: TSpeedButton;
    ChangePasswdButton: TSpeedButton;
    ChangePasswdAction: TAction;
    EncryptFileAction: TAction;
    DecryptAction: TAction;
    EncryptTextButton: TSpeedButton;
    Bevel3: TBevel;
    EncryptTextAction: TAction;
    SubkeyIDLabel: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure ExitActionExecute(Sender: TObject);
    procedure AboutActionExecute(Sender: TObject);
    procedure LoadDecodeKeyActionExecute(Sender: TObject);
    procedure ExportEncryptBagActionExecute(Sender: TObject);
    procedure ChangePasswdActionExecute(Sender: TObject);
    procedure EncryptFileActionExecute(Sender: TObject);
    procedure EncryptTextActionExecute(Sender: TObject);
    procedure DecryptActionExecute(Sender: TObject);
  private
    { Private 錾 }
    FLoadKey: Boolean;
    FSecKeyFileName: string;
    FSecKey: TSecKey;
  protected
    procedure LoadFile(FileName: string);
    procedure SetLoadKey(Value: Boolean);
    procedure ChangePasswd(CurrentPW, NewPW: string);
    procedure OnEncryptTerminate(Sender: TObject);
    procedure OnDecryptTerminate(Sender: TObject);
  public
    { Public 錾 }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property LoadKey: Boolean read FLoadKey write SetLoadKey;
  end;

var
  MainForm: TMainForm;

implementation

uses
  AboutFormUnit, ChangePasswdFormUnit, EncryptFormUnit, EncryptThreadUnit,
  DecryptFormUnit, DecryptThreadUnit, InputPasswdFormUnit, InputTextFormUnit;

{$R *.dfm}

procedure TMainForm.LoadFile(FileName: string);
var
  I: TFileStream;
begin
  try
    I := TFileStream.Create(FileName, fmOpenRead or fmShareExclusive);
    try
      FSecKey.LoadFromStream(I);
      LoadKey := FSecKey.CheckKey;
      if LoadKey then;
        FSecKeyFileName := FileName;
    finally
      I.Free;
    end;
  except
    on E: Exception do
      MessageDlg('ÍL[̓ǂݍ݂ŃG[܂' + #13 + E.Message, mtError, [mbAbort], 0);
    else
      MessageDlg('ÍL[̓ǂݍ݂ŃG[܂', mtError, [mbAbort], 0);
  end;
end;

procedure TMainForm.SetLoadKey(Value: Boolean);
begin
  FLoadKey := Value;

  DecryptButton.Enabled      := FLoadKey;
  EncryptFileButton.Enabled  := FLoadKey;
  EncryptTextButton.Enabled  := FLoadKey;
  ExportBagButton.Enabled    := FLoadKey;
  ChangePasswdButton.Enabled := FLoadKey;

  if FLoadKey then
  begin
    StatusBar.SimpleText  := '';
    UserIDLabel.Caption   := FSecKey.UserID.UID;
    DateLabel.Caption     := DateTimeToStr(FSecKey.SecKey.Date);
    KeyIDLabel.Caption    := FSecKey.SecKey.StrKeyID;
    SubkeyIDLabel.Caption := FSecKey.SecSubKey.StrKeyID;
  end;
end;

procedure TMainForm.ChangePasswd(CurrentPW, NewPW: string);
var
  O: TFileStream;
begin
  try
    if not FSecKey.Unlock(CurrentPW) then
    begin
      MessageDlg('pX[hႢ܂', mtError, [mbOK], 0);
      Exit;
    end;

    FSecKey.Lock(NewPW);

    O := TFileStream.Create(FSecKeyFileName, fmCreate or fmShareExclusive);
    try
      FSecKey.SaveToStream(O);
      MessageDlg('pX[h͕ύX܂', mtInformation, [mbOK], 0);
    finally
      O.Free;
    end;
  except
    on E: Exception do
      MessageDlg('pX[h̕ύXŃG[܂' + #13 + E.Message, mtInformation, [mbOK], 0)
    else
      MessageDlg('pX[h̕ύXŃG[܂', mtInformation, [mbOK], 0);
  end;
end;

procedure TMainForm.OnEncryptTerminate(Sender: TObject);
var
  E: Exception;
begin
  if not (Sender is TThread) then
    Exit;

  if TThread(Sender).FatalException is Exception then
  begin
    E := Exception(TThread(Sender).FatalException);
    MessageDlg('ÍŃG[܂' + #13 + E.Message, mtError, [mbAbort], 0);
    Close;
  end;
end;

procedure TMainForm.OnDecryptTerminate(Sender: TObject);
var
 E: Exception;
begin
  if not (Sender is TThread) then
    Exit;

  if TThread(Sender).FatalException is Exception then
  begin
    E := Exception(TThread(Sender).FatalException);
    MessageDlg('ÍǏŃG[܂' + #13 + E.Message, mtError, [mbAbort], 0);
  end;
end;

constructor TMainForm.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  gcry_control(GCRYCTL_INIT_SECMEM, 16384, 0);

  FLoadKey := False;
  FSecKeyFileName := '';
  FSecKey := TSecKey.Create;
end;

destructor TMainForm.Destroy;
begin
  FSecKey.Free;
  inherited;
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
  UserIDLabel.Caption        := '';
  DateLabel.Caption          := '';
  KeyIDLabel.Caption         := '';
  SubkeyIDLabel.Caption      := '';
  StatusBar.SimpleText       := 'ÍL[ǂݍ܂Ă܂';
  DecryptButton.Enabled      := False;
  EncryptFileButton.Enabled  := False;
  EncryptTextButton.Enabled  := False;
  ExportBagButton.Enabled    := False;
  ChangePasswdButton.Enabled := False;

  if ParamStr(1) <> '' then
    LoadFile(ParamStr(1));
end;

procedure TMainForm.ExitActionExecute(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.AboutActionExecute(Sender: TObject);
var
  DLG: TAboutForm;
begin
  DLG := TAboutForm.Create(Self);
  try
    DLG.ShowModal;
  finally
    DLG.Free;
  end;
end;

procedure TMainForm.LoadDecodeKeyActionExecute(Sender: TObject);
begin
  OpenDialog.Title    := 'ÍL[J';
  OpenDialog.Filter   := 'ÍL[ (*.key)|*.key';
  OpenDialog.FileName := '';
  OpenDialog.Options  := [ofHideReadOnly, ofFileMustExist, ofDontAddToRecent];

  if OpenDialog.Execute then
    LoadFile(OpenDialog.FileName);
end;

procedure TMainForm.ExportEncryptBagActionExecute(Sender: TObject);
var
  O: TFileStream;
begin
  SaveDialog.Title    := 'ÍobNt@Co';
  SaveDialog.Filter   := 'ÍobN (*.bag)|*.bag';
  SaveDialog.FileName := ChangeFileExt(FSecKeyFileName, '.bag');
  SaveDialog.Options  := [ofHideReadOnly, ofOverwritePrompt, ofDontAddToRecent];

  if not SaveDialog.Execute then
    Exit;

  try
    if ExtractFileExt(SaveDialog.Filename) = '' then
      SaveDialog.Filename := SaveDialog.Filename + '.bag';
      
    O := TFileStream.Create(SaveDialog.FileName, fmCreate or fmShareExclusive);
    try
      FSecKey.SavePubKeyToStream(O);
      MessageDlg('ÍobNZ[u܂', mtInformation, [mbOK], 0);
    finally
      O.Free;
    end;
  except
    on E: Exception do
      MessageDlg('ÍobN̍ĐŃG[܂' + #13 + E.Message, mtError, [mbAbort], 0);
    else
      MessageDlg('ÍobN̍ĐŃG[܂', mtError, [mbAbort], 0);
  end;
end;

procedure TMainForm.ChangePasswdActionExecute(Sender: TObject);
var
  DLG: TChangePasswdForm;
begin
  DLG := TChangePasswdForm.Create(Self);
  try
    if DLG.ShowModal = mrOK then
      ChangePasswd(DLG.PasswdEdit.Text, DLG.NewPasswdEdit.Text);
  finally
    DLG.Free;
  end; 
end;

procedure TMainForm.EncryptFileActionExecute(Sender: TObject);
var
  DLG: TEncryptForm;
  MS: TMemoryStream;
  PubKey: TPubKey;
  ET: TEncryptThread;
begin
  try
    OpenDialog.Title    := 'ÍobN֓t@CJ';
    OpenDialog.FileName := '';
    OpenDialog.Filter   := 't@C (*.*)|*.*';
    OpenDialog.Options  := [ofHideReadOnly, ofFileMustExist, ofDontAddToRecent];

    if not OpenDialog.Execute then
      Exit;

    SaveDialog.Title    := 'ÍobN֓ꂽt@C̕ۑ';
    SaveDialog.Filter   := 'bNꂽÍobN (*.lbg)|*.lbg';
    SaveDialog.FileName := ChangeFileExt(OpenDialog.FileName, '.lbg');
    SaveDialog.Options  := [ofHideReadOnly, ofOverwritePrompt, ofDontAddToRecent];

    if not SaveDialog.Execute then
      Exit;

    if ExtractFileExt(SaveDialog.Filename) = '' then
      SaveDialog.Filename := SaveDialog.Filename + '.lbg';

    PubKey := TPubKey.Create;
    try
      MS := TMemoryStream.Create;
      try
        FSecKey.SavePubKeyToStream(MS);
        MS.Position := 0;
        PubKey.LoadFromStream(MS);
      finally
        MS.Free;
      end;

      DLG := TEncryptForm.Create(Self);
      try
        ET := TEncryptThread.Create(True, PubKey, OpenDialog.FileName, SaveDialog.FileName, DLG);
        ET.FreeOnTerminate := True;
        ET.OnTerminate := OnEncryptTerminate;
        ET.Resume;

        DLG.ShowModal;
      finally
        DLG.Free;
      end;
    finally
      PubKey.Free;
    end;
  except
    on E: Exception do
      MessageDlg('ÍŃG[܂' + #13 + E.Message, mtError, [mbAbort], 0)
    else
      MessageDlg('ÍŃG[܂', mtError, [mbAbort], 0);
  end;
end;

procedure TMainForm.EncryptTextActionExecute(Sender: TObject);
var
  TextDLG: TInputTextForm;
  DLG: TEncryptForm;
  MS: TMemoryStream;
  PubKey: TPubKey;
  ET: TEncryptThread;
begin
  try
    TextDLG := TInputTextForm.Create(Self);
    try
      if TextDLG.ShowModal = mrCancel then
        Exit;

      SaveDialog.Title    := 'ÍobN֓ꂽt@C̕ۑ';
      SaveDialog.Filter   := 'bNꂽÍobN (*.lbg)|*.lbg';
      SaveDialog.FileName := ChangeFileExt(FSecKeyFileName, '.lbg');
      SaveDialog.Options  := [ofHideReadOnly, ofOverwritePrompt, ofDontAddToRecent];

      if not SaveDialog.Execute then
        Exit;

      if ExtractFileExt(SaveDialog.Filename) = '' then
        SaveDialog.Filename := SaveDialog.Filename + '.lbg';
      
      PubKey := TPubKey.Create;
      try
        MS := TMemoryStream.Create;
        try
          FSecKey.SavePubKeyToStream(MS);
          MS.Position := 0;
          PubKey.LoadFromStream(MS);
        finally
          MS.Free;
        end;
        DLG := TEncryptForm.Create(Self);
        try
          ET := TEncryptThread.Create(True, PubKey, TextDLG.Memo.Lines, SaveDialog.FileName, DLG);
          ET.FreeOnTerminate := True;
          ET.OnTerminate := OnEncryptTerminate;
          ET.Resume;

          DLG.ShowModal;
        finally
          DLG.Free;
        end;
      finally
        PubKey.Free;
      end;
    finally
      TextDLG.Free;
    end;
  except
    on E: Exception do
      MessageDlg('ÍŃG[܂' + #13 + E.Message, mtError, [mbAbort], 0)
    else
      MessageDlg('ÍŃG[܂', mtError, [mbAbort], 0);
  end;
end;

procedure TMainForm.DecryptActionExecute(Sender: TObject);
var
  I: TFileStream;
  LH : TLiteralPacketHeader;
  PW: string;
  PasswdDLG: TInputPasswdForm;
  DT: TDecryptThread;
  DecryptDLG: TDecryptForm;
  TextDLG: TInputTextForm;
begin
  OpenDialog.Title    := 'bNꂽÍobNJ';
  OpenDialog.FileName := '';
  OpenDialog.Filter   := 'bNꂽÍobN (*.lbg)|*.lbg';
  OpenDialog.Options  := [ofHideReadOnly, ofFileMustExist, ofDontAddToRecent];

  if not OpenDialog.Execute then
    Exit;

  try
    while FSecKey.IsLocked do
    begin
      PasswdDLG := TInputPasswdForm.Create(Self);
      try
        if PasswdDLG.ShowModal <> mrOK then
          Exit;

        PW := PasswdDLG.PasswdEdit.Text;

        if not FSecKey.Unlock(PW) then
          if MessageDlg('ÍL[̃pX[hႢ܂', mtError, [mbRetry, mbCancel], 0) <> IDRETRY then
            Exit;
      finally
        PasswdDLG.Free;
      end;
    end;

    LH := TLiteralPacketHeader.Create;
    try
      I := TFileStream.Create(OpenDialog.FileName, fmOpenRead or fmShareExclusive);
      try
        FSecKey.DecodeFileHeader(I, LH);
      finally
        I.Free;
      end;

      if (LH.FileName = '_CONSOLE') and (LH.Mode = FILE_MODE_TEXT) then
      begin
        FSecKey.Lock(PW);
        TextDLG := TInputTextForm.Create(Self);
        TextDLG.Caption := 'ÍĂeLXg';
        TextDLG.OKButton.Visible := False;
        TextDLG.CancelButton.Caption := '(&C)';
        TextDLG.Memo.ReadOnly := True;
        try
          DecryptDLG := TDecryptForm.Create(Self);
          try
            DT := TDecryptThread.Create(True, FSecKey, PW, OpenDialog.FileName, TextDLG, DecryptDLG);
            DT.FreeOnTerminate := True;
            DT.OnTerminate := OnDecryptTerminate;
            DT.Resume;

            DecryptDLG.ShowModal;
          finally
            DecryptDLG.Free;
          end;
          TextDLG.ShowModal;
        finally
          TextDLG.Free;
        end;
      end
      else
      begin
        SaveDialog.Title    := 'ot@CZ[u';
        SaveDialog.Filter   := 't@C (*.*)|*.*';
        SaveDialog.FileName := ExtractFilePath(OpenDialog.FileName) + LH.FileName;
        SaveDialog.Options  := [ofHideReadOnly, ofOverwritePrompt, ofDontAddToRecent];

        if not SaveDialog.Execute then
          Exit;

        FSecKey.Lock(PW);
        DecryptDLG := TDecryptForm.Create(Self);
        try
          DT := TDecryptThread.Create(True, FSecKey, PW, OpenDialog.FileName, SaveDialog.FileName, DecryptDLG);
          DT.FreeOnTerminate := True;
          DT.OnTerminate := OnDecryptTerminate;
          DT.Resume;

          DecryptDLG.ShowModal;
        finally
          DecryptDLG.Free;
        end;
      end;
    finally
      LH.Free;
      FSecKey.Lock(PW);
    end;
  except
    on E: Exception do
      MessageDlg('bN̉ŃG[܂' + #13 + E.Message, mtError, [mbOK], 0);
    else
      MessageDlg('bN̉ŃG[܂', mtError, [mbOK], 0);
  end;
end;

end.
