/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.dom.stylesheets;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.trix.cuery.sac.SACMediaListImpl;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.stylesheets.MediaList;
import org.w3c.dom.stylesheets.StyleSheet;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: StyleSheetImpl.java,v 1.0 2005/08/11 3:07:24 Teletha Exp $
 */
public class StyleSheetImpl implements StyleSheet {

    /** The pattern for processing instruction data. */
    private static final Pattern PI_DATA = Pattern.compile("([^=\"']+)=[\"']([^\"']*)[\"']");

    /** The href. */
    private String href;

    /** The title. */
    private String title;

    /** The type. */
    private String type;

    /** The owner node. */
    private Node owner;

    /** The media list. */
    private MediaList list = new SACMediaListImpl();

    /** The disabled. */
    private boolean disabled = false;

    /**
     * Create StyleSheetImpl instance.
     * 
     * @param pi A processing instruction.
     * @throws DOMException If this pi is not suitable.
     */
    public StyleSheetImpl(ProcessingInstruction pi) throws DOMException {
        // check target
        if (pi == null || !pi.getTarget().equals("xml-stylesheet")) {
            throw new DOMException(DOMException.SYNTAX_ERR, "The value 'xml-stylesheet' is only accepted.");
        }

        Matcher matcher = PI_DATA.matcher(pi.getData());

        while (matcher.find()) {
            String name = matcher.group(1).trim();
            String value = matcher.group(2);

            if (name.equals("href")) {
                href = value;
            } else if (name.equals("title")) {
                title = value;
            } else if (name.equals("type")) {
                type = value;
            } else if (name.equals("media")) {
                list.setMediaText(value);
            }
        }
        this.owner = pi.getOwnerDocument();
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#getType()
     */
    public String getType() {
        return type;
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#getDisabled()
     */
    public boolean getDisabled() {
        return disabled;
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#setDisabled(boolean)
     */
    public void setDisabled(boolean disabled) {
        this.disabled = disabled;
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#getOwnerNode()
     */
    public Node getOwnerNode() {
        return owner;
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#getParentStyleSheet()
     */
    public StyleSheet getParentStyleSheet() {
        return null;
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#getHref()
     */
    public String getHref() {
        return href;
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#getTitle()
     */
    public String getTitle() {
        return title;
    }

    /**
     * @see org.w3c.dom.stylesheets.StyleSheet#getMedia()
     */
    public MediaList getMedia() {
        return list;
    }

}
