/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.property;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.trix.cuery.property.config.DefaultFontConfiguration;
import org.trix.cuery.property.config.FontConfiguration;
import org.trix.cuery.value.AttrFunction;
import org.trix.cuery.value.CSSAngle;
import org.trix.cuery.value.CSSColor;
import org.trix.cuery.value.CSSFrequency;
import org.trix.cuery.value.CSSLength;
import org.trix.cuery.value.CSSNumber;
import org.trix.cuery.value.CSSPercentage;
import org.trix.cuery.value.CSSString;
import org.trix.cuery.value.CSSTime;
import org.trix.cuery.value.CSSURI;
import org.trix.cuery.value.CSSValue;
import org.trix.cuery.value.CounterFunction;
import org.trix.cuery.value.CountersFunction;
import org.trix.cuery.value.Identifier;
import org.trix.cuery.value.RectFunction;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: PropertyRegistry.java,v 1.03 2005/09/10 12:12:21 Teletha Exp $
 */
public final class PropertyRegistry {

    /** The default font configuration. */
    private static FontConfiguration fontConfiguration = new DefaultFontConfiguration();

    /** The path to the default property definition. */
    private static final String PATH = "org/trix/cuery/property/css2/definition.xml";

    /** The property definition mapping. */
    private static final Map DEFINITIONS = new HashMap();

    /** The xml reader. */
    private static SAXParser parser;

    // initialize
    static {
        SAXParserFactory factory = SAXParserFactory.newInstance();

        try {
            parser = factory.newSAXParser();

            InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(PATH);
            readPropertyDefinition(new InputSource(stream));
        } catch (Exception e) {
            System.out.println("PropertyRegistery initialization phase   " + e);
        }
    }

    /**
     * Avoid creating PropertyRegistry instance.
     */
    private PropertyRegistry() {
    }

    /**
     * Return a current font configuration.
     * 
     * @return A current font configuration.
     */
    public static FontConfiguration getFontConfiguration() {
        return fontConfiguration;
    }

    /**
     * Set a current font configuration.
     * 
     * @param fontConfiguration A font configuration to set.
     */
    public static void setFontConfiguration(FontConfiguration fontConfiguration) {
        if (fontConfiguration == null) {
            return;
        }
        PropertyRegistry.fontConfiguration = fontConfiguration;
    }

    /**
     * Retrive a specified property definition.
     * 
     * @param name A property name.
     * @return A definition.
     */
    public static PropertyDefinition getDefinition(String name) {
        return (PropertyDefinition) DEFINITIONS.get(name);
    }

    /**
     * Input property difinition file.
     * 
     * @param source A property definition file.
     * @throws IOException If the file has I/O error.
     */
    public static void readPropertyDefinition(InputSource source) throws IOException {
        try {
            parser.parse(source, new DefinitionReader());
        } catch (SAXException e) {
            throw new IOException(e.getLocalizedMessage());
        } catch (IOException e) {
            throw new IOException(e.getLocalizedMessage());
        }
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: DefinitionReader.java,v 1.0 2005/08/19 15:01:26 Teletha Exp $
     */
    private static class DefinitionReader extends DefaultHandler {

        /** The class type mapping. */
        private static final Map CLASSTYPES = new HashMap();

        // initialize
        static {
            CLASSTYPES.put("angle", CSSAngle.class);
            CLASSTYPES.put("attr", AttrFunction.class);
            CLASSTYPES.put("color", CSSColor.class);
            CLASSTYPES.put("counter", CounterFunction.class);
            CLASSTYPES.put("counters", CountersFunction.class);
            CLASSTYPES.put("frequency", CSSFrequency.class);
            CLASSTYPES.put("ident", Identifier.class);
            CLASSTYPES.put("integer", CSSNumber.class);
            CLASSTYPES.put("length", CSSLength.class);
            CLASSTYPES.put("number", CSSNumber.class);
            CLASSTYPES.put("percentage", CSSPercentage.class);
            CLASSTYPES.put("rect", RectFunction.class);
            CLASSTYPES.put("string", CSSString.class);
            CLASSTYPES.put("time", CSSTime.class);
            CLASSTYPES.put("uri", CSSURI.class);
        }

        /** The current processing property definition. */
        private PropertyDefinition current;

        /**
         * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String,
         *      java.lang.String, org.xml.sax.Attributes)
         */
        public void startElement(String namespaceURI, String localName, String qName, Attributes attributes)
                throws SAXException {
            if (qName.equals("property-def")) {
                String name = attributes.getValue("name");
                String inherited = attributes.getValue("inherited");
                String initialValue = attributes.getValue("initial");
                String className = attributes.getValue("class");
                PropertyDefinition definition;

                if (className == null) {
                    definition = new SimplePropertyDefinition();
                } else {
                    try {
                        definition = (PropertyDefinition) Class.forName(className).newInstance();
                    } catch (InstantiationException e) {
                        System.out.println(e);
                        return;
                    } catch (IllegalAccessException e) {
                        System.out.println(e);
                        return;
                    } catch (ClassNotFoundException e) {
                        System.out.println(e);
                        return;
                    }
                }

                CSSValue initial = null;

                if (initialValue == null) {
                    initial = new Identifier(initialValue, null);
                } else if (initialValue.endsWith(CSSPercentage.UNIT)) {
                    initial = new CSSPercentage(initialValue, null);
                } else {
                    initial = new Identifier(initialValue, null);
                }

                definition.setup(name, inherited.equals("yes"), initial);

                if (!DEFINITIONS.containsKey(name)) {
                    DEFINITIONS.put(name, definition);
                }
                current = definition;
            } else if (qName.equals("keyword")) {
                current.addAcceptable(attributes.getValue("name"));
            } else if (qName.equals("datatype")) {
                Class clazz = (Class) CLASSTYPES.get(attributes.getValue("name"));

                if (clazz != null) {
                    current.addAcceptable(clazz);
                }
            } else if (qName.equals("function")) {
                Class clazz = (Class) CLASSTYPES.get(attributes.getValue("name"));

                if (clazz != null) {
                    current.addAcceptable(clazz);
                }
            } else if (qName.equals("property")) {
                PropertyDefinition definition = (PropertyDefinition) DEFINITIONS.get(attributes.getValue("name"));

                if (definition != null) {
                    current.addAcceptable(definition);
                }
            }
        }
    }
}
