/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.value;

import org.trix.cuery.util.CSSUtil;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: CSSPercentage.java,v 1.01 2005/09/08 17:22:32 Teletha Exp $
 */
public class CSSPercentage extends AbstractCSSValue {

    /** The unit. */
    public static final String UNIT = "%";

    /**
     * Create CSSPercentage instance.
     * 
     * @param percentage A percentage value.
     */
    public CSSPercentage(float percentage) {
        this(percentage, null);
    }

    /**
     * Create CSSPercentage instance.
     * 
     * @param percentage A percentage value.
     */
    public CSSPercentage(String percentage) {
        this(percentage, null);
    }

    /**
     * Create CSSPercentage instance.
     * 
     * @param percentage A percentage value.
     * @param previous A previous value of this value.
     */
    public CSSPercentage(float percentage, CSSValue previous) {
        super(previous);

        this.stringValue = CSSUtil.displayFloat(percentage);
        this.floatValue = percentage;
    }

    /**
     * Create CSSPercentage instance.
     * 
     * @param percentage A percentage value.
     * @param previous A previous value of this value.
     */
    public CSSPercentage(String percentage, CSSValue previous) {
        super(previous);

        // check unit
        if (percentage.endsWith(UNIT)) {
            percentage = percentage.substring(0, percentage.length() - 1);
        }

        this.stringValue = percentage;
        this.floatValue = Float.parseFloat(percentage);
    }

    /**
     * @see org.trix.cuery.value.AbstractCSSValue#getDimensionUnitText()
     */
    public String getDimensionUnitText() {
        return UNIT;
    }

    /**
     * @see org.w3c.css.sac.LexicalUnit#getLexicalUnitType()
     */
    public short getLexicalUnitType() {
        return SAC_PERCENTAGE;
    }

    /**
     * @see org.w3c.css.sac.LexicalUnit#getFloatValue()
     */
    public float getFloatValue() {
        return floatValue;
    }

    /**
     * @see org.w3c.css.sac.LexicalUnit#getStringValue()
     */
    public String getStringValue() {
        return stringValue;
    }

    /**
     * @see org.w3c.dom.css.CSSPrimitiveValue#getPrimitiveType()
     */
    public short getPrimitiveType() {
        return CSS_PERCENTAGE;
    }

    /**
     * @see org.trix.cuery.value.AbstractCSSValue#getCssText()
     */
    public String getCssText() {
        return stringValue + getDimensionUnitText();
    }

}
