/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.value;

import java.math.BigDecimal;

import org.trix.cuery.property.PropertyRegistry;

import org.w3c.css.sac.LexicalUnit;
import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: ValueUtil.java,v 1.0 2005/09/09 18:44:08 Teletha Exp $
 */
public final class ValueUtil {

    /** The zero point. */
    public static final PointLength ZERO_POINT = new PointLength(0);

    /** The standard scale size. */
    private static final int SCALE = 1;

    /**
     * Avoid creating ValueUtil instance.
     */
    private ValueUtil() {
    }

    /**
     * Convert a value of absolute length to point length.
     * 
     * @param length A css value for length.
     * @return A point length value.
     */
    public static PointLength convertToPointLength(CSSLength length) {
        return new PointLength(convertToPointLength(length.getFloatValue(), length.getLexicalUnitType()));
    }

    /**
     * Convert a value of absolute length to point length.
     * 
     * @param value A float value for length.
     * @param type A unit type.
     * @return A point length value.
     */
    public static float convertToPointLength(float value, short type) {
        float result;

        switch (type) {
        case LexicalUnit.SAC_CENTIMETER:
            result = 12 * value / 2.54f;
            break;

        case LexicalUnit.SAC_INCH:
            result = 72 * value;
            break;

        case LexicalUnit.SAC_MILLIMETER:
            result = 12 * value / 25.4f;
            break;

        case LexicalUnit.SAC_PICA:
            result = 12 * value;
            break;

        case LexicalUnit.SAC_PIXEL:
            result = 72 * value / PropertyRegistry.getFontConfiguration().getDisplayDPI();
            break;

        case LexicalUnit.SAC_POINT:
            result = value;
            break;

        default:
            result = value;
        }
        return round(result);
    }

    /**
     * Divide CSSLength by CSSPercentage.
     * 
     * @param length A base length
     * @param scale A scale.
     * @return A computed CSSLength.
     */
    public static CSSLength divide(CSSLength length, float scale) {
        // check null
        if (length == null) {
            return ZERO_POINT;
        }

        // compute
        return createCSSLength(length.getLexicalUnitType(), round(length.getFloatValue() / scale));
    }

    /**
     * Check wether this css value is CSSLength. This is a faster than by using instanceOf
     * operation.
     * 
     * @param value A target value.
     * @return A result.
     */
    public static boolean isCSSLength(CSSValue value) {
        short type = value.getPrimitiveType();
        return CSSPrimitiveValue.CSS_EMS <= type && type <= CSSPrimitiveValue.CSS_PC;
    }

    /**
     * Check wether this css value is absolute CSSLength. This is a faster than by using instanceOf
     * operation.
     * 
     * @param value A target value.
     * @return A result.
     */
    public static boolean isAbsoluteCSSLength(CSSValue value) {
        short type = value.getPrimitiveType();
        return CSSPrimitiveValue.CSS_PX <= type && type <= CSSPrimitiveValue.CSS_PC;
    }

    /**
     * Multiplt CSSLength and CSSPercentage.
     * 
     * @param length A base length
     * @param percentage A scale.
     * @return A computed CSSLength.
     */
    public static CSSLength multiply(CSSLength length, CSSPercentage percentage) {
        // check null
        if (percentage == null) {
            return multiply(length, 1);
        }
        return multiply(length, percentage.floatValue / 100);
    }

    /**
     * Multiplt CSSLength and scale value.
     * 
     * @param length A base length
     * @param scale A scale.
     * @return A computed CSSLength.
     */
    public static CSSLength multiply(CSSLength length, float scale) {
        // check null
        if (length == null) {
            return ZERO_POINT;
        }

        // compute
        return createCSSLength(length.getLexicalUnitType(), round(length.getFloatValue() * scale));
    }

    /**
     * Create CSSLength by type with a value.
     * 
     * @param type A length type.
     * @param value A value.
     * @return A created CSSLength.
     */
    private static CSSLength createCSSLength(short type, float value) {
        switch (type) {
        case LexicalUnit.SAC_CENTIMETER:
            return new CentimeterLength(value);

        case LexicalUnit.SAC_EM:
            return new EMLength(value);

        case LexicalUnit.SAC_EX:
            return new EXLength(value);

        case LexicalUnit.SAC_INCH:
            return new InchLength(value);

        case LexicalUnit.SAC_MILLIMETER:
            return new MillimeterLength(value);

        case LexicalUnit.SAC_PICA:
            return new PicaLength(value);

        case LexicalUnit.SAC_PIXEL:
            return new PixelLength(value);

        case LexicalUnit.SAC_POINT:
            return new PointLength(value);

        default:
            return ZERO_POINT;
        }
    }

    /**
     * Round float value off in the first place of the decimal point.
     * 
     * @param value A target value.
     * @return A result.
     */
    public static float round(float value) {
        return new BigDecimal(String.valueOf(value)).setScale(SCALE, BigDecimal.ROUND_HALF_EVEN).floatValue();
    }
}
