/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery;

import java.io.IOException;
import java.util.Locale;

import org.trix.cuery.parser.CueryParser;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.ConditionFactory;
import org.w3c.css.sac.DocumentHandler;
import org.w3c.css.sac.ErrorHandler;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.Parser;
import org.w3c.css.sac.Selector;
import org.w3c.css.sac.SelectorFactory;
import org.w3c.css.sac.SelectorList;

/**
 * <p>
 * This class implements Parser interface of SAC API.
 * </p>
 * <p>
 * Parser interface represents the parser itself. All SAC parsers must implement this basic
 * interface. It allows applications to register handlers for differents types of events, to set the
 * factories for the CSS selectors, and to initiate the parsing of the CSS document or a partial CSS
 * document.The conventional solution to create instance of objects in object-oriented design is to
 * define factory methods. Objects implementing some interface "X" are created by a "createX()".
 * </p>
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: CSSParser.java,v 1.0 2005/09/11 21:06:50 Teletha Exp $
 */
public class CSSParser implements Parser {

    /** The actual parser. */
    protected CueryParser parser = new CueryParser();

    /**
     * @see org.w3c.css.sac.Parser#getParserVersion()
     */
    public String getParserVersion() {
        return parser.getParserVersion();
    }

    /**
     * @see org.w3c.css.sac.Parser#parsePriority(org.w3c.css.sac.InputSource)
     */
    public boolean parsePriority(InputSource input) throws CSSException, IOException {
        return parser.parsePriority(input);
    }

    /**
     * @see org.w3c.css.sac.Parser#parsePropertyValue(org.w3c.css.sac.InputSource)
     */
    public LexicalUnit parsePropertyValue(InputSource input) throws CSSException, IOException {
        return parser.parsePropertyValue(input);
    }

    /**
     * @see org.w3c.css.sac.Parser#parseRule(org.w3c.css.sac.InputSource)
     */
    public void parseRule(InputSource input) throws CSSException, IOException {
        parser.parseRule(input);
    }

    /**
     * Parse a selector expression.
     * 
     * @param expression A selector expression for css.
     * @return A parsed selector instance.
     * @throws CSSException If this expression is invalid.
     */
    public Selector parseSelector(String expression) throws CSSException {
        return parser.parseSelector(expression);
    }

    /**
     * @see org.w3c.css.sac.Parser#parseSelectors(org.w3c.css.sac.InputSource)
     */
    public SelectorList parseSelectors(InputSource input) throws CSSException, IOException {
        return parser.parseSelectors(input);
    }

    /**
     * @see org.w3c.css.sac.Parser#parseStyleDeclaration(org.w3c.css.sac.InputSource)
     */
    public void parseStyleDeclaration(InputSource input) throws CSSException, IOException {
        parser.parseStyleDeclaration(input);
    }

    /**
     * @see org.w3c.css.sac.Parser#parseStyleSheet(org.w3c.css.sac.InputSource)
     */
    public void parseStyleSheet(InputSource input) throws CSSException, IOException {
        parser.parseStyleSheet(input);
    }

    /**
     * @see org.w3c.css.sac.Parser#parseStyleSheet(java.lang.String)
     */
    public void parseStyleSheet(String input) throws CSSException, IOException {
        parser.parseStyleSheet(input);
    }

    /**
     * @see org.w3c.css.sac.Parser#setConditionFactory(org.w3c.css.sac.ConditionFactory)
     */
    public void setConditionFactory(ConditionFactory factory) {
        parser.setConditionFactory(factory);
    }

    /**
     * @see org.w3c.css.sac.Parser#setDocumentHandler(org.w3c.css.sac.DocumentHandler)
     */
    public void setDocumentHandler(DocumentHandler handler) {
        parser.setDocumentHandler(handler);
    }

    /**
     * @see org.w3c.css.sac.Parser#setErrorHandler(org.w3c.css.sac.ErrorHandler)
     */
    public void setErrorHandler(ErrorHandler handler) {
        parser.setErrorHandler(handler);
    }

    /**
     * @see org.w3c.css.sac.Parser#setLocale(java.util.Locale)
     */
    public void setLocale(Locale locale) throws CSSException {
        parser.setLocale(locale);
    }

    /**
     * @see org.w3c.css.sac.Parser#setSelectorFactory(org.w3c.css.sac.SelectorFactory)
     */
    public void setSelectorFactory(SelectorFactory factory) {
        parser.setSelectorFactory(factory);
    }

}
