/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.filter;

import org.w3c.css.sac.AttributeCondition;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: AttributeFilter.java,v 1.03 2005/12/23 16:21:57 Teletha Exp $
 */
public class AttributeFilter implements Filter, AttributeCondition {

    /** The namespace prefix. */
    protected String prefix;

    /** The namespace URI. */
    protected String namespaceURI;

    /** The name. */
    protected String name = null;

    /** The value. */
    protected String value = null;

    /**
     * Create AttributeFilter instance.
     * 
     * @param name An attribute name.
     * @param value An attribute value.
     */
    public AttributeFilter(String name, String value) {
        this(null, null, name, value);
    }

    /**
     * Create AttributeFilter instance.
     * 
     * @param prefix A namespace prefix.
     * @param namespaceURI A namespace URI.
     * @param name An attribute name.
     * @param value An attribute value.
     */
    public AttributeFilter(String prefix, String namespaceURI, String name, String value) {
        this.prefix = prefix;
        this.namespaceURI = namespaceURI;
        this.name = name;
        this.value = value;
    }

    /**
     * @see org.trix.cuery.filter.Filter#accept(org.w3c.dom.Element)
     */
    public boolean accept(Element target) {
        // check null
        if (target == null || name == null) {
            return false;
        }

        // "*|E" attributes with name E and any declared namespace
        if (namespaceURI == null) {
            NamedNodeMap attributes = target.getAttributes();

            for (int i = 0; i < attributes.getLength(); i++) {
                Attr attr = (Attr) attributes.item(i);

                if (attr.getNodeName().equals(name) && acceptAttribute(attr.getValue())) {
                    return true;
                }
            }
            return false;
        }

        // "|E" attributes with name E without any declared namespace
        if (namespaceURI.length() == 0) {
            NamedNodeMap attributes = target.getAttributes();

            for (int i = 0; i < attributes.getLength(); i++) {
                Attr attr = (Attr) attributes.item(i);

                if (attr.getNamespaceURI() == null && attr.getName().equals(name)) {
                    return acceptAttribute(attr.getValue());
                }
            }
            return false;
        }

        // "ns|E" attributes with name E in namespace ns
        String attribute = target.getAttributeNS(namespaceURI, name);

        // check attribute
        if (attribute == null || attribute.length() == 0) {
            return false;
        }
        return acceptAttribute(attribute);
    }

    /**
     * @see org.trix.cuery.filter.Filter#getSpecificity()
     */
    public int getSpecificity() {
        return 10;
    }

    /**
     * @see org.w3c.css.sac.Condition#getConditionType()
     */
    public short getConditionType() {
        return SAC_ATTRIBUTE_CONDITION;
    }

    /**
     * @see org.w3c.css.sac.Selector#getSelectorType()
     */
    public short getSelectorType() {
        return SAC_CONDITIONAL_SELECTOR;
    }

    /**
     * @see org.w3c.css.sac.AttributeCondition#getLocalName()
     */
    public String getLocalName() {
        return name;
    }

    /**
     * @see org.w3c.css.sac.AttributeCondition#getNamespaceURI()
     */
    public String getNamespaceURI() {
        return namespaceURI;
    }

    /**
     * @see org.w3c.css.sac.AttributeCondition#getSpecified()
     */
    public boolean getSpecified() {
        return false;
    }

    /**
     * @see org.w3c.css.sac.AttributeCondition#getValue()
     */
    public String getValue() {
        return value;
    }

    /**
     * Check that the target attribute value fills the condition.
     * 
     * @param attributeValue A target attribute value.
     * @return A result.
     */
    protected boolean acceptAttribute(String attributeValue) {
        if (value == null) {
            return true;
        }
        return value.equals(attributeValue);
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        if (prefix == null) {
            if (value == null) {
                return "[" + name + "]";
            } else {
                return "[" + name + "=\"" + value + "\"]";
            }
        } else {
            if (value == null) {
                return "[" + prefix + "|" + name + "]";
            } else {
                return "[" + prefix + "|" + name + "=\"" + value + "\"]";
            }
        }
    }
}
