/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.property;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.trix.cuery.value.CSSValue;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: SimpleProperty.java,v 1.02 2005/12/06 11:34:15 Teletha Exp $
 */
public class SimpleProperty extends AbstractProperty {

    /** The map of properties. */
    private List properties = new ArrayList(8);

    /**
     * @see org.trix.cuery.property.Property#getValue(java.lang.String)
     */
    public CSSValue getValue(String name) {
        // assert null
        if (name == null || name.length() == 0) {
            return null;
        }

        for (int i = 0; i < properties.size(); i++) {
            Object[] objects = (Object[]) properties.get(i);

            if (name.equals(objects[0])) {
                return (CSSValue) objects[1];
            }
        }
        return null;
    }

    /**
     * @see org.trix.cuery.property.Property#isImportant(java.lang.String)
     */
    public boolean isImportant(String name) {
        // assert null
        if (name == null || name.length() == 0) {
            return false;
        }

        for (int i = 0; i < properties.size(); i++) {
            Object[] objects = (Object[]) properties.get(i);

            if (name.equals(objects[0])) {
                return ((Boolean) objects[2]).booleanValue();
            }
        }
        return false;
    }

    /**
     * Add a property.
     * 
     * @param name A property name.
     * @param value A prperty value.
     * @param important A important priority.
     */
    public void setProperty(String name, CSSValue value, boolean important) {
        PropertyDefinition definition = PropertyRegistry.getDefinition(name);

        // user agents must ignore a declaration with an unknown property
        if (definition == null) {
            return;
        }

        // user agents must ignore a declaration with an illegal value
        Map result = definition.parse(value);

        if (result == null) {
            return;
        }

        // store property
        Boolean importance = Boolean.valueOf(important);
        properties.add(new Object[] {name, value, importance});

        // store parsed shorthand property
        Iterator iterator = result.entrySet().iterator();

        while (iterator.hasNext()) {
            Map.Entry entry = (Map.Entry) iterator.next();
            properties.add(new Object[] {entry.getKey(), entry.getValue(), importance});
        }
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append('{');

        for (int i = 0; i < properties.size(); i++) {
            Object[] objects = (Object[]) properties.get(i);

            buffer.append(objects[0]);
            buffer.append(" : ");
            buffer.append(objects[1]);

            Boolean importance = (Boolean) objects[2];

            if (importance.booleanValue()) {
                buffer.append(" !important");
            }

            if (i == properties.size() - 1) {
                buffer.append(';');
            } else {
                buffer.append(";    ");
            }
        }
        buffer.append('}');

        return buffer.toString();
    }
}
