/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.filter;

import java.io.File;
import java.io.IOException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import junit.framework.TestCase;

import org.trix.cuery.CSSParser;
import org.trix.cuery.util.DOMUtil;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import org.xml.sax.SAXException;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: FilterTest.java,v 1.0 2005/08/05 13:38:10 Teletha Exp $
 */
public class FilterTest extends TestCase {

    /** The test parser. */
    private static final CSSParser parser = new CSSParser();

    private static final Document document = createDocument("src/test/resource/query/cssQuery.xml");

    private static Document createDocument(String path) {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);

        try {
            DocumentBuilder builder = factory.newDocumentBuilder();

            return builder.parse(new File(path));
        } catch (ParserConfigurationException e) {
            e.printStackTrace();
        } catch (SAXException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        throw new IllegalArgumentException("Unread the target xml '" + path + "'.");
    }

    /*
     * Test method for 'org.trix.cuery.filter.Filter.getSpecificity()'
     */
    public void testGetSpecificity() {
        Filter filter = (Filter) parser.parseSelector("*");
        assertEquals(0, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("html");
        assertEquals(1, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("html body");
        assertEquals(2, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("html > body");
        assertEquals(2, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("p + p + p.alt");
        assertEquals(13, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("#id some text *");
        assertEquals(102, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("div.class1.class2 span.checked");
        assertEquals(32, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("div.class1.class2[title]");
        assertEquals(31, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("p a:first-child");
        assertEquals(12, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("p a:link");
        assertEquals(12, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("p:first-line");
        assertEquals(1, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("p:before");
        assertEquals(1, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("#id:not(foo)");
        assertEquals(101, filter.getSpecificity());

        filter = (Filter) parser.parseSelector("h1 + *[rel=up] ");
        assertEquals(11, filter.getSpecificity());
    }

    /**
     * Test child selector.
     * 
     * @throws Exception Test fails.
     */
    public void testChild() throws Exception {
        Element element = DOMUtil.getElementById(document, "heading1");
        Filter filter = (Filter) parser.parseSelector("h1");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("body > h1");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("html > body > h1");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("html > h1");
        assertEquals(false, filter.accept(element));

        filter = (Filter) parser.parseSelector("test > h1");
        assertEquals(false, filter.accept(element));
    }

    /**
     * Test descendant selector.
     * 
     * @throws Exception Test fails.
     */
    public void testDescendant() throws Exception {
        Element element = DOMUtil.getElementById(document, "heading1");
        Filter filter = (Filter) parser.parseSelector("body h1");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("html body h1");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("test body  h1");
        assertEquals(false, filter.accept(element));

        filter = (Filter) parser.parseSelector("html test h1");
        assertEquals(false, filter.accept(element));

        filter = (Filter) parser.parseSelector("test html h1");
        assertEquals(false, filter.accept(element));
    }

    /**
     * Test sibling selector.
     * 
     * @throws Exception Test fails.
     */
    public void testSibling() throws Exception {
        Element element = DOMUtil.getElementById(document, "identified");
        Filter filter = (Filter) parser.parseSelector("p + p");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("h1 ~ p");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("address + p");
        assertEquals(false, filter.accept(element));

        filter = (Filter) parser.parseSelector("address ~ p");
        assertEquals(false, filter.accept(element));

        filter = (Filter) parser.parseSelector("html h1 ~ p");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("body > h1 ~ p");
        assertEquals(true, filter.accept(element));
    }

    /**
     * Test attribute selector.
     * 
     * @throws Exception Test fails.
     */
    public void testAttribute() throws Exception {
        Element element = DOMUtil.getElementById(document, "heading1");
        Filter filter = (Filter) parser.parseSelector("[class]");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("html [class]");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("body > [class]");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("test ~ [class]");
        assertEquals(false, filter.accept(element));

        filter = (Filter) parser.parseSelector("[class=\"heading1\"]");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("html [class=\"heading1\"]");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("body > [class=\"heading1\"]");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("test ~ [class=\"heading1\"]");
        assertEquals(false, filter.accept(element));

        filter = (Filter) parser.parseSelector("body > [class=\"test\"]");
        assertEquals(false, filter.accept(element));
    }

    /**
     * Test pseudo selector.
     * 
     * @throws Exception Test fails.
     */
    public void testPseudo() throws Exception {
        Element element = DOMUtil.getElementById(document, "last");
        Filter filter = (Filter) parser.parseSelector("p");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("#last");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("p:last-of-type");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("html p:last-of-type");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("body > p:last-of-type");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("body > p:last-child");
        assertEquals(false, filter.accept(element));
    }

    /**
     * Test sibling selector.
     * 
     * @throws Exception Test fails.
     */
    public void testComplex() throws Exception {
        Element element = DOMUtil.getElementById(document, "nth-child");
        Filter filter = (Filter) parser.parseSelector("[title] ~ dl");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("[title] + dl");
        assertEquals(true, filter.accept(element));

        filter = (Filter) parser.parseSelector("[id=\"number\"] ~ dl");
        assertEquals(true, filter.accept(element));
    }
}
